# -*- coding: utf-8 -*-
from datetime import datetime

from django.conf import settings
from django.utils.functional import cached_property
from passport.backend.core.geobase import Region
from passport.backend.core.logging_utils.loggers.statbox import (
    AntifraudLogger,
    CredentialsLogger,
)
from passport.backend.oauth.core.common.constants import GRANT_TYPE_REFRESH_TOKEN
from passport.backend.oauth.core.common.crypto import (
    BaseCryptoError,
    decrypt,
    encrypt,
)
from passport.backend.oauth.core.db.eav import EavAttr
from passport.backend.oauth.core.db.eav.model import StatelessEavModel
from passport.backend.oauth.core.db.eav.types import DB_NULL
from passport.backend.oauth.core.db.mixins.client_mixin import ClientMixin
from passport.backend.oauth.core.db.mixins.scopes_mixin import ScopesMixin
from passport.backend.oauth.core.db.mixins.ttl_mixin import TtlMixin
from passport.backend.oauth.core.logs.historydb import to_event_log
from passport.backend.oauth.core.logs.statbox import StatboxLogger


TOKEN_TYPE_NORMAL = 'normal'
TOKEN_TYPE_STATELESS = 'stateless'


class TokenBase(StatelessEavModel, ClientMixin, ScopesMixin, TtlMixin):
    _entity_name = 'token'

    access_token = EavAttr()
    uid = EavAttr()
    device_id = EavAttr()
    device_name = EavAttr()
    is_refreshable = EavAttr()
    created = EavAttr()
    issued = EavAttr()
    meta = EavAttr()
    x_token_id = EavAttr()
    alias = EavAttr()
    payment_auth_context_id = EavAttr()
    payment_auth_scope_addendum = EavAttr()
    login_id = EavAttr()
    app_platform = EavAttr()
    is_xtoken_trusted = EavAttr()

    @classmethod
    def create(cls, **kwargs):
        raise NotImplementedError()  # pragma: no cover

    @classmethod
    def _parse(cls, oauth_block):
        return oauth_block['token_attributes']

    def is_invalidated_by_user_logout(self, revoke_time):
        # Проверка нестрогая, для консистентности с ЧЯ
        return bool(revoke_time) and self.issued < revoke_time

    def is_invalidated_by_client_glogout(self, client):
        """Инвалидирован ли токен глогаутом клиента (при изменении/удалении клиента)"""
        if client.id != self.client_id:
            raise ValueError('Bad client: %s != %s' % (client.id, self.client_id))
        return bool(client.glogouted) and self.issued <= client.glogouted

    def is_valid(self, client, revoke_time):
        return not (
            self.is_expired or
            self.is_invalidated_by_client_glogout(client) or
            self.is_invalidated_by_user_logout(revoke_time)
        )

    @property
    def is_app_password(self):
        # у обычных токенов нет алиаса, а у токенов-паролей приложений есть
        return bool(self.alias)

    @property
    def masked_body(self):
        # Возвращает замаскированную тушку, пригодную для записи в логи и отображения в админке
        raise NotImplementedError()  # pragma: no cover

    @cached_property
    def refresh_token(self):
        if self.is_app_password or not self.uid:
            # Для ПП или деперсонализированных токенов нельзя получить refresh-токен
            raise ValueError('Unable to issue refresh_token for token id=%s' % self.id)
        return encrypt(
            keys=settings.REFRESH_TOKEN_GENERATION_KEYS,
            message=self.access_token,
            version=settings.REFRESH_TOKEN_DEFAULT_VERSION,
        ).decode()

    @property
    def total_ttl(self):
        return get_ttl_by_scopes(self.scopes) or 0


def get_ttl_by_scopes(scopes):
    nonzero_ttls = [scope.ttl for scope in scopes if scope.ttl]
    return min(nonzero_ttls) if nonzero_ttls else None


def log_token_issue(token, client, grant_type, env, dt=None, statbox_logger=None, antifraud_logger=None,
                    credentials_logger=None, token_type=TOKEN_TYPE_NORMAL, password_passed=False, token_reused=False,
                    login=None, passport_track_id=None, auth_source=None):
    statbox_logger = statbox_logger or StatboxLogger(mode='issue_token')
    antifraud_logger = antifraud_logger or AntifraudLogger()
    credentials_logger = credentials_logger or CredentialsLogger()
    dt = dt or datetime.now()
    device_id = token.device_id if token.device_id != DB_NULL else None
    password_passed = password_passed or grant_type == 'password'
    login_id = token.login_id or 't:{}'.format(token.x_token_id or token.id) or None
    statbox_logger.log(
        action='issue',
        dt=dt,
        auth_source=auth_source,
        status='ok',
        token_type=token_type,
        token_id=token.id,
        uid=token.uid,
        client_id=client.display_id,
        scopes=','.join(map(str, token.scopes)),
        device_id=device_id,
        login_id=login_id,
        grant_type=grant_type,
        create_time=token.created,
        issue_time=token.issued,
        expire_time=token.expires,
        is_ttl_refreshable=token.is_refreshable if token.expires else None,
        has_alias=token.is_app_password,
        password_passed=password_passed,
        token_reused=token_reused,
        passport_track_id=passport_track_id,
        user_ip=env.user_ip,
        consumer_ip=env.consumer_ip,
        user_agent=env.user_agent,
        yandexuid=env.cookies.get('yandexuid'),
    )
    to_event_log(
        dt=dt,
        action='create',
        target='token',
        token_type=token_type,
        grant_type=grant_type,
        uid=token.uid,
        token_id=token.id,
        client_id=client.display_id,
        device_id=device_id,
        device_name=token.device_name if device_id else None,
        scopes=','.join(map(str, token.scopes)),
        has_alias=token.is_app_password,
        password_passed=password_passed,
        user_ip=env.user_ip,
        consumer_ip=env.consumer_ip,
        user_agent=env.user_agent,
        yandexuid=env.cookies.get('yandexuid'),
    )
    region_info = Region(ip=str(env.user_ip))
    antifraud_logger.log(
        channel='auth',
        auth_source=auth_source,
        sub_channel=settings.ANTIFRAUD_AUTH_SUB_CHANNEL,
        external_id='track-{}'.format(passport_track_id) if passport_track_id else 'req-{}'.format(env.request_id),
        status='OK',
        grant_type=grant_type,
        token_id=token.id,
        uid=token.uid,
        client_id=client.display_id,
        is_client_yandex=client.is_yandex,
        device_id=device_id,
        scopes=','.join(map(str, token.scopes)),
        login_id=token.login_id or None,  # TODO: заменить на login_id в PASSP-35611
        previous_login_id=token.login_id or None,
        password_passed=password_passed,
        is_app_password=token.is_app_password,
        ip=env.user_ip,
        AS=region_info.AS_list[0] if region_info.AS_list else None,
        yandexuid=env.cookies.get('yandexuid'),
        user_agent=env.user_agent,
    )

    if grant_type not in [GRANT_TYPE_REFRESH_TOKEN, ]:
        credentials_logger.log(
            client_id=client.display_id,
            credential_type='token',
            device_id=device_id,
            ip=env.user_ip,
            is_new=not token_reused,
            login=login,
            region_id=region_info.id,
            surface='oauth_%s' % grant_type,
            token_id=token.id,
            track_id=passport_track_id,
            uid=token.uid,
            uids_count=1,
            user_agent=env.user_agent,
            yandexuid=env.cookies.get('yandexuid'),
        )


def get_access_token_from_refresh_token(refresh_token):
    """
    Возвращает тушку токена, связанного с переданным refresh-токеном, или None
    """
    try:
        _, access_token = decrypt(
            keys=settings.REFRESH_TOKEN_GENERATION_KEYS,
            encrypted_message=refresh_token,
        )
        return access_token.decode()
    except BaseCryptoError:
        return
