# -*- coding: utf-8 -*-
from django.conf import settings
from passport.backend.oauth.core.common.error_logs import log_warning
from passport.backend.oauth.core.db.eav import (
    BaseDBError,
    UPDATE,
)
from passport.backend.oauth.core.db.errors import PaymentAuthNotPassedError
from passport.backend.oauth.core.db.limits import (
    assert_is_allowed,
    restrict_non_yandex_clients,
)
from passport.backend.oauth.core.db.schemas import TOKEN_VIRTUAL_ATTR_IS_STATELESS
from passport.backend.oauth.core.db.scope import is_payment_auth_required
from passport.backend.oauth.core.db.token.base import (
    log_token_issue,
    TOKEN_TYPE_NORMAL,
    TOKEN_TYPE_STATELESS,
)
from passport.backend.oauth.core.db.token.normal_token import (
    issue_normal_token,
    Token,
)
from passport.backend.oauth.core.db.token.stateless_token import (
    issue_stateless_token,
    StatelessToken,
)
from passport.backend.oauth.core.db.turboapp import try_exclude_scopes_specific_for_turboapps


def parse_token(bb_response):
    token_attributes = bb_response.get('oauth', {}).get('token_attributes', {})
    is_stateless = bool(int(token_attributes.get(str(TOKEN_VIRTUAL_ATTR_IS_STATELESS), '0')))
    if is_stateless:
        return StatelessToken.parse(bb_response)
    else:
        return Token.parse(bb_response)


def issue_token(uid, client, grant_type, env, device_id=None, device_name=None, scopes=None, x_token_id=None,
                meta=None, statbox_logger=None, antifraud_logger=None, credentials_logger=None,
                make_alias=False, raise_error_if_limit_exceeded=False,
                allow_reuse=True, tokens_revoked_at=None, app_passwords_revoked_at=None,
                token_id=None, token_type=TOKEN_TYPE_NORMAL, allow_fallback_to_stateless=False,
                payment_auth_context_id=None, payment_auth_scope_addendum=None, password_passed=False,
                login=None, login_id=None, app_platform=None, redirect_uri=None, passport_track_id=None,
                is_xtoken_trusted=False, auth_source=None):

    scopes = try_exclude_scopes_specific_for_turboapps(
        client=client,
        scopes=set(scopes or client.scopes),
        redirect_uri=redirect_uri,
    )

    payment_auth_passed = bool(payment_auth_context_id and payment_auth_scope_addendum)
    if is_payment_auth_required(scopes) and not payment_auth_passed:
        # Сюда IRL мы попадать не должны, валидация должна случаться раньше. Тут проверка лишь для надёжности.
        raise PaymentAuthNotPassedError()

    assert_is_allowed(scopes, grant_type=grant_type, client_id=client.display_id, ip=env.consumer_ip, uid=uid)
    restrict_non_yandex_clients(grant_type=grant_type, client=client, user_ip=env.user_ip)

    if not device_id:
        device_id = None
        device_name = None

    common_token_params = dict(
        uid=uid,
        client=client,
        grant_type=grant_type,
        env=env,
        device_id=device_id,
        device_name=device_name,
        scopes=scopes,
        meta=meta,
        x_token_id=x_token_id,
        payment_auth_context_id=payment_auth_context_id,
        payment_auth_scope_addendum=payment_auth_scope_addendum,
        statbox_logger=statbox_logger,
        credentials_logger=credentials_logger,
        antifraud_logger=antifraud_logger,
        password_passed=password_passed,
        login=login,
        login_id=login_id,
        passport_track_id=passport_track_id,
        auth_source=auth_source,
    )
    if token_type == TOKEN_TYPE_NORMAL:
        try:
            return issue_normal_token(
                make_alias=make_alias,
                raise_error_if_limit_exceeded=raise_error_if_limit_exceeded,
                allow_reuse=allow_reuse,
                tokens_revoked_at=tokens_revoked_at,
                app_passwords_revoked_at=app_passwords_revoked_at,
                app_platform=app_platform,  # только для stateful, так как требуется только для показа списка токенов
                is_xtoken_trusted=is_xtoken_trusted,  # только для stateful, так как читается из БД
                **common_token_params
            )
        except BaseDBError:
            if not (settings.ALLOW_FALLBACK_TO_STATELESS_TOKENS and allow_fallback_to_stateless):
                raise
            # Фоллбечимся, выдавая stateless вместо пятисотки
            log_warning('issue_token: issuing stateless token instead of normal (client_id=%s)' % client.display_id)
            return issue_stateless_token(
                token_id=token_id,
                **common_token_params
            )

    elif token_type == TOKEN_TYPE_STATELESS:
        return issue_stateless_token(
            token_id=token_id,
            **common_token_params
        )
    else:
        raise NotImplementedError('Token type %s not implemented yet' % token_type)  # pragma: no cover


def try_refresh_token(token, client, env, statbox):
    if isinstance(token, StatelessToken):
        token = issue_token(
            uid=token.uid,
            client=client,
            grant_type='refresh_token',
            env=env,
            device_id=token.device_id,
            device_name=token.device_name,
            scopes=token.scopes,
            x_token_id=token.x_token_id,
            meta=token.meta,
            statbox_logger=statbox,
            token_id=token.id,
            token_type=TOKEN_TYPE_STATELESS,
            login_id=token.login_id,
        )
    else:
        with UPDATE(token):
            token.try_refresh()

        log_token_issue(
            token=token,
            client=client,
            grant_type='refresh_token',
            env=env,
            statbox_logger=statbox,
        )

    return token
