# -*- coding: utf-8 -*-
from collections import defaultdict
from datetime import (
    datetime,
    timedelta,
)
import logging
import random
from typing import List
import uuid
import zlib

from django.conf import settings
from django.utils.encoding import smart_bytes
from passport.backend.oauth.core.common.decorators import retry
from passport.backend.oauth.core.common.utils import (
    first_or_none,
    int_to_bytes,
    make_hash,
    make_token_alias,
    mask_string,
    now,
    to_base64_url,
)
from passport.backend.oauth.core.db.client import Client
from passport.backend.oauth.core.db.device_info import (
    app_platform_to_name,
    AppPlatform,
    parse_app_platform,
)
from passport.backend.oauth.core.db.eav import (
    BaseDBError,
    CREATE,
    DBTemporaryError,
    DELETE,
    Index,
    UPDATE,
)
from passport.backend.oauth.core.db.eav.errors import DBIntegrityError
from passport.backend.oauth.core.db.eav.model.mixins import EavModelDBMixin
from passport.backend.oauth.core.db.eav.sharder import get_sharder
from passport.backend.oauth.core.db.eav.types import DB_NULL
from passport.backend.oauth.core.db.eav.value_serializers import make_collection_serializer
from passport.backend.oauth.core.db.errors import TokenLimitExceededError
from passport.backend.oauth.core.db.schemas import (
    token_attributes_table,
    token_by_access_token_table,
    token_by_alias_table,
    token_by_params_table,
)
from passport.backend.oauth.core.db.scope import Scope
from passport.backend.oauth.core.db.token.base import (
    log_token_issue,
    TokenBase,
)
from passport.backend.oauth.core.logs.historydb import to_event_log
from passport.backend.oauth.core.logs.statbox import (
    StatboxLogger,
    to_statbox,
)
from passport.backend.utils.time import datetime_to_integer_unixtime


log = logging.getLogger('db.token')


class Token(EavModelDBMixin, TokenBase):
    _table = token_attributes_table
    _indexes = {
        'params': Index(token_by_params_table),
        'access_token': Index(token_by_access_token_table),
        'alias': Index(token_by_alias_table, nullable_fields=['alias']),
    }

    @classmethod
    def create(cls, uid, client_id, scopes, device_id, device_name=None, meta=None):
        datetime_now = now()
        token = cls(
            uid=uid,
            client_id=client_id,
            device_id=device_id,
            device_name=device_name,
            created=datetime_now,
            issued=datetime_now,
            meta=meta,
        )
        token.set_scopes(scopes)
        return token

    def pre_save_created(self, generated_id):
        super(Token, self).pre_save_created(generated_id)

        uid_byte_count = 8
        # соберём access_token из параметров
        shard_id = get_sharder(self.entity_name).shard_function(generated_id)
        shard_part = int_to_bytes(shard_id, 1)
        uid_part = int_to_bytes(self.uid, uid_byte_count)
        client_part = int_to_bytes(self.client_id, 4)
        token_part = int_to_bytes(generated_id, 8)
        random_part = uuid.uuid4().bytes  # 16 байт

        if settings.TOKEN_41_BYTE_FORMAT:
            crc_part = int_to_bytes(zlib.crc32(
                b''.join([shard_part, uid_part, client_part, token_part,  random_part]),
            ), 4)

            base = to_base64_url(
                b''.join([shard_part, uid_part, client_part, token_part, random_part, crc_part]),
            )

            self.access_token = f'y{str(settings.ENV_ID)}_{base.decode()}'
        else:
            self.access_token = to_base64_url(
                b''.join([shard_part, uid_part, client_part, random_part]),
            ).decode()

    @property
    def masked_body(self):
        # Возвращает замаскированную тушку, пригодную для записи в логи и отображения в админке
        string = self.alias if self.is_app_password else self.access_token
        show_first_n = len(string) // 4 if self.is_app_password else len(string) // 2
        return mask_string(string, show_first_n)

    @classmethod
    def by_access_token(cls, access_token):
        return first_or_none(cls.by_index('access_token', access_token=make_hash(access_token), limit=1))

    @classmethod
    def by_uid_and_alias(cls, uid, alias):
        return first_or_none(Token.by_index('alias', uid=uid, alias=make_hash(alias), limit=1))

    def set_scopes(self, scopes: list[Scope]):
        super(Token, self).set_scopes(scopes)
        self.is_refreshable = check_if_is_refreshable(scopes)
        self.reset_ttl()

    def reset_ttl(self):
        """Подновляет токен, невзирая на настройки "подновляемости" скоупов"""
        self.expires = now() + timedelta(seconds=self.total_ttl) if self.total_ttl else None

    def try_refresh(self):
        if self.ttl and self.can_be_operated and 0 < self.ttl <= self.total_ttl * settings.TOKEN_REFRESH_RATIO:
            self.reset_ttl()

    @property
    def can_be_refreshed(self) -> bool:
        """Можно ли прямо сейчас подновить токен"""
        if not (self.ttl and self.can_be_operated and self.is_refreshable):
            return False
        return 0 < self.ttl <= self.total_ttl * settings.TOKEN_REFRESH_RATIO

    def __eq__(self, other):
        return (
            self.id == other.id and
            self.uid == other.uid and
            self.device_id == other.device_id and
            self.access_token == other.access_token and
            self.scopes == other.scopes and
            self.meta == other.meta
        )

    def __hash__(self):
        return hash((
            self.id,
            self.uid,
            self.device_id,
            self.access_token,
            tuple(sorted(self.scope_ids)),
            self.meta,
        ))


def check_if_is_refreshable(scopes):
    return all(s.is_ttl_refreshable for s in scopes if s.ttl) and any(s.ttl for s in scopes)


def list_tokens_by_uid(uid, limit=None) -> List[Token]:
    """Возвращает все токены пользователя, в том числе, уже невалидные"""
    return Token.by_index('params', uid=uid, limit=limit)


def list_clients_by_user(uid, tokens_revoked_at=None, app_passwords_revoked_at=None):
    """Возвращает приложения, обладающие валидными токенами на указанного пользователя"""
    tokens = [token for token in list_tokens_by_uid(uid) if not token.is_expired]
    clients = Client.by_ids(entity_ids=set(token.client_id for token in tokens))

    result = set()
    for token in tokens:
        revoke_time = app_passwords_revoked_at if token.is_app_password else tokens_revoked_at
        client = clients.get(token.client_id)
        if client is not None and token.is_valid(client, revoke_time):
            result.add(client)

    return list(result)


def list_tokens_with_clients_by_user(uid, tokens_revoked_at=None, app_passwords_revoked_at=None,
                                     omit_deleted_clients=True, limit=None):
    """Возвращает валидные и невалидные токены указанного пользователя
    (вместе с приложениями и статусом валидности)"""
    tokens = [token for token in list_tokens_by_uid(uid, limit=limit)]
    clients = Client.by_ids(entity_ids=set(token.client_id for token in tokens))

    result = []
    for token in tokens:
        revoke_time = app_passwords_revoked_at if token.is_app_password else tokens_revoked_at
        client = clients.get(token.client_id)
        if client is not None:
            result.append((token, client, token.is_valid(client, revoke_time)))
        elif not omit_deleted_clients:
            result.append((token, None, False))

    return result


def get_valid_tokens_with_clients(tokens, clients, revoked_at=None):
    result = []
    for token in tokens:
        client = clients.get(token.client_id)
        if client is not None and token.is_valid(client, revoked_at):
            result.append((token, client))
    return result


def list_tokens_by_user_and_device(uid, device_id, revoked_at=None):
    """
    Получаем валидные токены пользователя с данным device_id
    У ПП device_id рандомный, они сюда не попадут
    """
    device_tokens = Token.by_index('params', uid=uid, device_id=smart_bytes(device_id))
    clients = Client.by_ids(entity_ids=set(token.client_id for token in device_tokens))

    return get_valid_tokens_with_clients(device_tokens, clients, revoked_at)


def list_app_passwords(uid, revoked_at=None):
    """
    Возвращает список валидных ПП пользователя
    :rtype: list
    """
    tokens = [token for token in list_tokens_by_uid(uid) if token.is_app_password]
    clients = Client.by_ids(entity_ids=set(token.client_id for token in tokens))

    return get_valid_tokens_with_clients(tokens, clients, revoked_at)


def split_tokens_to_devices(tokens_with_clients):
    """Разбивает токены на группы по device_id, агрегируя информацию об устройствах"""
    last_updated_tokens = defaultdict(dict)
    tokens_by_device = defaultdict(list)

    for token, client in tokens_with_clients:
        issued = last_updated_tokens[token.device_id].setdefault('issued', token.issued)
        last_updated_tokens[token.device_id].setdefault('device_name', token.device_name)
        last_updated_tokens[token.device_id].setdefault('app_platform', AppPlatform.Unknown)
        last_updated_tokens[token.device_id].setdefault('has_xtoken', False)

        current_is_xtoken = token.has_xtoken_grant

        if current_is_xtoken:
            last_updated_tokens[token.device_id]['app_platform'] = token.app_platform
            last_updated_tokens[token.device_id]['has_xtoken'] = current_is_xtoken
        if issued < token.issued and token.device_name:
            last_updated_tokens[token.device_id]['issued'] = token.issued
            last_updated_tokens[token.device_id]['device_name'] = token.device_name

        if not current_is_xtoken:
            tokens_by_device[token.device_id].append((token, client))

    return [
        {
            'device_id': device_id,
            'device_name': group_data['device_name'],
            'app_platform': app_platform_to_name(group_data['app_platform']),
            'has_xtoken': group_data['has_xtoken'],
            'tokens_with_clients': tokens_by_device.get(device_id, []),
        } for device_id, group_data in last_updated_tokens.items()
    ]


def _log_token_invalidation(token, client, env, reason, dt=None):
    to_statbox(
        dt=dt,
        mode='invalidate_tokens',
        status='ok',
        target='token',
        reason=reason,
        token_id=token.id,
        has_alias=token.is_app_password,
        uid=token.uid,
        client_id=client.display_id,
        device_id=token.device_id if token.device_id != DB_NULL else None,
        device_name=token.device_name or None,
        created=datetime_to_integer_unixtime(token.created),
        user_ip=env.user_ip,
        consumer_ip=env.consumer_ip,
        user_agent=env.user_agent,
        yandexuid=env.cookies.get('yandexuid'),
    )
    to_event_log(
        dt=dt,
        action='invalidate',
        target='token',
        reason=reason,
        uid=token.uid,
        token_id=token.id,
        client_id=client.display_id,
        device_id=token.device_id if token.device_id != DB_NULL else None,
        device_name=token.device_name or None,
        scopes=','.join(s.keyword for s in token.scopes),
        has_alias=token.is_app_password,
        user_ip=env.user_ip,
        consumer_ip=env.consumer_ip,
        user_agent=env.user_agent,
        yandexuid=env.cookies.get('yandexuid'),
    )


def invalidate_tokens(tokens, client, env, reason):
    datetime_now = datetime.now()
    for token in tokens:
        invalidate_single_token(token, client, env, reason, dt=datetime_now)


def invalidate_single_token(token, client, env, reason, dt=None):
    with UPDATE(token):
        token.expires = datetime.now()
    _log_token_invalidation(token, client, env, reason, dt=dt)


def delete_single_token(token, client, env, reason, dt=None):
    with DELETE(token):
        pass
    _log_token_invalidation(token, client, env, reason, dt=dt)


def get_existing_token(uid, client, scopes, device_id, revoke_time=None):
    existing_token = first_or_none(Token.by_index(
        'params',
        limit=1,
        uid=uid,
        client_id=client.id,
        scope_ids=make_collection_serializer()([scope.id for scope in scopes]),
        device_id=smart_bytes(device_id or DB_NULL),
    ))
    is_existing_token_valid = (
        existing_token is not None and
        existing_token.is_valid(client, revoke_time) and
        existing_token.can_be_operated
    )
    return is_existing_token_valid, existing_token


def get_already_granted_scopes(uid, client, device_id, revoke_time=None):
    existing_tokens_for_device = Token.by_index(
        'params',
        uid=uid,
        client_id=client.id,
        device_id=smart_bytes(device_id or DB_NULL),  # FIXME: неоптимальный запрос с точки зрения индекса
    )
    valid_tokens_for_device = [
        token for token in existing_tokens_for_device
        if token.is_valid(client=client, revoke_time=revoke_time)
    ]
    total_scopes = set()
    for token in valid_tokens_for_device:
        total_scopes |= token.scopes
    return total_scopes & client.scopes


def get_max_tokens_per_uid_and_client(uid=None, client_id=None):
    return (
        settings.MAX_TOKENS_PER_UID_AND_CLIENT__BY_CLIENT_ID_AND_UID.get(client_id, {}).get(uid) or
        settings.MAX_TOKENS_PER_UID_AND_CLIENT__BY_CLIENT_ID.get(client_id) or
        settings.MAX_TOKENS_PER_UID_AND_CLIENT__DEFAULT
    )


def get_oldest_token(tokens):
    """
    Возвращает токен, который можно удалить: самый "старый" из имеющих device_id.
    Мотивация:
    1) Не удалять токены без device_id, чтобы не ломать совместимость.
    2) Если приложение на одном и том же устройстве навыдавало себе токенов с разными device_id,
    то логично считать, что оно будет использовать последний, а самый первый можно удалять.
    """
    tokens_with_device_id = [token for token in tokens if token.device_id != DB_NULL]
    tokens_with_ttl = [token for token in tokens_with_device_id if token.expires]
    if tokens_with_ttl:
        return min(tokens_with_ttl, key=lambda token: token.expires)
    else:
        return min(tokens_with_device_id, key=lambda token: token.issued)


def get_token_to_reuse(tokens, scopes, device_id, client, revoke_time, fuzzy_scopes_check=True):
    """
    Возвращает пару can_be_reused, token_to_reuse.

    token_to_reuse - токен, имеющийся в БД и наиболее удовлетворяющий запросу.
    can_be_reused - можно ли выдавать (возможно, обновив) найденный токен
      (иначе - нужно его удалить и создать новый)
    """
    tokens_for_device = [
        token
        for token in tokens
        if token.device_id == (device_id or DB_NULL)
    ]

    # Попытаемся найти подходящий токен среди валидных выданных
    token_to_reuse = first_or_none(sorted(
        [
            token
            for token in tokens_for_device
            if (
                scopes.issubset(token.scopes) and
                token.is_valid(client=client, revoke_time=revoke_time) and
                token.can_be_operated
            )
        ],
        key=lambda t: len(t.scopes),  # выбираем токен с самым "жирным" скоупом
        reverse=True,
    ))
    can_be_reused = token_to_reuse is not None

    if token_to_reuse is None:
        # Не получилось - ищем точно такой же, но невалидный (чтобы его удалить)
        token_to_reuse = first_or_none([
            token
            for token in tokens_for_device
            if scopes == token.scopes
        ])
        can_be_reused = False

    if token_to_reuse is None and fuzzy_scopes_check:
        # Снова не получилось - выберем один валидный из остальных, чтобы добавить скоупы в него
        token_to_reuse = first_or_none(sorted(
            [
                token
                for token in tokens_for_device
                if (
                    token.is_valid(client=client, revoke_time=revoke_time) and
                    token.can_be_operated
                )
            ],
            key=lambda t: len(scopes - t.scopes),  # выбираем токен, в котором не хватает меньше всего скоупов
        ))
        can_be_reused = token_to_reuse is not None

    return can_be_reused, token_to_reuse


def issue_normal_token(uid, client, grant_type, env, device_id=None, device_name=None, scopes=None, meta=None,
                       x_token_id=None, statbox_logger=None, antifraud_logger=None, credentials_logger=None,
                       make_alias=False, raise_error_if_limit_exceeded=False,
                       allow_reuse=True, tokens_revoked_at=None, app_passwords_revoked_at=None,
                       payment_auth_context_id=None, payment_auth_scope_addendum=None, password_passed=False,
                       login=None, login_id=None, app_platform=None, passport_track_id=None,
                       is_xtoken_trusted=False, auth_source=None):
    statbox_logger = statbox_logger or StatboxLogger(mode='issue_token')
    scopes = set(scopes or client.scopes)
    revoke_time = app_passwords_revoked_at if make_alias else tokens_revoked_at

    if device_name:
        statbox_logger.bind(device_name=device_name)

    app_platform = parse_app_platform(app_platform)

    # не используем get_existing_token, так как нам нужно получить все токены (для проверки лимита)
    existing_tokens = Token.by_index(
        'params',
        uid=uid,
        client_id=client.id,
    )
    can_be_reused, token_to_reuse = get_token_to_reuse(
        tokens=existing_tokens,
        scopes=scopes,
        device_id=device_id,
        client=client,
        revoke_time=revoke_time,
        fuzzy_scopes_check=allow_reuse,  # если надо выдавать новый токен, нет смысла проверять похожие
    )
    should_reuse = allow_reuse and can_be_reused
    datetime_now = datetime.now()

    if should_reuse:
        # Если токен уже существует, не протух и не собирается протухать, то мы его реюзаем.
        # При этом в некоторых случаях при ошибке БД можем отдавать не пятисотку, а старый токен.
        force_db_write = (
            token_to_reuse.meta != (meta or '') or
            token_to_reuse.device_name != (device_name or '') or
            token_to_reuse.payment_auth_context_id != (payment_auth_context_id or '') or
            token_to_reuse.payment_auth_scope_addendum != (payment_auth_scope_addendum or '') or
            (is_xtoken_trusted and not token_to_reuse.is_xtoken_trusted) or
            not scopes.issubset(token_to_reuse.scopes)
        )
        if (
            not force_db_write and
            token_to_reuse.issued >= now() - timedelta(seconds=settings.MINIMAL_TIME_BETWEEN_TOKEN_UPDATING)
        ):
            # Даже и не пытаемся писать в БД - нечего нас так часто дёргать
            log.info('issue_token: not attempting to refresh token with id=%s' % token_to_reuse.id)
            token = token_to_reuse
        else:
            try:
                with UPDATE(token_to_reuse) as token:
                    token.meta = meta
                    token.x_token_id = x_token_id
                    token.issued = now()
                    token.device_name = device_name
                    token.payment_auth_context_id = payment_auth_context_id
                    token.payment_auth_scope_addendum = payment_auth_scope_addendum
                    token.login_id = login_id
                    token.app_platform = app_platform
                    token.is_xtoken_trusted |= is_xtoken_trusted
                    token.set_scopes(token.scopes | scopes)
                    token.reset_ttl()
            except DBTemporaryError:
                if force_db_write:
                    # Без записи в БД токен не отдадим
                    raise
                else:
                    # Обойдёмся без подновления issue_time и прочих неважных полей
                    log.warning('issue_token: failed to refresh token with id=%s' % token_to_reuse.id)
    else:
        if token_to_reuse is not None and token_to_reuse.scopes == scopes:
            # Из-за ограничений уникальности в БД надо сперва удалить старый токен
            delete_single_token(token_to_reuse, client, env, reason='delete_expired', dt=datetime_now)
        elif device_id is not None:
            # Перед выдачей нового токена проверяем, не превысили ли лимит.
            limit = get_max_tokens_per_uid_and_client(uid=uid, client_id=client.display_id)
            if len(existing_tokens) >= limit:
                valid_tokens, invalid_tokens = [], []
                for existing_token in existing_tokens:
                    if existing_token.is_valid(client=client, revoke_time=revoke_time):
                        valid_tokens.append(existing_token)
                    else:
                        invalid_tokens.append(existing_token)

                to_statbox(
                    mode='issue_token',
                    grant_type=grant_type,
                    status='warning',
                    reason='limit.exceeded',
                    uid=uid,
                    client_id=client.display_id,
                    user_ip=env.user_ip,
                    limit=limit,
                    valid_token_count=len(valid_tokens),
                    total_token_count=len(existing_tokens),
                )
                if len(valid_tokens) >= limit and raise_error_if_limit_exceeded:
                    # отказываем в выдаче токена
                    raise TokenLimitExceededError()

                # Пытаемся удалить несколько старых токенов.
                tokens_with_statuses_to_delete = []
                max_count_to_delete = settings.MAX_TOKENS_TO_DELETE_WHEN_ISSUING_NEW
                # Валидные удаляем сверх лимита, начиная с самых старых.
                while len(valid_tokens) >= limit and len(tokens_with_statuses_to_delete) < max_count_to_delete:
                    oldest_token = get_oldest_token(valid_tokens)
                    valid_tokens.remove(oldest_token)
                    tokens_with_statuses_to_delete.append((oldest_token, True))
                # Невалидные удаляем в случайном порядке, чтобы минимизировать влияние гонок.
                while invalid_tokens and len(tokens_with_statuses_to_delete) < max_count_to_delete:
                    random_token = random.choice(invalid_tokens)
                    invalid_tokens.remove(random_token)
                    tokens_with_statuses_to_delete.append((random_token, False))

                for token_to_delete, is_valid in tokens_with_statuses_to_delete:
                    # Нет необходимости удалять одной транзакцией: даже если один из запросов отвалится - лучше
                    # удалить хоть что-то.
                    delete_single_token(
                        token_to_delete,
                        client,
                        env,
                        reason='limit_reached' if is_valid else 'delete_expired',
                        dt=datetime_now,
                    )

        try:
            # При выдаче ПП ретраимся и при ошибках сети, и при коллизиях алиасов.
            # При выдаче обычного токена - только при ошибках сети.
            @retry(
                exception_to_catch=BaseDBError if make_alias else DBTemporaryError,
                retries=settings.TOKEN_ALIAS_GENERATION_RETRIES,
                logger=log,
                name_for_logger='Database',
            )
            def create():
                with CREATE(Token.create(
                    uid=uid,
                    client_id=client.id,
                    scopes=scopes,
                    device_id=device_id or DB_NULL,
                    device_name=device_name,
                    meta=meta,
                ), retries=1) as token:
                    token.x_token_id = x_token_id
                    token.payment_auth_context_id = payment_auth_context_id
                    token.payment_auth_scope_addendum = payment_auth_scope_addendum
                    token.login_id = login_id
                    token.app_platform = app_platform
                    token.is_xtoken_trusted = is_xtoken_trusted
                    if make_alias:
                        token.alias = make_token_alias()
                return token
            token = create()
        except DBIntegrityError:
            # Если это действительно гоночная ситуация, то просто отдадим токен
            is_valid, token = get_existing_token(uid, client, scopes, device_id, revoke_time)
            if not is_valid:
                log.warning('issue_token: unable to get or issue token (client_id=%s)', client.display_id)
                raise
            log.warning(
                'issue_token: probable race condition for token with id=%s (client_id=%s)' % (
                    token.id,
                    client.display_id,
                ),
            )

    log_token_issue(
        token=token,
        client=client,
        grant_type=grant_type,
        dt=datetime_now,
        env=env,
        statbox_logger=statbox_logger,
        antifraud_logger=antifraud_logger,
        credentials_logger=credentials_logger,
        password_passed=password_passed,
        token_reused=should_reuse,
        login=login,
        passport_track_id=passport_track_id,
        auth_source=auth_source,
    )
    return token
