# -*- coding: utf-8 -*-
from datetime import timedelta
from time import time

from django.conf import settings
from passport.backend.oauth.core.common.blackbox import get_blackbox
from passport.backend.oauth.core.common.utils import now
from passport.backend.oauth.core.db.eav.sharder import get_sharder
from passport.backend.oauth.core.db.eav.types import DB_NULL
from passport.backend.oauth.core.db.token.base import (
    get_ttl_by_scopes,
    log_token_issue,
    TOKEN_TYPE_STATELESS,
    TokenBase,
)
from passport.backend.oauth.core.logs.statbox import StatboxLogger


class StatelessToken(TokenBase):
    @classmethod
    def create(cls, token_id, access_token, uid, client, scopes, device_id, device_name=None,
               meta=None, x_token_id=None, ttl=None,
               payment_auth_context_id=None, payment_auth_scope_addendum=None, login_id=None):
        create_time = now()
        expire_time = create_time + timedelta(seconds=ttl or get_ttl_by_scopes(scopes))
        return cls(
            entity_id=token_id,
            access_token=access_token,
            uid=uid,
            client_id=client.id,
            scope_ids=[scope.id for scope in scopes],
            device_id=device_id or DB_NULL,
            device_name=device_name,
            created=create_time,
            issued=create_time,
            expires=expire_time,
            meta=meta,
            x_token_id=x_token_id,
            payment_auth_context_id=payment_auth_context_id,
            payment_auth_scope_addendum=payment_auth_scope_addendum,
            login_id=login_id,
        )

    @property
    def can_be_refreshed(self):
        return False

    @property
    def masked_body(self):
        # Возвращает замаскированную тушку, пригодную для записи в логи и отображения в админке
        body, _, signature = self.access_token.rpartition('.')
        return '%s.*' % body


def issue_stateless_token(uid, client, grant_type, env, device_id=None, device_name=None, scopes=None,
                          meta=None, x_token_id=None, token_id=None, statbox_logger=None, antifraud_logger=None,
                          credentials_logger=None, payment_auth_context_id=None, payment_auth_scope_addendum=None,
                          password_passed=False, login=None, login_id=None, passport_track_id=None, auth_source=None):
    statbox_logger = statbox_logger or StatboxLogger(mode='issue_token')
    scopes = set(scopes or client.scopes)
    ttl = get_ttl_by_scopes(scopes) or settings.DEFAULT_TOKEN_TTL

    # будут ли скоупы токена меняться при изменении скоупов приложения
    are_scopes_expandable = (
        scopes == client.scopes and
        client.is_yandex
    )

    rv = get_blackbox().create_oauth_token(
        uid=uid,
        ip=env.user_ip,
        client_id=client.id,
        scope_ids=None if are_scopes_expandable else [scope.id for scope in scopes],
        expire_time=int(time()) + ttl,
        device_id=device_id,
        token_id=token_id,
        xtoken_id=x_token_id,
        xtoken_shard=get_sharder('token').shard_function(int(x_token_id)) if x_token_id else None,
        meta=meta,
        payment_auth_context_id=payment_auth_context_id,
        payment_auth_scope_addendum=payment_auth_scope_addendum,
        login_id=login_id,
    )

    token = StatelessToken.create(
        token_id=rv['token_id'],
        access_token=rv['access_token'],
        uid=uid,
        client=client,
        scopes=scopes,
        device_id=device_id,
        device_name=device_name,
        meta=meta,
        x_token_id=x_token_id,
        ttl=ttl,
        payment_auth_context_id=payment_auth_context_id,
        payment_auth_scope_addendum=payment_auth_scope_addendum,
        login_id=login_id,
    )

    log_token_issue(
        token=token,
        client=client,
        grant_type=grant_type,
        env=env,
        statbox_logger=statbox_logger,
        antifraud_logger=antifraud_logger,
        credentials_logger=credentials_logger,
        token_type=TOKEN_TYPE_STATELESS,
        password_passed=password_passed,
        login=login,
        passport_track_id=passport_track_id,
        auth_source=auth_source,
    )
    return token
