# -*- coding: utf-8 -*-
import logging
from urllib.parse import (
    urlparse,
    urlunparse,
)

from django.conf import settings
from passport.backend.oauth.core.common.error_logs import log_warning
from passport.backend.oauth.core.db.config import get_client_lists
from passport.backend.oauth.core.db.scope import Scope
from passport.backend.oauth.core.logs.statbox import to_statbox


log = logging.getLogger('db.turboapp')


TURBOAPP_URL_SCHEME = 'yandexta'


def _get_host(url):
    parsed = urlparse(url)
    host = parsed.netloc.encode('idna').lower()
    return host


def make_turboapp_redirect_uri(turboapp_base_url):
    scheme, host, path, params, query, fragment = urlparse(turboapp_base_url)
    scheme = TURBOAPP_URL_SCHEME
    path = params = query = fragment = ''
    return urlunparse([scheme, host, path, params, query, fragment])


def is_redirect_uri_turboapp(redirect_uri):
    return redirect_uri.startswith('%s://' % TURBOAPP_URL_SCHEME)


def is_psuid_allowed_for_redirect_uri(client, redirect_uri):
    # FIXME: убрать импорты на уровень модуля после того, как oauth/tvm_api перестанет зависеть от oauth/core
    from passport.backend.core.turboapp_partners import is_psuid_allowed
    from passport.backend.core.ydb.exceptions import BaseYdbError

    error = 'redirect_uri.missing'
    error_description = 'redirect_uri is missing'
    if redirect_uri:
        host = _get_host(redirect_uri)
        try:
            if is_psuid_allowed(host):
                return True
            error = 'redirect_uri.not_trusted'
            error_description = 'redirect_uri=%r is not trusted (host=%r)' % (redirect_uri, host)
        except BaseYdbError as e:
            log_warning('YDB request failed: %s %s' % (e.__class__.__name__, e))
            error = 'ydb.unavailable'
            error_description = 'YDB unavailable'

    log.warning('Client %s not allowed to get PSUID: %s', client.display_id, error_description)
    to_statbox(
        mode='issue_psuid',
        action='turboapp_validation',
        status='error',
        error=error,
        client_id=client.display_id,
        redirect_uri=redirect_uri,
    )
    return False


def try_exclude_scopes_specific_for_turboapps(client, scopes, redirect_uri):
    """
    Убирает специфичные скоупы из переданного списка, если они не разрешены к использованию в данном случае
    """

    if settings.DEFAULT_PHONE_SCOPE_KEYWORD is not None:
        # FIXME: убрать импорты на уровень модуля после того, как oauth/tvm_api перестанет зависеть от oauth/core
        from passport.backend.core.turboapp_partners import get_partner_id_by_host
        from passport.backend.core.ydb.exceptions import BaseYdbError

        default_phone_scope = Scope.by_keyword(settings.DEFAULT_PHONE_SCOPE_KEYWORD)
        if default_phone_scope in scopes:
            allow_default_phone_scope = False
            error = ''
            error_description = ''

            if client.display_id in get_client_lists().whitelist_for_scope(settings.DEFAULT_PHONE_SCOPE_KEYWORD):
                log.debug(
                    'Client %s is allowed to use scope %s, because it is whitelisted',
                    client.display_id,
                    default_phone_scope.keyword,
                )
                allow_default_phone_scope = True
            elif not redirect_uri:
                error = 'redirect_uri.missing'
                error_description = 'redirect_uri is missing'
            elif not is_redirect_uri_turboapp(redirect_uri):
                error = 'redirect_uri.not_turboapp'
                error_description = 'redirect_uri=%r is not turboapp' % redirect_uri
            else:
                host = _get_host(redirect_uri)
                try:
                    if get_partner_id_by_host(host) is not None:
                        allow_default_phone_scope = True
                    else:
                        error = 'redirect_uri.not_trusted'
                        error_description = 'redirect_uri=%r is not trusted (host=%r)' % (redirect_uri, host)
                except BaseYdbError as e:
                    log_warning('YDB request failed: %s %s' % (e.__class__.__name__, e))
                    error = 'ydb.unavailable'
                    error_description = 'YDB unavailable'

            if not allow_default_phone_scope:
                scopes.discard(default_phone_scope)
                log.warning(
                    'Client %s not allowed to use scope %s: %s',
                    client.display_id,
                    default_phone_scope.keyword,
                    error_description,
                )
                to_statbox(
                    mode='issue_token',
                    action='turboapp_validation',
                    status='error',
                    error=error,
                    client_id=client.display_id,
                    redirect_uri=redirect_uri,
                    scope=default_phone_scope.keyword,
                )

    return scopes
