# -*- coding: utf-8 -*-
from collections import namedtuple

from django.conf import settings
from passport.backend.oauth.core.common.portallib import (
    get_net,
    InvalidIpHeaderError,
)
from passport.backend.oauth.core.db.limits import check_substitute_user_ip_grant
from werkzeug.http import parse_cookie


class Environment(namedtuple(
    'EnvBase',
    'user_ip consumer_ip user_agent raw_cookies cookies host request_id authorization service_ticket user_ticket',
)):
    @classmethod
    def from_request(cls, request):
        host = request.headers.get('Host')
        user_ip = consumer_ip = request.headers.get('X-Real-Ip', request.META['REMOTE_ADDR'])
        user_agent = request.headers.get('User-Agent')
        service_ticket = None
        user_ticket = None

        if request.headers.get('Ya-Client-Cookie'):
            cookie_header = request.headers['Ya-Client-Cookie']
            cookies = dict(parse_cookie(cookie_header))
        else:
            cookie_header = request.headers.get('Cookie')
            cookies = dict(request.COOKIES)

        if settings.ENV_TYPE == 'development' or host in settings.INTERNAL_HOSTS:
            # Для internal-хостов разрешаем явно передавать значения в заголовках,
            # а ip пользователя - ещё и в параметре
            user_ip = request.headers.get('Ya-Consumer-Client-Ip', request.REQUEST.get('user_ip', user_ip))
            user_agent = request.headers.get('Ya-Client-User-Agent', user_agent)
            host = request.headers.get('Ya-Client-Host', host)
            service_ticket = request.headers.get('X-Ya-Service-Ticket')
            user_ticket = request.headers.get('X-Ya-User-Ticket')

        # Если запрос пришёл из mobileproxy через украинский туннель, то нам нужно заменить user_ip
        # на переданный в хедере. Чтобы убедиться, что это наш случай, проверим, является ли user_ip
        # адресом туннельного балансера.
        # На этом этапе нет нужды пропускать user_ip_raw через IPReg, так как после туннельного балансера
        # запрос не проходит через недоверенные прокси.
        user_ip_from_proxy = request.headers.get('X-YProxy-Header-Ip')
        if user_ip_from_proxy and check_substitute_user_ip_grant(user_ip):
            user_ip = user_ip_from_proxy

        try:
            # Используем ipreg для отфильтровывания Турбо ЯБраузера
            user_ip = get_net(str(user_ip), request.headers)['real_ip']
            # consumer_ip во внутреннем инстансе не фильтруем (там он заведомо ятимный)
            if host not in settings.INTERNAL_HOSTS:
                consumer_ip = get_net(str(consumer_ip), request.headers)['real_ip']
        except InvalidIpHeaderError:
            pass

        return cls(
            host=host,
            user_ip=user_ip,
            consumer_ip=consumer_ip,
            user_agent=user_agent,
            raw_cookies=cookie_header,
            cookies=cookies,
            request_id=request.headers.get('X-Request-Id'),
            authorization=request.headers.get('Ya-Consumer-Authorization', request.headers.get('Authorization')),
            service_ticket=service_ticket,
            user_ticket=user_ticket,
        )
