# -*- coding: utf-8 -*-
from contextlib import contextmanager
from datetime import datetime

from passport.backend.oauth.core.common.utils import escape


TSKV_MAX_VALUE_LENGTH = 1000


def tskv_value(value):
    if isinstance(value, bool):
        return '1' if value else '0'
    else:
        return escape(value)[:TSKV_MAX_VALUE_LENGTH]


class BaseTskvLogger(object):
    """
    TSKV-логгер. Имеет контекст (данные, которые записываются при каждой записи)
    и текущие данные (которые будут единоразово записаны при следующем вызове log).
    """
    def __init__(self, logger, **context):
        self._logger = logger
        self._context = context.copy()
        self._current_values = {}
        self._context_stack = []

    def get_child(self, **context):
        """Возвращает новый логгер с контекстом, равным текущему контексту + context"""
        child_logger = self.__class__(**self._context)
        child_logger.bind_context(**context)
        return child_logger

    def bind(self, **values):
        """Добавляет значения к текущим данным"""
        self._current_values.update(values)

    def bind_context(self, **context):
        """Добавляет значения к контексту логгера"""
        self._context.update(context)

    @property
    def all_values(self):
        """Возвращает все значения, которые ожидают записи в лог"""
        all_values = self._context.copy()
        all_values.update(self._current_values)
        return all_values

    @property
    def has_data(self):
        return bool(self.all_values)

    def _dump(self):
        entry = '\t'.join(
            '%s=%s' % (key, tskv_value(value))
            for key, value in self.all_values.items()
            if value is not None
        )
        self._logger.debug(entry)

    def _format_time_values(self, datetime_):
        raise NotImplementedError()  # pragma: no cover

    def log(self, dt=None, **values):
        """Пишет в лог время + контекст + текущие данные + values, очищает текущие данные"""
        self.bind(**values)
        self.bind(**self._format_time_values(dt or datetime.now()))
        if self.has_data:
            self._dump()
            self._current_values = {}

    @contextmanager
    def make_context(self, **context):
        """
        Этот контекстный менеджер может на время своей работы дополнить
        или переписать контекст.
        """
        self._context_stack.append(self._context.copy())
        self._context.update(context)
        try:
            yield
        finally:
            self._context = self._context_stack.pop()
