# -*- coding: utf-8 -*-
import logging

from passport.backend.core.logging_utils.formatters import ExceptionFormatter as BaseExceptionFormatter
from passport.backend.core.logging_utils.handlers import ExceptionHandler as BaseExceptionHandler
from passport.backend.oauth.core.common.globals import get_request
from passport.backend.oauth.core.common.utils import mask_string


SENSITIVE_FIELDS = (
    'password',
    'client_secret',
    'old_client_secret',
    'secret',
    'jwt_secret',
    'access_token',
    'oauth_token',
    'token_alias',
    'refresh_token',
    'sessionid',
    'code',
    'otp',
    'track',
    'credential_to_search',
)


def mask_sensitive_fields(data, fields=SENSITIVE_FIELDS):
    if isinstance(data, (list, tuple, set)):
        return [
            mask_sensitive_fields(item, fields)
            for item in data
        ]
    elif isinstance(data, dict):
        data = data.copy()
        for field, value in data.items():
            if field in fields:
                if isinstance(value, list):
                    # В функцию может приходить QueryDict,
                    # поэтому value может быть списком длины 1
                    value = value[0]
                if field in ('oauth_token', 'access_token', 'refresh_token'):
                    data[field] = mask_string(value, len(value) // 2)
                elif field == 'token_alias':
                    data[field] = mask_string(value, len(value) // 4)
                elif field == 'track' and '.' in value:
                    data[field] = mask_string(value, value.rfind('.') + 1)
                else:
                    data[field] = '*****'
            else:
                data[field] = mask_sensitive_fields(value, fields)
    return data


class ExceptionFormatter(BaseExceptionFormatter):
    def make_full_request_info(self, record):
        lines = []
        if getattr(record, 'request', None):
            lines.append('Request id: %s' % record.request.headers.get('X-Request-Id', ''))
            lines.append('Path: %s' % record.request.path)
            lines.append('GET: %s' % mask_sensitive_fields(record.request.GET))
            lines.append('POST: %s' % mask_sensitive_fields(record.request.POST))
        return lines


class ExceptionHandler(BaseExceptionHandler):
    def __init__(self, log_path):
        super(ExceptionHandler, self).__init__(log_path, formatter_class=ExceptionFormatter)


class RequestIdFilter(logging.Filter):
    def filter(self, record):
        request = get_request()
        if request is not None:
            record.request_id = request.env.request_id or ''
        else:
            # юнит-тесты, где реквеста просто нет
            record.request_id = ''
        return True
