# -*- coding: utf-8 -*-
from passport.backend.library.nginx_config_generator import BaseConfig
import yenv


class BaseOAuthConfig(BaseConfig):
    _localhost_ips = []

    @property
    def is_applicable(self):
        return super(BaseOAuthConfig, self).is_applicable and bool(self._get_server_names())


class BaseOAuthExternalConfig(BaseOAuthConfig):
    _cert_names = {
        'intranet': {
            'development': None,
            'testing': 'oauth-yateam-test',
            'rc': 'oauth-yateam-rc',
            'production': 'oauth-yateam',
        },
        'localhost': {
            'development': 'oauth-dev',
            'testing': 'oauth-test',
            'rc': 'oauth-rc',
            'production': 'oauth',
        },
        'stress': {
            'stress': 'oauth-load',
        },
    }
    _slb_ips = {
        'intranet': {
            'development': (),
            'testing': (
                '95.108.254.9',
                '2a02:6b8:0:3400::4:9',
            ),
            'rc': (
                '93.158.134.16',
                '2a02:6b8::1e8',
            ),
            'production': (
                '87.250.250.24',
                '2a02:6b8::3:24',
            ),
        },
        'localhost': {
            'development': (),
            'testing': (
                '95.108.254.59',
                '2a02:6b8:0:3400::3:59',
            ),
            'rc': (
                '213.180.193.114',
                '2a02:6b8::2:114',
            ),
            'production': (
                '93.158.134.114',
                '2a02:6b8::1:114',
            ),
        },
        'stress': {
            'stress': (),
        },
    }


class YandexOauthApiInternal(BaseOAuthConfig):
    _cert_names = {
        'intranet': {
            'development': None,
            'testing': 'oauth-yateam-test-internal',
            'rc': 'oauth-yateam-rc-internal',
            'production': 'oauth-yateam-internal',
        },
        'localhost': {
            'development': None,
            'testing': 'oauth-test-internal',
            'rc': 'oauth-rc-internal',
            'production': 'oauth-internal',
        },
        'stress': {
            'stress': 'oauth-load',
        },
    }
    _server_names = {
        'intranet': {
            'development': None,
            'testing': ['oauth-test-internal.yandex-team.ru'],
            'rc': ['oauth-rc-internal.yandex-team.ru'],
            'production': ['oauth-internal.yandex-team.ru'],
        },
        'localhost': {
            'development': None,
            'testing': ['oauth-test-internal.yandex.ru'],
            'rc': ['oauth-rc-internal.yandex.ru'],
            'production': ['oauth-internal.yandex.ru'],
        },
        'stress': {
            'stress': ['oauth-load-internal.yandex.ru'],
        },
    }
    _slb_ips = {
        'intranet': {
            'development': (),
            'testing': (
                '2a02:6b8:0:3400::922',
            ),
            'rc': (
                '93.158.157.252',
                '2a02:6b8:0:3400:0:e:0:10',
            ),
            'production': (
                '141.8.146.84',
                '2a02:6b8:0:3400::1:84',
            ),
        },
        'localhost': {
            'development': (),
            'testing': (
                '95.108.254.60',
                '2a02:6b8:0:3400::2:60',
            ),
            'rc': (
                '213.180.205.121',
                '2a02:6b8:0:3400::4:121',
            ),
            'production': (
                '93.158.157.1',
                '2a02:6b8:0:3400::1:1',
            ),
        },
        'stress': {
            'stress': (),
        },
    }
    # Порты, на которых доступно внутреннее АПИ.
    # Для торчащих наружу хостов нельзя ставить 80 порт.
    # (Для SLB нового образца (как в Финке) неактуально, там всегда 80)
    _internal_ports = {
        'intranet': {
            'development': None,
            'testing': 8080,
            'rc': 8080,
            'production': 8080,
        },
        'localhost': {
            'development': None,
            'testing': 8080,
            'rc': 8080,
            'production': 8080,
        },
        'stress': {
            'stress': 80,
        },
    }
    _internal_ssl_ports = {
        'intranet': {
            'development': None,
            'testing': 8443,
            'rc': 8443,
            'production': 8443,
        },
        'localhost': {
            'development': None,
            'testing': 8443,
            'rc': 8443,
            'production': 8443,
        },
        'stress': {
            'stress': 443,
        },
    }

    def _get_internal_port(self):
        return self._internal_ports.get(yenv.name, {}).get(yenv.type)

    def _get_internal_ssl_port(self):
        return self._internal_ssl_ports.get(yenv.name, {}).get(yenv.type)

    def get_custom_settings(self):
        return dict(
            super(YandexOauthApiInternal, self).get_custom_settings(),
            internal_port=self._get_internal_port(),
            internal_ssl_port=self._get_internal_ssl_port(),
        )


class YandexOauthApi(BaseOAuthExternalConfig):
    _server_name_prefixes = {
        'intranet': {
            'development': None,
            'testing': 'oauth-test.yandex-team',
            'rc': 'oauth-rc.yandex-team',
            'production': 'oauth.yandex-team',
        },
        'localhost': {
            'development': 'oauth-dev-bionic.yandex',
            'testing': 'oauth-test.yandex',
            'rc': 'oauth-rc.yandex',
            'production': 'oauth.yandex',
        },
        'stress': {
            'stress': 'oauth-load.yandex',
        },
    }
    _tld_ru = ['ru']
    _default_tlds = [
        'ru',
        'ua',
        'com',
        'com.tr',
        'by',
        'kz',
        'fr',  # Франция
        'az',  # Азербайджан
        'com.am',  # Армения
        'com.ge',  # Грузия
        'co.il',  # Израиль
        'kg',  # Киргизия
        'lt',  # Латвия
        'lv',  # Литва
        'md',  # Молдавия
        'tj',  # Таджикистан
        'tm',  # Туркмения
        'uz',  # Узбекистан
        'ee',  # Эстония
    ]

    def _get_tlds(self):
        return self._tld_ru if yenv.name in ('intranet', 'stress') else self._default_tlds


class YandexOauthAdmin(BaseOAuthExternalConfig):
    _server_names = {
        'intranet': {
            'development': None,
            'testing': ['admin.oauth-test.yandex-team.ru'],
            'rc': ['admin.oauth-rc.yandex-team.ru'],
            'production': ['admin.oauth.yandex-team.ru'],
        },
        'localhost': {
            'development': ['admin.oauth-dev-bionic.yandex.ru'],
            'testing': ['admin.oauth-test.yandex.ru'],
            'rc': None,
            'production': None,
        },
        'stress': {
            'stress': None,
        },
    }

    def __init__(self):
        super(YandexOauthAdmin, self).__init__()
        # В ятим-проде по соображениям безопасности используем отдельный внутренний балансер (недоступный из внешней
        # сети). В остальных окружениях дефолтные балансеры и так внутренние.
        self._slb_ips['intranet']['production'] = (
            '141.8.146.147',
            '2a02:6b8:0:3400:0:e:0:3',
        )


class YandexLogin(BaseOAuthExternalConfig):
    _server_names = {
        'intranet': {
            'development': None,
            'testing': ['login-test.yandex-team.ru'],
            'rc': ['login-rc.yandex-team.ru'],
            'production': ['login.yandex-team.ru'],
        },
        'localhost': {
            'development': ['login.oauth-dev-bionic.yandex.ru'],
            'testing': ['login-test.yandex.ru'],
            'rc': ['login-rc.yandex.ru'],
            'production': ['login.yandex.ru'],
        },
        'stress': {
            'stress': ['login-load.yandex.ru'],
        },
    }


class YandexOpenID(BaseOAuthExternalConfig):
    template_name = 'yandex_openid'
    result_name = 'yandex_openid'

    _server_names = {
        'intranet': {
            'development': None,
            'testing': None,
            'rc': None,
            'production': None,
        },
        'localhost': {
            'development': None,
            'testing': None,
            'rc': None,
            'production': ['openid.yandex.ru', 'www.openid.yandex.ru'],
        },
        'stress': {
            'stress': None,
        },
    }


CONFIG_CLASSES_WITH_PRIORITIES = {
    0: YandexOauthApi,
    10: YandexOauthApiInternal,
    20: YandexLogin,
    30: YandexOauthAdmin,
    90: YandexOpenID,
}
