# -*- coding: utf-8 -*-
from __future__ import absolute_import

from datetime import datetime
import sys

from passport.backend.oauth.settings.common_settings.common import (
    DEBUG,
    IS_UNDER_TESTS,
)
import pytz


TIME_ZONE_OFFSET = datetime.now(tz=pytz.timezone('Europe/Moscow')).strftime('%z')
STATBOX_DATETIME_FORMAT = '%Y-%m-%d %H:%M:%S'
GRAPHITE_DATETIME_FORMAT = '%d/%b/%Y:%H:%M:%S'


def make_logging_config(log_path, extra_formatters=None, extra_handlers=None, extra_loggers=None):
    if IS_UNDER_TESTS:
        return {}

    if DEBUG:
        log_path = ''
        historydb_log_path = log_path + 'historydb.'
    else:
        historydb_log_path = log_path + 'historydb/'

    logging_config = {
        'version': 1,
        'disable_existing_loggers': False,
        'filters': {
            'request_id': {
                '()': 'passport.backend.oauth.core.logs.common.RequestIdFilter',
            },
        },
        'formatters': {
            'default': {'()': 'ylog.FileFormatter'},
            'simple': {
                'format': '%(message)s',
            },
            'statbox': {
                'format': 'tskv\trequest_id=%(request_id)s\t%(message)s',
            },
            'historydb': {
                'format': 'v=1\t%(message)s',
            },
            'graphite': {
                'format': 'tskv\t%(message)s\trequest_id=%(request_id)s',
            },
            'passport_graphite': {
                'format': '%(message)s\trequest_id=%(request_id)s',
            },
            'request': {
                'format': '%(asctime)s %(request_id)s %(name)-15s %(levelname)-10s %(message)s',
            },
            'console': {
                'format': '%(asctime)s %(levelname)-10s %(message)s',
            },
        },
        'handlers': {
            'null': {
                'class': 'logging.NullHandler',
            },
            'console': {
                'class': 'logging.StreamHandler',
                'stream': sys.stdout,
                'level': 'DEBUG',
                'formatter': 'console',
            },
            'exception': {
                'class': 'passport.backend.oauth.core.logs.common.ExceptionHandler',
                'log_path': log_path,
                'level': 'ERROR',
            },
            'statbox': {
                'class': 'ylog.handlers.OSFileHandler',
                'filename': log_path + 'statbox.log',
                'formatter': 'statbox',
                'filters': ['request_id'],
            },
            'historydb_event': {
                'class': 'ylog.handlers.OSFileHandler',
                'filename': historydb_log_path + 'oauth.event.log',
                'formatter': 'historydb',
            },
            'graphite': {
                'class': 'ylog.handlers.OSFileHandler',
                'filename': log_path + 'graphite/graphite.log',
                'formatter': 'graphite',
                'filters': ['request_id'],
            },
            'passport_graphite': {
                'class': 'ylog.handlers.OSFileHandler',
                'filename': log_path + 'graphite/graphite.log',
                'formatter': 'passport_graphite',
                'filters': ['request_id'],
            },
            'debug': {
                'class': 'ylog.handlers.OSFileHandler',
                'filename': log_path + 'debug.log',
                'formatter': 'request',
                'filters': ['request_id'],
                'level': 'DEBUG',
            },
            'warning': {
                'class': 'ylog.handlers.OSFileHandler',
                'filename': log_path + 'warning.log',
                'formatter': 'request',
                'filters': ['request_id'],
                'level': 'WARNING',
            },
        },
        'root': {
            'handlers': ['debug', 'warning'],
            'level': 'DEBUG',
        },
        'loggers': {
            # Логгеры для сторонних библиотек
            'requests': {
                'handlers': ['warning'],
                'level': 'WARNING',
                'propagate': False,
            },
            'kazoo.client': {
                'handlers': ['warning'],
                'level': 'WARNING',
                'propagate': False,
            },
            'urllib3.connectionpool': {
                'handlers': ['null'],
                'level': 'DEBUG',
                'propagate': False,
            },
            # Логгер Django
            'django.request': {
                'handlers': ['debug', 'warning', 'exception'],
                'level': 'WARNING',
                'propagate': False,
            },
            # Логгеры для passport-core
            'passport.useragent': {
                'handlers': ['debug'],
                'level': 'WARNING',
                'propagate': False,
            },
            'passport.graphite_logger': {
                'handlers': ['passport_graphite'],
                'level': 'DEBUG',
                'propagate': False,
            },
            'passport.backend.core.ydb_client.connection': {
                'handlers': ['debug'],
                'level': 'WARNING',
                'propagate': False,
            },
            'passport.backend.core.ydb_client.resolver.DiscoveryEndpointsResolver': {
                'handlers': ['debug'],
                'level': 'WARNING',
                'propagate': False,
            },
            # Логгеры OAuth
            'exception': {
                'handlers': ['debug', 'exception'],
                'level': 'DEBUG',
                'propagate': False,
            },
            'db.eav': {
                'handlers': ['debug', 'warning'],
                'level': 'WARNING',
                'propagate': False,
            },
            'statbox': {
                'handlers': ['statbox'],
                'level': 'DEBUG',
                'propagate': False,
            },
            'historydb.event': {
                'handlers': ['historydb_event'],
                'level': 'DEBUG',
                'propagate': False,
            },
            'graphite': {
                'handlers': ['graphite'],
                'level': 'DEBUG',
                'propagate': False,
            },
            'console': {
                'handlers': ['console'],
                'level': 'DEBUG',
                'propagate': False,
            },
        },
    }
    logging_config['formatters'].update(extra_formatters or {})
    logging_config['handlers'].update(extra_handlers or {})
    logging_config['loggers'].update(extra_loggers or {})

    if DEBUG:
        for logger, config in logging_config['loggers'].items():
            config['level'] = 'DEBUG'

    return logging_config
