# -*- coding: utf-8 -*-
import os

from django.conf import settings
from passport.backend.oauth.core.common.error_logs import (
    log_error,
    log_warning,
)
from passport.backend.oauth.core.common.utils import (
    now,
    to_base64_url,
)
from passport.backend.oauth.core.db.eav import (
    CREATE,
    EavAttr,
    EavModel,
    EntityNotFoundError,
    Index,
    UPDATE,
)
from passport.backend.oauth.core.db.mixins.deletion_mixin import DeletionMixin
from passport.backend.oauth.tvm_api.tvm_api.db.schemas import (
    tvm_client_attributes_table,
    tvm_client_by_abc_request_id_table,
    tvm_client_by_creator_uid_table,
)
from passport.backend.oauth.tvm_api.tvm_api.db.tvm_secret_key import TVMSecretKey
from passport.backend.oauth.tvm_api.tvm_api.db.vault_client import get_vault_client


class TVMClient(EavModel, DeletionMixin):
    """Приложение для TVM (Ticket Vending Machine)"""
    _entity_name = 'tvm_client'
    _table = tvm_client_attributes_table
    _indexes = {
        'creator_uid': Index(tvm_client_by_creator_uid_table),
        'abc_request_id': Index(
            tvm_client_by_abc_request_id_table,
            nullable_fields=['abc_request_id'],
        ),
    }

    creator_uid = EavAttr()
    name = EavAttr()
    client_secret = EavAttr()
    old_client_secret = EavAttr()
    secret_key_ids = EavAttr()
    created = EavAttr()
    modified = EavAttr()
    abc_request_id = EavAttr()
    abc_service_id = EavAttr()
    vault_secret_uuid = EavAttr()
    vault_version_uuid = EavAttr()

    @classmethod
    def create(cls, creator_uid, name):
        time_now = now()
        obj = cls(
            creator_uid=creator_uid,
            name=name,
            secret_key_ids=[],
            created=time_now,
            modified=time_now,
        )
        obj.generate_client_secret()
        return obj

    def generate_client_secret(self):
        self.old_client_secret = self.client_secret
        self.client_secret = to_base64_url(os.urandom(settings.TVM_CLIENT_SECRET_SIZE)).decode()

    def is_secret_saved_to_vault(self):
        return bool(self.vault_secret_uuid and self.vault_version_uuid)

    def can_save_secret_to_vault(self):
        return bool(self.abc_service_id)

    def get_vault_roles(self):
        return [
            dict(role='OWNER', abc_id=self.abc_service_id, abc_role_id=abc_role_id)
            for abc_role_id in settings.VAULT_SECRET_TVM_ROLES
        ]

    def try_save_client_secret_to_vault(self):
        if self.can_save_secret_to_vault():
            try:
                vault_client = get_vault_client()
                self.vault_secret_uuid, self.vault_version_uuid = vault_client.try_create_secret(
                    name='tvm.secret.{}'.format(self.id),
                    comment='TVM-secret, ID {}'.format(self.id),
                    value=dict(
                        client_secret=self.client_secret,
                    ),
                    secret_uuid=self.vault_secret_uuid,
                    version_uuid=self.vault_version_uuid,
                    roles=self.get_vault_roles(),
                    tags=settings.VAULT_SECRET_TAGS,
                )
            except:
                log_error(
                    None,
                    source='tvm_client.try_save_client_secret_to_vault',
                )

    def try_update_secret_roles(self):
        errors = list()

        if self.is_secret_saved_to_vault():
            roles = self.get_vault_roles()
            vault_client = get_vault_client()
            for role in roles:
                vault_error = vault_client.try_update_secret_role(
                    self.vault_secret_uuid,
                    role=role,
                )
                if vault_error:
                    errors.append(vault_error)

        return not errors, errors

    def restore_old_client_secret(self):
        if not self.old_client_secret:
            raise ValueError('No old secret found')
        self.old_client_secret, self.client_secret = self.client_secret, self.old_client_secret

    def can_be_edited(self, uid):
        return uid == self.creator_uid

    @property
    def secret_keys(self):
        result = {}
        for secret_key_id in self.secret_key_ids:
            try:
                result[secret_key_id] = TVMSecretKey.by_id(secret_key_id)
            except EntityNotFoundError:
                log_warning()
        return result

    def add_new_secret_key(self):
        with CREATE(TVMSecretKey.create()) as key:
            pass
        self.secret_key_ids.append(key.id)

    def delete_secret_key(self, secret_key_id):
        current_keys = self.secret_keys
        key = current_keys.get(secret_key_id)

        self.secret_key_ids.remove(secret_key_id)
        current_keys.pop(secret_key_id, None)

        if key is not None:
            with UPDATE(key):
                key.deleted = now()


def list_tvm_clients_by_creator(uid):
    return TVMClient.by_index('creator_uid', creator_uid=uid)
