# -*- coding: utf-8 -*-

from collections import OrderedDict
import logging

from django.conf import settings
from library.python.vault_client import VaultClient as OriginalVaultClient
from library.python.vault_client.auth import RSAPrivateKeyAuth
from library.python.vault_client.errors import ClientError as VaultClientError
from passport.backend.core.lazy_loader import (
    lazy_loadable,
    LazyLoader,
)
from passport.backend.core.logging_utils.loggers.tskv import TskvLogEntry
from passport.backend.oauth.core.common.error_logs import (
    log_error,
    log_warning,
)
import passport.backend.oauth.core.common.utils.datetime_helpers as datetime_helpers
from passport.backend.utils.common import remove_none_values


class SortedTskvLogEntry(TskvLogEntry):
    def __init__(self, **params):
        super(SortedTskvLogEntry, self).__init__(**params)
        self.params = OrderedDict(sorted(self.params.items(), key=lambda t: t[0]))


class NativeVaultClient(OriginalVaultClient):
    def __init__(self, *args, **kwargs):
        super(NativeVaultClient, self).__init__(*args, **kwargs)

    def _call_native_client(self, *args, **kwargs):
        response = super(NativeVaultClient, self)._call_native_client(*args, **kwargs)
        self._log_response(response)
        return response

    def _log_response(self, response):
        if response is not None:
            status = None
            error = None
            try:
                parsed_response = response.json()
                status = parsed_response.get('status')
                error = parsed_response.get('message')
            except ValueError:
                pass

            logging.getLogger('tvm_api.vault_client').info(
                str(SortedTskvLogEntry(**remove_none_values(dict(
                    url=response.url,
                    http_status=response.status_code,
                    vault_request_id=self._last_request_id,
                    elapsed=response.elapsed.total_seconds() if response.elapsed is not None else 0,
                    timestamp=str(datetime_helpers.now()),
                    vault_status=status,
                    vault_error=error,
                )))),
            )


@lazy_loadable('VaultClient')
class VaultClient(object):
    SAFE_VAULT_ERRORS_CODES = [
        'abc_service_not_found',
        'abc_role_not_found',
    ]

    def __init__(self, native_vault_client=None):
        self.native_vault_client = native_vault_client
        if self.native_vault_client is None:
            self.native_vault_client = NativeVaultClient(
                host=settings.VAULT_API_URL,
                rsa_login=settings.VAULT_USER_LOGIN,
                rsa_auth=RSAPrivateKeyAuth(settings.VAULT_USER_PRIVATE_KEY),
                check_status=False,
            )

    def try_create_secret(self, name, value, comment=None, roles=None, tags=None, secret_uuid=None, version_uuid=None):
        try:
            if not secret_uuid:
                response = self.native_vault_client.create_complete_secret(
                    name,
                    comment=comment,
                    secret_version=dict(
                        value=value,
                    ),
                    roles=roles,
                    tags=tags,
                )
                return response['uuid'], response['secret_version']

            if secret_uuid and not version_uuid:
                version_uuid = self.native_vault_client.create_secret_version(
                    secret_uuid,
                    value=value,
                )
                return secret_uuid, version_uuid
        except Exception as e:
            if isinstance(e, VaultClientError) and e.kwargs.get('code') in self.SAFE_VAULT_ERRORS_CODES:
                log_warning(
                    e.kwargs.get('message') or e,
                )
            else:
                log_error(
                    None,
                    source='vault_client.create_secret',
                    vault_request_id=self.native_vault_client._last_request_id,
                )

        return secret_uuid, version_uuid

    def try_update_secret_role(self, secret_uuid, role):
        error = None
        try:
            self.native_vault_client.add_user_role_to_secret(
                secret_uuid,
                **role
            )
        except VaultClientError as e:
            error = e.kwargs.get('message') or e
            log_warning(error)
        except:
            log_error(
                None,
                source='vault_client.add_user_role_to_secret',
                vault_request_id=self.native_vault_client._last_request_id,
            )

        return error


def get_vault_client():
    return LazyLoader.get_instance('VaultClient')
