# -*- coding: utf-8 -*-
import logging

from django.core.management import BaseCommand
from passport.backend.oauth.core.db.eav import (
    BaseDBError,
    EntityNotFoundError,
    UPDATE,
)
from passport.backend.oauth.tvm_api.tvm_api.db.tvm_client import TVMClient


log = logging.getLogger('console')


class Command(BaseCommand):
    help = 'Sets or changes the ABC service TVM client belongs to'

    def add_arguments(self, parser):
        parser.add_argument(
            '--client_id',
            action='store',
            dest='client_id',
            type=int,
            help='ID of client to edit',
        )
        parser.add_argument(
            '--abc_service_id',
            action='store',
            dest='abc_service_id',
            type=int,
            help='ID of ABC service',
        )
        parser.add_argument(
            '--allow_overwrite',
            action='store_true',
            dest='allow_overwrite',
            help='Allow editing TVM client already belonging to ABC service',
        )
        parser.add_argument(
            '--allow_deleted',
            action='store_true',
            dest='allow_deleted',
            help='Allow editing deleted TVM client',
        )

    def handle(self, *args, **options):
        try:
            if not options.get('client_id'):
                log.error('Error: `client_id` arg is required')
                exit(1)

            try:
                client = TVMClient.by_id(options['client_id'], allow_deleted=options['allow_deleted'])
            except EntityNotFoundError:
                error_message = 'Client %s not found.' % options['client_id']
                if not options['allow_deleted']:
                    error_message += ' Add --allow_deleted if you are sure.'
                log.error(error_message)
                exit(1)

            old_abc_service_id = client.abc_service_id
            new_abc_service_id = options['abc_service_id'] or None

            if old_abc_service_id is not None:
                if not options['allow_overwrite']:
                    log.error(
                        'Client %s already belongs to ABC service %s. Add --allow_overwrite to overwrite.',
                        options['client_id'],
                        old_abc_service_id
                    )
                    exit(1)
                else:
                    log.info('Old abc_service_id: %s' % old_abc_service_id)

            with UPDATE(client):
                client.abc_service_id = new_abc_service_id

                updated, errors = client.try_update_secret_roles()
                if not updated:
                    log.debug(
                        'Failed to update roles (secret_uuid: %s, ABC: %s) for client %s. Errors: %s',
                        client.vault_secret_uuid,
                        client.abc_service_id,
                        client.id,
                        ', '.join(errors),
                    )

            log.info('Done')

        except BaseDBError as e:
            log.warning('DB error: %s', e)
            exit(1)
