# -*- coding: utf-8 -*-
import logging

from django.core.management import BaseCommand
from passport.backend.oauth.core.common.utils import now
from passport.backend.oauth.core.db.eav import (
    DELETE,
    EntityNotFoundError,
    UPDATE,
)
from passport.backend.oauth.tvm_api.tvm_api.db.tvm_client import TVMClient


log = logging.getLogger('console')


ACTION_DELETE = 'delete'
ACTION_PURGE = 'purge'
ACTION_RESTORE = 'restore'

ALL_ACTIONS = (ACTION_DELETE, ACTION_PURGE, ACTION_RESTORE)


class Command(BaseCommand):
    help = 'Marks TVM client as deleted'

    def add_arguments(self, parser):
        parser.add_argument(
            '--action',
            action='store',
            dest='action',
            choices=ALL_ACTIONS,
            help='Desired action. Valid choices are %s' % (ALL_ACTIONS, ),
        )
        parser.add_argument(
            '--client_id',
            action='store',
            dest='client_id',
            type=int,
            help='ID of client to delete',
        )

    def _restore(self, client):
        if not client.deleted:
            log.info('Client is not deleted - no action needed')
            exit(0)

        with UPDATE(client):
            client.deleted = None

    def _delete(self, client):
        if client.deleted:
            log.info('Client is already marked as deleted - no action needed')
            exit(0)

        with UPDATE(client):
            client.deleted = now()

    def _purge(self, client):
        with DELETE(client):
            pass

    def handle(self, *args, **options):
        if not options.get('action'):
            log.error('Error: `action` arg is required')
            exit(1)
        action = options['action']

        if not options.get('client_id'):
            log.error('Error: `client_id` arg is required')
            exit(1)

        try:
            client = TVMClient.by_id(options['client_id'], allow_deleted=True)
        except EntityNotFoundError:
            log.error('Client %s not found', options['client_id'])
            exit(1)

        if action == ACTION_RESTORE:
            self._restore(client)
        elif action == ACTION_DELETE:
            self._delete(client)
        elif action == ACTION_PURGE:
            self._purge(client)
        else:
            log.error('Unknown action: "%s"', action)
            exit(1)

        log.info('Done')
