# -*- coding: utf-8 -*-
from datetime import (
    datetime,
    timedelta,
)
import logging
from time import sleep
import traceback

from django.conf import settings
from django.core.management.base import BaseCommand
from passport.backend.oauth.core.common.zookeeper import run_exclusively
from passport.backend.oauth.core.db.eav import (
    BaseDBError,
    EntityNotFoundError,
    UPDATE,
)
from passport.backend.oauth.tvm_api.tvm_api.db.tvm_client import TVMClient


log = logging.getLogger('management.tvm_secret_key_rotator')

MAX_KEY_COUNT = 14
MIN_KEY_TIMEDELTA = timedelta(hours=23)
# Максимальная разница времени (в секундах) между машинами кластера в нормальной ситуации
MAX_TIME_DISCREPANCY = 10


def try_update_clients():
    updated_client_count = 0
    for client_id in settings.TVM_CLIENT_IDS_FOR_KEY_ROTATION:
        try:
            client = TVMClient.by_id(client_id)

            secret_keys = sorted(client.secret_keys.values(), key=lambda k: k.id, reverse=True)
            newest_key = secret_keys[0]
            if datetime.now() - newest_key.created < MIN_KEY_TIMEDELTA:
                log.error(
                    'Not trying to update keys for client %s: previous successful attempt was on %s',
                    client.id,
                    newest_key.created,
                )
                continue

            # Выполняем в два этапа, чтобы гарантированно не оставить приложение без ключей
            with UPDATE(client):
                client.add_new_secret_key()

            current_key_ids = sorted(client.secret_key_ids, reverse=True)
            created_key_id = current_key_ids[0]
            key_ids_to_delete = current_key_ids[MAX_KEY_COUNT:]

            with UPDATE(client):
                for key_id in key_ids_to_delete:
                    client.delete_secret_key(secret_key_id=key_id)

            updated_client_count += 1
            log.debug(
                'Added keys [%s] and removed keys [%s] for client %s',
                created_key_id,
                ','.join(map(str, key_ids_to_delete)),
                client_id,
            )

        except EntityNotFoundError:
            log.error('TVM client with id=%s not found' % client_id)
        except BaseDBError as e:
            log.warning('DB error: %s', e)
        except Exception as e:
            log.error('Unhandled error: %s\n%s', e, traceback.format_exc())

    return updated_client_count


class Command(BaseCommand):
    help = 'Rotates secret keys for predefined set of TVM clients'

    @run_exclusively('/passport/oauth/management/rotate_tvm_secret_keys')
    def handle(self, *args, **options):
        try:
            log.info('Task started')
            count = try_update_clients()
            log.info('Task complete. %s clients updated.', count)
            if count > 0:
                # Делаем паузу, чтобы не успели исполниться скрипты на соседних машинах
                # (на случай расхождения времени между машинами)
                sleep(MAX_TIME_DISCREPANCY)
        except Exception as e:
            log.error('Unhandled error: %s', e, exc_info=True)
            exit(1)
