# -*- coding: utf-8 -*-
import logging
from time import sleep

from django.core.management.base import BaseCommand
from passport.backend.oauth.core.common.zookeeper import run_exclusively
from passport.backend.oauth.core.db.eav import (
    BaseDBError,
    UPDATE,
)
from passport.backend.oauth.tvm_api.tvm_api.db.tvm_client import TVMClient


log = logging.getLogger('management.update_tvm_secrets_roles')

# Максимальная разница времени (в секундах) между машинами кластера в нормальной ситуации
MAX_TIME_DISCREPANCY = 10


def try_update_clients(chunk_size, use_pauses=False, pause_length=1.0):
    updated_client_count = 0
    failed_client_count = 0
    try:
        for chunk in TVMClient.iterate_by_chunks(chunk_size):
            for client in chunk:
                if not client.is_secret_saved_to_vault():
                    continue

                with UPDATE(client):
                    updated, errors = client.try_update_secret_roles()

                if not updated:
                    failed_client_count += 1
                    log.debug(
                        'Failed to update roles (secret_uuid: %s, ABC: %s) for client %s. Errors: %s',
                        client.vault_secret_uuid,
                        client.abc_service_id,
                        client.id,
                        ', '.join(errors),
                    )
                    continue

                updated_client_count += 1
                log.debug(
                    'Secrets roles updated (secret_uuid: %s, ABC: %s) for client %s',
                    client.vault_secret_uuid,
                    client.abc_service_id,
                    client.id,
                )

            if use_pauses:
                sleep(pause_length)
    except BaseDBError as e:
        log.warning('DB error: %s', e)
        exit(1)

    return updated_client_count, failed_client_count


class Command(BaseCommand):
    help = 'Updates roles for secrets stored in Vault'

    def add_arguments(self, parser):
        parser.add_argument(
            '--chunk_size',
            action='store',
            dest='chunk_size',
            type=int,
            default=500,
            help='Number of objects read from DB at a time',
        )
        parser.add_argument(
            '--use_pauses',
            action='store_true',
            dest='use_pauses',
            default=False,
            help='Use pauses between processing token chunks',
        )
        parser.add_argument(
            '--pause_length',
            action='store',
            dest='pause_length',
            type=float,
            default=1.0,
            help='Pause length (in seconds)',
        )

    @run_exclusively('/passport/oauth/management/update_tvm_secrets_roles')
    def handle(self, *args, **options):
        try:
            log.info('Task started')
            updated, failed = try_update_clients(
                chunk_size=options['chunk_size'],
                use_pauses=options['use_pauses'],
                pause_length=options['pause_length'],
            )
            log.info('Task complete. %s clients updated. %s clients failed.', updated, failed)
            if updated > 0:
                # Делаем паузу, чтобы не успели исполниться скрипты на соседних машинах
                # (на случай расхождения времени между машинами)
                sleep(MAX_TIME_DISCREPANCY)
        except Exception as e:
            log.error('Unhandled error: %s', e, exc_info=True)
            exit(1)
