# -*- coding: utf-8 -*-
from django.conf import settings
from passport.backend.core.builders.abc import ABC
from passport.backend.core.builders.blackbox.blackbox import BlackboxTemporaryError
from passport.backend.oauth.core.api.base import BaseApiView
from passport.backend.oauth.core.api.errors import (
    BaseApiError,
    ClientNotFoundError,
    OAuthTokenValidationError,
)
from passport.backend.oauth.core.common.error_logs import log_error
from passport.backend.oauth.core.common.utils import (
    first_or_none,
    now,
)
from passport.backend.oauth.core.db.eav import (
    CREATE,
    DBIntegrityError,
    EntityNotFoundError,
    UPDATE,
)
from passport.backend.oauth.tvm_api.tvm_api.db.tvm_client import TVMClient
from passport.backend.oauth.tvm_api.tvm_api.tvm_abc.errors import (
    AbcTeamMemberRequiredError,
    HaveNotPutSecretToVaultError,
    OldSecretExistsError,
    OldSecretNotFoundError,
)
from passport.backend.oauth.tvm_api.tvm_api.tvm_abc.forms import (
    ChangeClientAbcServiceForm,
    ClientForm,
    CreateClientForm,
    CreateClientV1Form,
    EditClientForm,
    EditClientV1Form,
    PaginationForm,
)
from passport.backend.oauth.tvm_api.tvm_api.tvm_abc.utils import client_to_response


VIEW_CLIENTS_GRANT = 'tvm_abc_api.view_clients'
EDIT_CLIENTS_GRANT = 'tvm_abc_api.edit_clients'
SAVE_CLIENT_SECRET_TO_VAULT_GRANT = 'tvm_abc_api.save_client_secret_to_vault'
CHANGE_ABC_SERVICE_GRANT = 'tvm_abc_api.change_abc_service'


DEFAULT_ALLOWED_ROLES = (
    'product_head',  # руководитель сервиса
    'tvm_manager',  # управление TVM-приложениями
)


class BaseTvmAbcApiView(BaseApiView):
    def __init__(self):
        super(BaseTvmAbcApiView, self).__init__()
        self.user_info = None
        self._uid = None
        self.client = None

    @property
    def abc(self):
        return ABC(use_tvm=settings.ABC_USE_TVM)

    def get_client(self):
        try:
            self.client = TVMClient.by_id(self.form_values['client_id'])
        except EntityNotFoundError:
            raise ClientNotFoundError()

    def get_user(self, optional=False):
        try:
            self.user_info = self.get_user_from_user_ticket(
                required_scope=[settings.ABC_SCOPE_KEYWORD, settings.SESSIONID_SCOPE],
            )
            self._uid = self.user_info['uid']
        except BaseApiError:
            if optional:
                self._uid = None
            else:
                raise

    def check_robot_token(self):
        user_info = self.get_user_from_token(require_x_token=False)
        if user_info['uid'] != settings.ABC_ROBOT_UID:
            raise OAuthTokenValidationError('Unknown robot')
        self._uid = None

    def user_has_permissions(self, abc_service_id, uid=None, allowed_roles=None):
        uid = uid or self.uid
        if uid is None:
            return False
        allowed_roles = set(allowed_roles or DEFAULT_ALLOWED_ROLES)
        rv = self.abc.get_service_members(
            filter_args=dict(
                service=abc_service_id,
                person__uid=uid,
                role__code__in=','.join(allowed_roles),
            ),
        )
        for entry in rv:
            # Перестраховываюсь (на случай опечатки в фильтрах запроса) и ещё раз проверяю все условия
            if (
                entry['service']['id'] == abc_service_id and
                int(entry['person']['uid']) == uid and
                entry['role']['code'] in allowed_roles
            ):
                return True
        return False

    @property
    def uid(self):
        return self._uid


class GetUserFromCookieOrTokenMixin(object):
    # TODO: удалить, когда закопают старые ручки, работающие по куке/токену
    def get_user(self, optional=False):
        try:
            if self.request.env.authorization:
                self.user_info = self.get_user_from_token(
                    require_x_token=False,
                    required_scope=settings.ABC_SCOPE_KEYWORD,
                )
            else:
                self.user_info = self.get_user_from_session(uid=self.form_values['uid'])

            self._uid = self.user_info['uid']
        except (BaseApiError, BlackboxTemporaryError):
            if optional:
                self._uid = None
            else:
                raise


class ListClientsView(BaseTvmAbcApiView):
    required_grants = [VIEW_CLIENTS_GRANT]
    allowed_methods = ['GET']
    base_form = PaginationForm

    def process_request(self, request):
        clients = TVMClient.by_index(
            'creator_uid',  # подойдёт любой индекс, в котором есть все приложения
            limit=self.form_values['page_size'],
            offset=(self.form_values['page'] - 1) * self.form_values['page_size'],
            allow_deleted=True,  # TODO: выкинуть, когда появится иной способ узнать о существовании следующей страницы
        )
        has_next_page = len(clients) == self.form_values['page_size']
        self.response_values.update(
            content=[
                client_to_response(client)
                for client in clients
                if not client.is_deleted and client.abc_service_id
            ],
            meta={
                'next_page': self.form_values['page'] + 1 if has_next_page else None,
            },
        )


class ClientInfoView(BaseTvmAbcApiView):
    required_grants = [VIEW_CLIENTS_GRANT]
    allowed_methods = ['GET']
    base_form = ClientForm

    def process_request(self, request):
        self.get_user(optional=True)
        self.get_client()
        show_full_info = self.user_has_permissions(abc_service_id=self.client.abc_service_id)
        self.response_values.update(
            content=[
                client_to_response(self.client, full_info=show_full_info),
            ],
            is_viewed_by_owner=show_full_info,
        )


class ClientInfoV1View(GetUserFromCookieOrTokenMixin, ClientInfoView):
    pass


class CreateClientView(BaseTvmAbcApiView):
    required_grants = [EDIT_CLIENTS_GRANT]
    allowed_methods = ['POST']
    base_form = CreateClientForm

    def process_request(self, request):
        self.get_user()
        abc_service_id = self.form_values['abc_service_id']
        abc_request_id = self.form_values['abc_request_id']

        if not self.user_has_permissions(abc_service_id=abc_service_id):
            raise AbcTeamMemberRequiredError()

        try:
            with CREATE(TVMClient.create(
                creator_uid=self.uid,
                name=self.form_values['name'],
            )) as client:
                client.abc_service_id = abc_service_id
                client.abc_request_id = abc_request_id

            with UPDATE(client):
                client.try_save_client_secret_to_vault()
        except DBIntegrityError:
            client = first_or_none(TVMClient.by_index('abc_request_id', abc_request_id=abc_request_id))
            if client is None:
                # Гонка: данные прошлого create-запроса записались в мастер, но ещё не доехали до слейва.
                # Ничего не поделаешь, отдадим временную ошибку.
                raise

        self.response_values.update(
            client_id=client.id,
        )


class CreateClientV1View(CreateClientView):
    base_form = CreateClientV1Form

    def get_user(self, **kwargs):
        self.check_robot_token()
        self._uid = self.form_values['uid']


class BaseClientModificationView(BaseTvmAbcApiView):
    required_grants = [EDIT_CLIENTS_GRANT]
    allowed_methods = ['POST']

    def modify_client(self, client):
        raise NotImplementedError()  # pragma: no cover

    def process_request(self, request):
        self.get_user()
        self.get_client()

        if not self.user_has_permissions(abc_service_id=self.client.abc_service_id):
            raise AbcTeamMemberRequiredError()

        self.modify_client(self.client)


class EditClientView(BaseClientModificationView):
    base_form = EditClientForm

    def modify_client(self, client):
        with UPDATE(client):
            client.name = self.form_values['name']
            client.modified = now()


class EditClientV1View(EditClientView):
    base_form = EditClientV1Form

    def get_user(self, **kwargs):
        self.check_robot_token()
        self._uid = self.form_values['uid']


class DeleteClientView(BaseClientModificationView):
    base_form = ClientForm

    def modify_client(self, client):
        with UPDATE(client):
            client.deleted = now()


class DeleteClientV1View(GetUserFromCookieOrTokenMixin, DeleteClientView):
    pass


class BaseClientSecretsModificationView(BaseClientModificationView):
    base_form = ClientForm

    def __init__(self, *args, **kwargs):
        super(BaseClientModificationView, self).__init__(*args, **kwargs)
        self.disable_check_secret_saved_to_vault = settings.DISABLE_CHECK_SECRET_SAVED_TO_VAULT

    def fill_response_with_secrets(self):
        self.response_values.setdefault('content', {}).setdefault('attributes', {}).update(
            client_secret=self.client.client_secret,
            old_client_secret=self.client.old_client_secret or None,
        )
        if self.client.is_secret_saved_to_vault():
            self.response_values.setdefault('content', {}).setdefault('attributes', {}).update(
                secret_uuid=self.client.vault_secret_uuid,
                version_uuid=self.client.vault_version_uuid,
                vault_link=settings.VAULT_SECRET_LINK_TEMPLATE.format(
                    secret_uuid=self.client.vault_secret_uuid,
                    version_uuid=self.client.vault_version_uuid,
                ),
            )

    def check_secret_saved_to_vault(self, client):
        if not self.disable_check_secret_saved_to_vault and not client.is_secret_saved_to_vault():
            raise HaveNotPutSecretToVaultError()

    def process_request(self, request):
        super(BaseClientSecretsModificationView, self).process_request(request)
        self.fill_response_with_secrets()


class SaveSecretToVaultView(BaseClientSecretsModificationView):
    required_grants = [SAVE_CLIENT_SECRET_TO_VAULT_GRANT]

    def fill_response_with_secrets(self):
        if self.client.is_secret_saved_to_vault():
            self.response_values.setdefault('content', {}).setdefault('attributes', {}).update(
                secret_uuid=self.client.vault_secret_uuid,
                version_uuid=self.client.vault_version_uuid,
                vault_link=settings.VAULT_SECRET_LINK_TEMPLATE.format(
                    secret_uuid=self.client.vault_secret_uuid,
                    version_uuid=self.client.vault_version_uuid,
                ),
            )

    def modify_client(self, client):
        with UPDATE(client):
            client.try_save_client_secret_to_vault()
            client.modified = now()

        if not client.is_secret_saved_to_vault():
            raise HaveNotPutSecretToVaultError()


class SaveSecretToVaultV1View(GetUserFromCookieOrTokenMixin, SaveSecretToVaultView):
    pass


class RecreateClientSecretView(BaseClientSecretsModificationView):
    def modify_client(self, client):
        if client.old_client_secret:
            raise OldSecretExistsError()

        self.check_secret_saved_to_vault(client)

        with UPDATE(client):
            client.generate_client_secret()
            del client.vault_version_uuid
            client.try_save_client_secret_to_vault()
            client.modified = now()


class RecreateClientSecretV1View(GetUserFromCookieOrTokenMixin, RecreateClientSecretView):
    pass


class RestoreClientSecretView(BaseClientSecretsModificationView):
    def modify_client(self, client):
        if not client.old_client_secret:
            raise OldSecretNotFoundError()

        self.check_secret_saved_to_vault(client)

        with UPDATE(client):
            client.restore_old_client_secret()
            del client.vault_version_uuid
            client.try_save_client_secret_to_vault()
            client.modified = now()


class RestoreClientSecretV1View(GetUserFromCookieOrTokenMixin, RestoreClientSecretView):
    pass


class DeleteOldClientSecretView(BaseClientSecretsModificationView):
    def modify_client(self, client):
        self.check_secret_saved_to_vault(client)

        with UPDATE(client):
            del client.old_client_secret
            client.modified = now()


class DeleteOldClientSecretV1View(GetUserFromCookieOrTokenMixin, DeleteOldClientSecretView):
    pass


class ChangeClientAbcServiceView(BaseClientModificationView):
    base_form = ChangeClientAbcServiceForm
    required_grants = [CHANGE_ABC_SERVICE_GRANT]
    allowed_methods = ['POST']

    def process_request(self, request):
        self.get_user()
        self.get_client()

        initiator_uid = self.form_values['initiator_uid']
        new_abc_service_id = self.form_values['abc_service_id']

        # Проверим права и подтверждающего передачу, и инициатора передачи
        if not (
            self.user_has_permissions(uid=self.uid, abc_service_id=new_abc_service_id) and
            self.user_has_permissions(uid=initiator_uid, abc_service_id=self.client.abc_service_id)
        ):
            raise AbcTeamMemberRequiredError()

        with UPDATE(self.client) as client:
            client.abc_service_id = new_abc_service_id

            updated, errors = client.try_update_secret_roles()
            if not updated:
                log_error(
                    None,
                    source='tvm_client.try_update_secret_roles',
                )
