from passport.backend.library.nginx_config_generator import BaseConfig


MDA1_TLDS = (
    'ru',
    'ua',
    'by',
    'kz',
)

PASSPORT_EN_TLDS = (
    'az',
    'co.il',
    'com',
    'com.am',
    'com.ge',
    'com.tr',
    'ee',
    'eu',
    'fi',
    'fr',
    'lt',
    'lv',
    'pl',
)

PASSPORT_RU_TLDS = (
    'by',
    'kg',
    'kz',
    'md',
    'ru',
    'tj',
    'tm',
    'ua',
    'uz',
)

PASSPORT_TLDS = sorted(PASSPORT_RU_TLDS + PASSPORT_EN_TLDS)


class BasePassportConfig(BaseConfig):
    _ssl_session_cache_sizes = {
        'intranet': {
            'development': '64m',
            'testing': '10m',
            'rc': '10m',
            'production': '256m',
        },
        'localhost': {
            'development': '64m',
            'testing': '10m',
            'rc': '10m',
            'production': '256m',
        },
        'stress': {
            'stress': '10m',
        },
    }
    _passport_host_prefixes = {
        'intranet': {
            'development': None,
            'testing': 'passport-test.yandex-team',
            'rc': 'passport-rc.yandex-team',
            'production': 'passport.yandex-team',
        },
        'localhost': {
            'development': '0.passportdev.yandex',
            'testing': 'passport-test.yandex',
            'rc': 'passport-rc.yandex',
            'production': 'passport.yandex',
        },
        'stress': {
            'stress': 'passport-load.yandex',
        },
    }

    def _get_ssl_session_cache_size(self):
        return self._ssl_session_cache_sizes.get(self.yenv_name, {}).get(self.yenv_type)

    def _get_passport_host_prefix(self):
        return self._passport_host_prefixes.get(self.yenv_name, {}).get(self.yenv_type)

    def get_custom_settings(self):
        settings = super(BasePassportConfig, self).get_custom_settings()
        settings.update(
            ssl_session_cache_size=self._get_ssl_session_cache_size(),
            passport_host_prefix=self._get_passport_host_prefix(),
        )
        return settings

    @property
    def is_plus_instance(self):
        return self.fqdn.startswith('passport-api-')

    def _get_slb_ips(self):
        if self.is_plus_instance:
            # Это машинки для долгих запросов. Конфиги фронта, sso и т.п. нужны для мониторингов,
            # но из балансера запросы не приходят, поэтому slb_ips не нужны.
            return ()
        return super(BasePassportConfig, self)._get_slb_ips()


class BasePassportInternalConfig(BasePassportConfig):
    """Конфигурация для внутренних потребителей (апи)"""

    _slb_ips = {
        'intranet': {
            'development': (),
            'testing': (
                '141.8.146.78',
                '2a02:6b8:0:3400::5:78',
            ),
            'rc': (
                '2a02:6b8::3400:0:e:0:17',
            ),
            'production': (
                '5.255.240.43',
                '2a02:6b8:0:3400::1:43',
            ),
        },
        'localhost': {
            'development': (),
            'testing': (
                '5.255.240.66',
                '2a02:6b8:0:3400::66',
            ),
            'rc': (
                '213.180.205.78',
                '2a02:6b8:0:3400::78',
            ),
            'production': (
                '213.180.205.19',
                '2a02:6b8:0:3400::3:19',
            ),
        },
        'stress': {
            'stress': (),
        },
    }

    @property
    def is_applicable(self):
        # В отличие от родителя, смотрим не на сертификат (его может не быть), а на имена хостов
        if self._get_server_names() is None:
            return False
        return True


class BasePassportExternalConfig(BasePassportConfig):
    """Конфигурация для внешних потребителей (фронтенд)"""
    _slb_ips = {
        'intranet': {
            'development': (),
            'testing': (
                '213.180.193.10',
                '2a02:6b8::ca',
            ),
            'rc': (
                '2a02:6b8::3400:0:e:0:16',
            ),
            'production': (
                '213.180.193.24',
                '2a02:6b8::1:24',
            ),
        },
        'localhost': {
            'development': (),
            'testing': (
                '5.255.240.16',
                '2a02:6b8:0:3400::3:16',
            ),
            'rc': (
                '213.180.204.114',
                '2a02:6b8::114',
            ),
            'production': (
                '213.180.204.24',
                '2a02:6b8::24',
            ),
        },
        'stress': {
            'stress': (),
        },
    }


class BaseSsoConfig(BasePassportExternalConfig):
    @property
    def template_name(self):
        return 'base_sso'


class BaseGuardConfig(BasePassportExternalConfig):
    @property
    def template_name(self):
        return 'base_guard'

    _tlds = {
        'intranet': {
            'development': None,
            'testing': ['ru'],
            'rc': ['ru'],
            'production': ['ru'],
        },
        'localhost': {
            'development': ['ru'],
            'testing': PASSPORT_TLDS,
            'rc': PASSPORT_TLDS,
            'production': PASSPORT_TLDS,
        },
        'stress': {
            'stress': ['ru'],
        },
    }
