# -*- coding: utf-8 -*-
import logging

from passport.backend.core.conf import settings
from passport.backend.core.lazy_loader import (
    lazy_loadable,
    LazyLoader,
)
from passport.backend.perimeter.auth_api.common.base_checker import CheckStatus
from passport.backend.perimeter.auth_api.db.checkers import (
    get_long_checker,
    get_mdm_checker,
    get_motp_checker,
    get_totp_checker,
)
from passport.backend.perimeter.auth_api.ldap.checker import get_ldap_checker
from passport.backend.perimeter.auth_api.oauth.checker import get_oauth_checker
from passport.backend.perimeter.auth_api.redis.checker import (
    get_redis_checker,
    make_redis_key,
)
from passport.backend.perimeter.auth_api.redis.storage import get_redis_storage
from passport.backend.perimeter.auth_api.utils.hash import hash_string


log = logging.getLogger('auth_logic')


@lazy_loadable()
class AuthLogic(object):
    """
    Класс, инкапсулирующий высокоуровневую логику проверки пароля
    """
    def __init__(self):
        self._checkers = {
            'redis': get_redis_checker(),
            'ldap': get_ldap_checker(),
            'totp': get_totp_checker(),
            'motp': get_motp_checker(),
            'long': get_long_checker(),
            'mdm': get_mdm_checker(),
            'oauth': get_oauth_checker(),
        }
        self._redis_cache = get_redis_storage()

    def cache_successful_auth(self, successful_check_result, login, password, ip, auth_type):
        """
        Кэшируем успешную авторизацию, чтобы частые повторные авторизации не нагружали БД и AD
        """
        second_steps = ','.join(successful_check_result.require_second_steps)
        log.info(
            'Caching successful auth for user "%s" (auth_type=%s, ip=%s, second_steps=%s) in Redis',
            login,
            auth_type,
            ip,
            second_steps or None,
        )
        redis_key = make_redis_key(login, auth_type, ip)
        self._redis_cache.set(
            redis_key,
            {
                'password_hash': hash_string(password),
                'second_steps': second_steps,
                'require_password_change': '1' if successful_check_result.require_password_change else '0',
            },
        )

    @staticmethod
    def make_description(check_result_values):
        description = '; '.join(
            check_result.description
            for check_result in check_result_values
            if check_result.description
        )
        return description or 'No applicable auth methods'

    def try_checker(self, checker_name, login, password, ip, auth_type,  **kwargs):
        checker = self._checkers[checker_name]
        log.debug('Trying to use %s checker', checker.alias)
        if checker.is_enabled:
            check_result = checker.check(
                login=login,
                password=password,
                ip=ip,
                auth_type=auth_type,
                **kwargs,
            )
        else:
            log.info('Skipping %s checker: it is disabled', checker.alias)
            check_result = CheckStatus(is_ok=False, description='%s disabled' % checker.alias)
        log.debug('Done with %s checker', checker.alias)
        return check_result

    def iterate_applicable_checkers(self, login, password, ip, is_ip_internal, is_ip_robot, is_robot, auth_type):
        """
        Генератор, реализующий логику проверки пароля и перебирающий все доступные методы.
        В ответе генератора - CheckStatus'ы.
        """
        def try_auth_via(checker_name, **kwargs):
            return self.try_checker(
                checker_name=checker_name,
                login=login,
                password=password,
                ip=ip,
                auth_type=auth_type,
                **kwargs,
            )

        # Самое важная часть этого класса: логика проверки пароля в зависимости от параметров запроса

        yield try_auth_via('redis')

        if auth_type in {'web', 'oauthcreate'}:
            yield try_auth_via('motp')

        if auth_type in {'imap', 'smtp', 'calendar', 'xmpp', 'xmpp_team'}:
            yield try_auth_via('mdm')

        if auth_type in {'imap', 'smtp'}:
            allowed_scopes = ['mail:imap_full']
            if auth_type == 'smtp':
                allowed_scopes.append('mail:smtp')
            yield try_auth_via('oauth', allowed_scopes=allowed_scopes)

        if (
            auth_type in {'calendar', 'xmpp', 'xmpp_team'} or
            (auth_type in {'imap', 'smtp'} and is_ip_internal)
        ):
            yield try_auth_via('long')

        if auth_type in {'web', 'oauthcreate'} or is_ip_internal:
            ldap_status = try_auth_via(
                'ldap',
                is_robot=is_robot,
                forbid_robots=not is_ip_internal and not is_ip_robot,
            )
            if (
                # пароль неверный - дальше проверять нечего
                not ldap_status.is_ok or
                # из внутренней сети достаточно просто пароля
                is_ip_internal or
                # роботам из роботной сети достаточно просто пароля
                (is_ip_robot and ldap_status.extra_data.get('is_account_robot'))
            ):
                yield ldap_status
            else:
                # иначе требуется второй фактор
                totp_status = self._checkers['totp'].check_if_totp_is_on(login)
                if totp_status.got_errors:
                    yield totp_status
                elif totp_status.is_ok:
                    ldap_status.require_second_steps = {'totp'}
                    yield ldap_status
                else:
                    ldap_status.require_second_steps = {'email_code'}
                    yield ldap_status

    def perform_auth(self, login, password, ip, is_ip_internal, is_ip_robot, is_robot, auth_type):
        """
        Процесс проверки пароля. В случае правильного пароля добавляет запись в кэш.
        """
        check_results = []
        for check_result in self.iterate_applicable_checkers(
            login=login,
            password=password,
            ip=ip,
            is_ip_internal=is_ip_internal,
            is_ip_robot=is_ip_robot,
            is_robot=is_robot,
            auth_type=auth_type,
        ):
            check_results.append(check_result)
            if check_result.is_ok:
                break

        if check_results and check_results[-1].is_ok:
            successful_check_result = check_results[-1]
            if settings.REDIS_ENABLED:
                self.cache_successful_auth(
                    successful_check_result=successful_check_result,
                    login=login,
                    password=password,
                    ip=ip,
                    auth_type=auth_type,
                )

            successful_check_result.description = self.make_description(check_results)
            return successful_check_result

        else:
            error_results = [r for r in check_results if r.got_errors]
            # Если случилась серверная ошибка - отдадим в ответ только данные об ошибках. Иначе отдадим полный отчёт.
            description = self.make_description(error_results or check_results)
            return CheckStatus(is_ok=False, description=description, got_errors=bool(error_results))


def get_auth_logic():
    return LazyLoader.get_instance('AuthLogic')
