# -*- coding: utf-8 -*-
import logging

from passport.backend.core.conf import settings
from passport.backend.perimeter.auth_api.common.base_checker import (
    BaseChecker,
    CheckStatus,
)
from passport.backend.perimeter.auth_api.common.exceptions import DBError
from passport.backend.perimeter.auth_api.db.connection import get_db_connection
from passport.backend.utils.string import smart_text
from sqlalchemy import select


log = logging.getLogger('db.checkers')


class BaseDbChecker(BaseChecker):
    db_table = None

    @property
    def is_enabled(self):
        return settings.DB_ENABLED

    def get_all_accounts(self, login):
        query = select([self.db_table]).where(
            self.db_table.c.username == smart_text(login),
        )
        self.accounts = get_db_connection().execute(query, many=True)


class BaseDbPasswordChecker(BaseDbChecker):
    def check_password(self, account, password):
        return account[self.db_table.c.password] == password

    def filter_by_password(self, password):
        self.accounts = [
            account
            for account in self.accounts
            if self.check_password(account, password)
        ]

    def check(self, login, password, **kwargs):
        try:
            self.get_all_accounts(login)
        except DBError as e:
            log.warning('%s error: DB not available: %s', self.alias, e)
            return CheckStatus(is_ok=False, description='%s DB error' % self.alias, got_errors=True)
        if not self.accounts:
            log.info('%s error: user %s not found', self.alias, login)
            return CheckStatus(is_ok=False, description='%s user not found' % self.alias)

        log.info('%s: found %d accounts for login %s. Processing...', self.alias, len(self.accounts), login)
        self.filter_by_password(password)
        if not self.accounts:
            log.info('%s error: password not matched', self.alias)
            return CheckStatus(is_ok=False, description='%s password not matched' % self.alias)

        log.info('%s auth successful', self.alias)
        return CheckStatus(is_ok=True, description='%s auth successful' % self.alias)
