# -*- coding: utf-8 -*-
import hashlib
import json
import logging
import time

from passport.backend.core.lazy_loader import (
    lazy_loadable,
    LazyLoader,
)
from passport.backend.perimeter.auth_api.common.base_checker import CheckStatus
from passport.backend.perimeter.auth_api.common.exceptions import DBError
from passport.backend.perimeter.auth_api.db.base_db_checker import (
    BaseDbChecker,
    BaseDbPasswordChecker,
)
from passport.backend.perimeter.auth_api.db.schemas import (
    long_users_table,
    mdm_users_table,
    motp_users_table,
    totp_users_table,
)
from passport.backend.perimeter.auth_api.utils.hash import truncated_hash


log = logging.getLogger('db.checkers')


@lazy_loadable()
class LongChecker(BaseDbPasswordChecker):
    db_table = long_users_table


@lazy_loadable()
class MDMChecker(BaseDbPasswordChecker):
    db_table = mdm_users_table


@lazy_loadable()
class MOTPChecker(BaseDbPasswordChecker):
    db_table = motp_users_table

    def check_password(self, account, password):
        login = account[self.db_table.c.username]
        secret = account[self.db_table.c.initsecret]
        pin = account[self.db_table.c.pin]
        current_time = int(time.time())

        motp_diag = dict(
            login=login,
            ts=current_time,
            secret_hash=truncated_hash(secret),
            secret_pin_hash=truncated_hash(secret + str(pin)),
            otp_hash=truncated_hash(password),
        )
        log.debug('MOTP diag: %s', json.dumps(motp_diag))

        max_period = 15 * 60
        step = 10
        t = current_time - max_period
        while t <= current_time + max_period:
            hasher = hashlib.md5()
            hasher.update(b'%d%s%s' % (int(t / step), secret.encode(), str(pin).encode()))
            md5 = hasher.hexdigest()[0:6]
            if password == md5:
                return True
            t += step
        return False


@lazy_loadable()
class TotpChecker(BaseDbChecker):
    db_table = totp_users_table

    @property
    def alias(self):
        return 'TOTP'

    def check_if_totp_is_on(self, login):
        try:
            self.get_all_accounts(login)
        except DBError as e:
            log.warning('%s error: DB not available: %s', self.alias, e)
            return CheckStatus(is_ok=False, description='%s DB error' % self.alias, got_errors=True)
        if not self.accounts:
            log.info('%s error: TOTP not enabled for user %s', self.alias, login)
            return CheckStatus(is_ok=False, description='%s not enabled' % self.alias)

        return CheckStatus(is_ok=True, description='%s is enabled' % self.alias)

    def check(self, **kwargs):
        raise NotImplementedError(
            'Don\'t call `TotpChecker.check` directly; '
            'use `LdapChecker.check` and `TotpChecker.check_if_totp_is_on` instead',
        )  # pragma: no cover


def get_long_checker():
    return LazyLoader.get_instance('LongChecker')


def get_mdm_checker():
    return LazyLoader.get_instance('MDMChecker')


def get_motp_checker():
    return LazyLoader.get_instance('MOTPChecker')


def get_totp_checker():
    return LazyLoader.get_instance('TotpChecker')
