# -*- coding: utf-8 -*-
import logging
import random
from time import time

from passport.backend.core.conf import settings
from passport.backend.core.lazy_loader import (
    lazy_loadable,
    LazyLoader,
)
from passport.backend.perimeter.auth_api.redis.storage import get_redis_storage


log = logging.getLogger('ldap.balancer')


@lazy_loadable()
class LdapBalancer(object):
    """
    Балансер AD-хостов, умеющий возвращать самый "быстрый" хост.
    Использует Redis для хранения истории времён ответов.
    """
    def __init__(self, servers=None):
        self._servers = dict.fromkeys(servers or settings.LDAP_SERVERS, 0)
        self._max_interval = settings.LDAP_SERVER_MONITORING_INTERVAL
        self._max_serie_size = settings.LDAP_SERVER_MONITORING_SERIE_SIZE
        self._redis = get_redis_storage(expire_time=self._max_interval)

    def _make_redis_key(self, server):
        return 'ldap:%s' % server

    def preprocess_and_calc_average(self, raw_timings_list):
        total_time, count = 0.0, 0
        for item in raw_timings_list:
            ts, response_time = item.split(':')
            ts = int(ts)
            response_time = float(response_time)
            if time() - ts > self._max_interval:
                break  # первыми идут самые свежие; если текущий для нас слишком стар, дальше лучше не станет.
            total_time += response_time
            count += 1

        if not count:
            return
        return total_time / count

    def get_timings(self):
        log.debug('Trying to get timings from Redis...')
        redis_keys = [self._make_redis_key(server) for server in self._servers]
        new_timings = {
            key.split(':', 1)[-1]: self.preprocess_and_calc_average(raw_timings_list)
            for key, raw_timings_list in self._redis.get_lists(*redis_keys).items()
        }

        if not new_timings:
            # Ошибка Redis, её мы уже залогировали
            return {}
        if set(self._servers.keys()) != set(new_timings.keys()):
            raise ValueError('Bad response from Redis: servers changed')  # недостижимый, но опасный кейс
        self._servers = new_timings
        return self._servers

    def get_random_server(self):
        server = random.choice(list(self._servers.keys()))
        log.debug('Random LDAP server chosen: %s', server)
        return server

    def get_best_server(self, max_time=None):
        """
        Возвращает самый "быстрый" хост с ожидаемым временем ответа не более max_time.
        Если таковых нет - возвращает None.
        """
        self.get_timings()

        # заменим неизвестные времена на максимальные, которые мы ещё можем себе позволить
        best_server = min(self._servers, key=lambda server: self._servers[server] or max_time)
        if (
            max_time is not None and
            self._servers[best_server] is not None and
            self._servers[best_server] > max_time
        ):
            # Время ответа лучшего из серверов известно и велико: даже от него мы, скорее всего, не дождёмся ответа
            return
        log.debug('Best LDAP server chosen: %s', best_server)
        return best_server

    def report_time(self, server, time_elapsed):
        """
        Пробует записать время ответа сервера в Redis, проглатывая ошибки недоступности Redis.
        """
        if server not in self._servers:
            raise ValueError('Unknown server: %s' % server)

        log.debug('Reporting timings to Redis...')
        time_now = time()
        self._redis.push_to_list(
            key=self._make_redis_key(server),
            value='%d:%.3f' % (time_now, time_elapsed),
            max_list_size=self._max_serie_size,
        )


def get_ldap_balancer():
    return LazyLoader.get_instance('LdapBalancer')
