# -*- coding: utf-8 -*-
import logging
from time import time

import ldap
from passport.backend.core.conf import settings
from passport.backend.core.lazy_loader import (
    lazy_loadable,
    LazyLoader,
)
from passport.backend.core.utils.ldap import (
    ldap_search,
    ROBOT_DN_PREFIX,
)
from passport.backend.perimeter.auth_api.common.base_checker import (
    BaseChecker,
    CheckStatus,
)
from passport.backend.perimeter.auth_api.ldap.balancer import get_ldap_balancer


log = logging.getLogger('ldap.checker')


# https://docs.microsoft.com/en-us/troubleshoot/windows-server/identity/useraccountcontrol-manipulate-account-properties
DONT_EXPIRE_PASSWORD_FLAG = 0x10000


@lazy_loadable()
class LdapChecker(BaseChecker):
    """
    Класс для проверки пароля в AD
    """
    def __init__(self):
        super(LdapChecker, self).__init__()
        self._balancer = get_ldap_balancer()

    @property
    def is_enabled(self):
        return True  # основной метод проверки пароля, доступен всегда

    @property
    def alias(self):
        return 'LDAP'

    def _get_connection(self, use_best_server=True):
        if use_best_server and settings.REDIS_ENABLED:
            server = self._balancer.get_best_server(
                max_time=settings.LDAP_NETWORK_TIMEOUT + settings.LDAP_TIMEOUT,
            )
            if server is None:
                return
        else:
            server = self._balancer.get_random_server()

        self.current_server = server
        ldap_url = '%s://%s:%d' % (settings.LDAP_SCHEME, server, settings.LDAP_PORT)
        return ldap.initialize(ldap_url)

    def _report_time(self, server, time_elapsed):
        if settings.REDIS_ENABLED:
            self._balancer.report_time(
                server=server,
                time_elapsed=time_elapsed,
            )

    def check(self, login, password, use_best_server=True, log_prefix='', is_robot=False, forbid_robots=False, **kwargs):
        if forbid_robots and is_robot:
            log.info('%s%s error: robots can\'t use %s from external IPs (not even checking password)', log_prefix, self.alias, self.alias)
            return CheckStatus(is_ok=False, description='%s account not suitable' % self.alias)

        time_start, time_elapsed = None, None
        try:
            ld = self._get_connection(use_best_server=use_best_server)
            if not ld:
                log.warning('%s%s error: no servers available', log_prefix, self.alias)
                return CheckStatus(is_ok=False, description='%s server unavailable' % self.alias, got_errors=True)
            time_start = time()  # замеряем время запроса, чтоб потом отправить его в Sentinel
            ld.set_option(ldap.OPT_X_TLS, ldap.OPT_X_TLS_DEMAND)
            ld.set_option(ldap.OPT_X_TLS_CACERTFILE, settings.LDAP_CA_CERT_FILE)
            ld.set_option(ldap.OPT_X_TLS_REQUIRE_CERT, ldap.OPT_X_TLS_DEMAND)
            ld.set_option(ldap.OPT_NETWORK_TIMEOUT, settings.LDAP_NETWORK_TIMEOUT)
            ld.set_option(ldap.OPT_TIMEOUT, settings.LDAP_TIMEOUT)
            # Форсируем создание нового контекста. Из-за бага опция должна указываться строго последней
            ld.set_option(ldap.OPT_X_TLS_NEWCTX, ldap.OPT_OFF)
            ld.protocol_version = ldap.VERSION3
            if settings.LDAP_START_TLS:
                ld.start_tls_s()
            ldap_login = '%s@%s' % (login, settings.LDAP_USER_DOMAIN)
            ld.simple_bind_s(ldap_login, password)
            # Если ошибок не возникло - значит, пароль верный
            time_elapsed = time() - time_start
        except ldap.INVALID_CREDENTIALS as e:
            time_elapsed = time() - time_start
            err_args = e.args[0].get('info', '')
            if 'data 773' in err_args or 'data 532' in err_args:
                log.info('%s%s: password correct, but change is required (%s)', log_prefix, self.alias, err_args)
                return CheckStatus(
                    is_ok=True,
                    description='%s auth successful' % self.alias,
                    require_password_change=True,
                    extra_data={},
                )
            else:
                log.info('%s%s error: invalid credentials (%s)', log_prefix, self.alias, err_args)
                return CheckStatus(is_ok=False, description='%s password invalid' % self.alias)
        except ldap.LDAPError as e:
            # репортим о большом времени ответа, чтоб в следующий раз не идти в этот сервер
            time_elapsed = settings.LDAP_FAILURE_RESPONSE_TIME_MULTIPLIER * (
                settings.LDAP_NETWORK_TIMEOUT + settings.LDAP_TIMEOUT
            )
            log.warning('%s%s error: %s: %s', log_prefix, self.alias, self.current_server, e)
            return CheckStatus(is_ok=False, description='%s server error' % self.alias, got_errors=True)
        finally:
            # Пишем в Sentinel затраченное время
            if time_elapsed is not None:
                self._report_time(server=self.current_server, time_elapsed=time_elapsed)

        # Пароль успешно проверили, проверим и другие свойства аккаунта
        search_result = ldap_search(login, ldap_client=ld, attrlist=['pwdLastSet', 'userAccountControl'])
        if not search_result:
            # Редкий случай, когда пользователя после ошибочного увольнения восстанавливают, но забывают перенести
            # в нужную группу. Для диагностики и починки стоит обращаться в хелпдеск.
            log.warning('Failed to find account `%s` after successful bind. Probably it is in wrong OU', login)
            return CheckStatus(is_ok=False, description='%s account probably in wrong OU' % self.alias)

        dn, attrs = search_result
        is_robot = is_robot or ROBOT_DN_PREFIX in dn.split(',')
        extra_data = {
            'is_account_robot': is_robot,
        }
        dont_expire_password = bool(int(attrs.get('userAccountControl', 0)) & DONT_EXPIRE_PASSWORD_FLAG)
        pwd_last_set = int(attrs.get('pwdLastSet', -1))
        log.debug(
            '%s%s account flags: dont_expire_password=%s, pwd_last_set=%s, is_account_robot=%s',
            log_prefix,
            self.alias,
            dont_expire_password,
            pwd_last_set,
            is_robot,
        )
        require_password_change = (
            settings.ALLOW_PASSWORD_CHANGE and
            pwd_last_set == 0 and
            not dont_expire_password
        )

        ld.unbind()

        if forbid_robots and extra_data['is_account_robot']:
            log.info('%s%s error: robots can\'t use %s from external IPs', log_prefix, self.alias, self.alias)
            return CheckStatus(is_ok=False, description='%s account not suitable' % self.alias)

        log.info('%s%s auth successful', log_prefix, self.alias)
        return CheckStatus(
            is_ok=True,
            description='%s auth successful' % self.alias,
            require_password_change=require_password_change,
            extra_data=extra_data,
        )


def get_ldap_checker():
    return LazyLoader.get_instance('LdapChecker')
