# -*- coding: utf-8 -*-
import logging

from passport.backend.core.builders.blackbox import (
    BaseBlackboxError,
    Blackbox,
    BLACKBOX_OAUTH_VALID_STATUS,
)
from passport.backend.core.conf import settings
from passport.backend.core.lazy_loader import (
    lazy_loadable,
    LazyLoader,
)
from passport.backend.core.types.login.login import normalize_login
from passport.backend.core.utils.decorators import cached_property
from passport.backend.perimeter.auth_api.common.base_checker import (
    BaseChecker,
    CheckStatus,
)


log = logging.getLogger('oauth.checker')


MIN_TOKEN_LENGTH = 32


@lazy_loadable()
class OAuthChecker(BaseChecker):
    """
    Класс для проверки oauth-токена в качестве пароля
    """

    @property
    def is_enabled(self):
        return True

    @cached_property
    def blackbox(self):
        return Blackbox(
            blackbox=settings.BLACKBOX_URL,
            timeout=settings.BLACKBOX_TIMEOUT,
            retries=settings.BLACKBOX_RETRIES,
            use_tvm=True,
        )

    def check(self, login, password, ip, allowed_scopes, **kwargs):
        if len(password) < MIN_TOKEN_LENGTH:
            log.info('%s error: password does not look like a token', self.alias)
            return CheckStatus(is_ok=False, description='%s not a token' % self.alias)
        try:
            bb_response = self.blackbox.oauth(
                oauth_token=password,
                ip=ip,
                dbfields=(),
                attributes=(),
                need_display_name=False,
                need_aliases=False,
                get_hidden_aliases=False,
            )
            if bb_response['status'] != BLACKBOX_OAUTH_VALID_STATUS:
                log.info('%s error: invalid token', self.alias)
                return CheckStatus(is_ok=False, description='%s invalid token' % self.alias)
            elif not any(scope in bb_response['oauth']['scope'] for scope in allowed_scopes):
                log.info(
                    '%s error: insufficient scope (neither of `%s` is in `%s`)',
                    self.alias,
                    ' '.join(allowed_scopes),
                    ' '.join(bb_response['oauth']['scope']),
                )
                return CheckStatus(is_ok=False, description='%s insufficient scope' % self.alias)
            elif normalize_login(login) != normalize_login(bb_response['login']):
                log.info(
                    '%s error: token from another account (%s != %s)',
                    self.alias,
                    bb_response['login'],
                    login,
                )
                return CheckStatus(is_ok=False, description='%s token from another account' % self.alias)
            else:
                log.info('%s auth successful', self.alias)
                return CheckStatus(is_ok=True, description='%s auth successful' % self.alias)
        except BaseBlackboxError as e:
            log.warning('%s error: %s', self.alias, e)
            return CheckStatus(is_ok=False, description='%s blackbox error' % self.alias, got_errors=True)


def get_oauth_checker():
    return LazyLoader.get_instance('OAuthChecker')
