# -*- coding: utf-8 -*-
import logging

from passport.backend.core.conf import settings
from passport.backend.core.lazy_loader import (
    lazy_loadable,
    LazyLoader,
)
from passport.backend.perimeter.auth_api.common.base_checker import (
    BaseChecker,
    CheckStatus,
)
from passport.backend.perimeter.auth_api.redis.storage import get_redis_storage
from passport.backend.perimeter.auth_api.utils.common import parse_bool
from passport.backend.perimeter.auth_api.utils.hash import check_hashed_string


log = logging.getLogger('redis.checker')


def make_redis_key(login, auth_type, ip):
    return '_'.join([login, auth_type, ip])


@lazy_loadable()
class RedisChecker(BaseChecker):
    """
    Класс для поиска авторизации в списке успешных закешированных
    """
    def __init__(self):
        super(RedisChecker, self).__init__()
        self._storage = get_redis_storage()

    @property
    def is_enabled(self):
        return settings.REDIS_ENABLED

    def check(self, login, password, ip, auth_type, **kwargs):
        redis_key = make_redis_key(login, auth_type, ip)
        value_dict = self._storage.get(redis_key) or {}
        cached_password = value_dict.get('password_hash')
        if check_hashed_string(password, cached_password):
            log.info('Redis auth successful')
            second_steps_raw = value_dict.get('second_steps')
            second_steps = second_steps_raw.split(',') if second_steps_raw else []
            require_password_change = parse_bool(value_dict.get('require_password_change', '0'))
            return CheckStatus(
                is_ok=True,
                description='Auth found in cache',
                require_second_steps=second_steps,
                require_password_change=require_password_change,
            )
        elif cached_password:
            log.info('%s: password differs from cached', self.alias)
            return CheckStatus(is_ok=False)
        else:
            log.info('%s: auth not found in cache', self.alias)
            return CheckStatus(is_ok=False)


def get_redis_checker():
    return LazyLoader.get_instance('RedisChecker')
