# -*- coding: utf-8 -*-
from __future__ import absolute_import

import logging

from passport.backend.core.conf import settings
from passport.backend.core.lazy_loader import (
    lazy_loadable,
    LazyLoader,
)
from redis import (
    Redis,
    RedisError,
)


log = logging.getLogger('redis.storage')


@lazy_loadable()
class RedisStorage(object):
    """
    Класс-обёртка над Redis, поддерживающая get- и set-операции над короткоживующими ключами
    """
    def __init__(self, host=None, port=None, expire_time=None, password=None,
                 timeout=None, connect_timeout=None):
        self._conn = None
        self._host = host or settings.REDIS_HOST
        self._port = port or settings.REDIS_PORT
        self._expire_time = expire_time or settings.REDIS_EXPIRE_TIME
        self._password = password or settings.REDIS_PASSWORD
        self._timeout = timeout or settings.REDIS_TIMEOUT
        self._connect_timeout = connect_timeout or settings.REDIS_CONNECT_TIMEOUT

    def connect(self):
        self._conn = Redis(
            host=self._host,
            port=self._port,
            db=0,
            password=self._password,
            socket_timeout=self._timeout,
            socket_connect_timeout=self._connect_timeout,
        )

    def _get_connection(self):
        if self._conn is None:
            self.connect()
        return self._conn

    def set(self, key, value_dict):
        conn = self._get_connection()
        try:
            pipe = conn.pipeline()
            pipe.multi()
            pipe.hmset(key, value_dict)
            pipe.expire(key, self._expire_time)
            result, _ = pipe.execute()
            if not result:
                raise RedisError('failed to set value for key [%s]' % key)
            return True
        except RedisError as e:
            log.warning('Redis error: %s' % e)
            return False

    def get(self, key):
        conn = self._get_connection()
        try:
            pipe = conn.pipeline()
            pipe.multi()
            pipe.hgetall(key)
            pipe.ttl(key)
            value_dict, ttl = pipe.execute()
            if not ttl:
                return
            return {
                key.decode(): value.decode()
                for key, value in value_dict.items()
            }
        except RedisError as e:
            log.warning('Redis error: %s' % e)
            return

    def push_to_list(self, key, value, max_list_size=None):
        conn = self._get_connection()
        try:
            pipe = conn.pipeline()
            pipe.multi()
            pipe.lpush(key, value)
            if max_list_size:
                pipe.ltrim(key, 0, max_list_size - 1)
            pipe.expire(key, self._expire_time)
            pipe.execute()
            return True
        except RedisError as e:
            log.warning('Redis error: %s' % e)
            return False

    def get_lists(self, *keys):
        conn = self._get_connection()
        result = {}
        try:
            pipe = conn.pipeline()
            pipe.multi()
            for key in keys:
                pipe.lrange(key, 0, -1)
                pipe.ttl(key)
            rv = pipe.execute()
            lrange_results, ttl_results = rv[::2], rv[1::2]
            for key, lrange_result, ttl_result in zip(keys, lrange_results, ttl_results):
                if not ttl_result:
                    value = []
                else:
                    value = [item.decode() for item in lrange_result]
                result[key] = value
            return result
        except RedisError as e:
            log.warning('Redis error: %s' % e)
            return {}


def get_redis_storage(**kwargs):
    return LazyLoader.get_instance('RedisStorage', **kwargs)
