# -*- coding: utf-8 -*-
import logging

from passport.backend.perimeter.auth_api.auth_logic import get_auth_logic
from passport.backend.perimeter.auth_api.common.exceptions import ValidationError
from passport.backend.perimeter.auth_api.logging_utils.helpers import mask_sensitive_fields
from passport.backend.perimeter.auth_api.logging_utils.request_id import get_request_id
from passport.backend.perimeter.auth_api.utils.common import parse_bool
from passport.backend.perimeter.auth_api.views.base import (
    get_request_values,
    json_response,
    validate_form,
)


log = logging.getLogger('views.auth')


def auth():
    form_values = get_request_values()
    log.info('Got request: %s', mask_sensitive_fields(form_values))
    try:
        form_values = validate_form(
            form_values,
            ['login', 'password', 'ip', 'is_ip_internal', 'is_ip_robot', 'is_robot', 'auth_type'],
        )
        login = form_values['login']
    except ValidationError as e:
        response_values = dict(
            status='error',
            description=str(e),
        )
        login = ''
    else:
        check_status = get_auth_logic().perform_auth(
            login=login,
            password=form_values['password'],
            ip=form_values['ip'],
            is_ip_internal=parse_bool(form_values['is_ip_internal']),
            is_ip_robot=parse_bool(form_values['is_ip_robot']),
            is_robot=parse_bool(form_values['is_robot']),
            auth_type=form_values['auth_type'],
        )

        response_values = dict(
            description=check_status.description,
        )
        if check_status.require_password_change:
            response_values.update(password_change_required=True)

        if not check_status.is_ok:
            response_values.update(status='error' if check_status.got_errors else 'password_invalid')
        elif check_status.require_second_steps:
            response_values.update(
                status='second_step_required',
                second_steps=list(check_status.require_second_steps),
            )
        else:
            response_values.update(status='ok')

    response_values.update(request_id=get_request_id())
    log.info('Responded with: %s (for login=%s)', response_values, login)
    return json_response(**response_values)
