# -*- coding: utf-8 -*-
import json
import logging

from django.conf import settings
from django.utils.deprecation import MiddlewareMixin
from django_idm_api.middleware import TVMMiddleware
from passport.backend.core.tvm import get_tvm_credentials_manager
from passport.backend.perimeter_api.perimeter_api.globals import set_request
from ticket_parser2.exceptions import TicketParsingException


log = logging.getLogger('perimeter_api.request')


MAX_RESPONSE_SIZE = 1000


def mask_sensitive_fields(data, fields=('context', 'secret')):
    for field in fields:
        if field in data:
            data[field] = '***'
    return data


class LoggingMiddleware(MiddlewareMixin):
    """Логирует запрос и ответ"""
    def process_request(self, request):
        log.info('Got request: %s %s (GET: %s, POST: %s)', request.method, request.path, request.GET, request.POST)

    def process_response(self, request, response):
        status = response.status_code

        if status != 200:
            masked_content = response.content.decode()
        elif response['Content-Type'] == 'application/json':
            data = json.loads(response.content)
            masked_content = json.dumps(mask_sensitive_fields(data))
        else:
            masked_content = '***'

        log.info('Responded with status=%s: %s', status, masked_content[:MAX_RESPONSE_SIZE])
        return response


class StoreRequestMiddleware(MiddlewareMixin):
    """Сохраняет реквест в глобальную переменную"""
    def process_request(self, request):
        set_request(request)

    def process_response(self, request, response):
        set_request(None)
        return response


class IDMAuthMiddleware(TVMMiddleware):
    """В отличие от родителя, проверяем тикеты через наш TvmCredentialsManager, а не общую либу TVM2"""
    def _check_tvm_ticket(self, request):
        tvm_ticket = request.headers.get('X-Ya-Service-Ticket')
        if not tvm_ticket:
            log.warning('No TVM ticket passed')
            return False

        try:
            parsed_ticket = get_tvm_credentials_manager().service_context.check(tvm_ticket)
        except TicketParsingException as ex:
            log.warning('Invalid TVM ticket: %s (%s): %s', ex.status, ex.message, ex.debug_info)
            return False

        if parsed_ticket.src != settings.IDM_TVM_CLIENT_ID:
            log.warning('Unknown TVM ticket src: %s != %s', parsed_ticket.src, settings.IDM_TVM_CLIENT_ID)
            return False

        return True
