# -*- coding: utf-8 -*-
from collections import defaultdict
import random
import subprocess

from django.conf import settings
from django.utils.functional import cached_property
from django_idm_api.exceptions import RoleNotFound
from django_idm_api.hooks import BaseHooks
from passport.backend.core.builders.passport import (
    BasePassportError,
    Passport,
)
from passport.backend.perimeter_api.perimeter_api.exceptions import UserAlreadyExists
from passport.backend.perimeter_api.perimeter_api.models import (
    LongUser,
    MDMUser,
    MotpUser,
    TotpUser,
)


def run_pwgen(arg_string):
    process = subprocess.Popen(['pwgen'] + arg_string.split(), stdout=subprocess.PIPE)
    output, err = process.communicate()
    result = output.strip().decode()
    if not result:
        raise RuntimeError('Failed to get data from pwgen')
    return result


def get_passport():
    return Passport(
        url=settings.PASSPORT_URL,
        consumer=settings.PASSPORT_CONSUMER,
        timeout=settings.PASSPORT_TIMEOUT,
        retries=settings.PASSPORT_RETRIES,
    )


class Hooks(BaseHooks):
    _successful_result = dict(code=0)

    _role_to_model = {
        'long': LongUser,
        'mdm': MDMUser,
        'motp': MotpUser,
        'totp': TotpUser,
    }
    _role_to_name = {
        'long': 'Длинный пароль',
        'mdm': 'Пароль для iOS',
        'motp': 'Одноразовый пароль (MOTP)',
        'totp': 'Одноразовый пароль (TOTP)',
    }

    @cached_property
    def passport_builder(self):
        return get_passport()

    def info(self):
        """Срабатывает, когда Управлятор спрашивает систему о том, какие роли она поддерживает.
        """
        data = dict(
            code=0,
            roles=dict(
                slug='role',
                name='роль',
                values=self._role_to_name,
            ),
        )
        return data

    def add_role(self, login, role, fields, subject_type):
        """Обрабатывает назначение новой роли пользователю с указанным логином.
        """
        role = role.get('role', '')

        if role not in self._role_to_model.keys():
            raise RoleNotFound('Нет такой роли')

        cls = self._role_to_model[role]
        if cls.objects.filter(username=login).exists():
            raise UserAlreadyExists('Пользователь уже имеет такую роль')

        if role in ('long', 'mdm'):
            gpassword = run_pwgen('-s -n -B -c 32 1')
            cls.objects.create(username=login, password=gpassword)

            retval = self._successful_result.copy()
            if role == 'mdm':
                # для MDM мы не сообщаем пароль пользователю
                # напрямую, пароль приезжает на устройство автоматически
                retval['context'] = {}
            else:
                retval['context'] = dict(password=gpassword)

            return retval

        elif role == 'motp':
            initsecret = run_pwgen('-n -B -c 16 1')
            pin = random.randint(1000, 9999)
            cls.objects.create(username=login, pin=pin, initsecret=initsecret.lower())
            retval = self._successful_result.copy()
            retval['context'] = dict(
                pin=pin,
                initsecret=initsecret.lower(),
            )
            return retval

        elif role == 'totp':
            rv = self.passport_builder.rfc_2fa_enable(login=login)
            cls.objects.create(username=login)
            retval = self._successful_result.copy()
            retval['context'] = dict(
                secret=rv['secret'],
            )
            return retval

    def remove_role(self, login, role,  data, is_fired, subject_type):
        """Вызывается, если у пользователя с указанным логином отзывают роль.
        """
        role = role.get('role', '')

        if role in self._role_to_model.keys():
            cls = self._role_to_model[role]
            cls.objects.filter(username=login).delete()

            if role == 'totp':
                try:
                    self.passport_builder.rfc_2fa_disable(login=login)
                except BasePassportError:
                    pass

            return self._successful_result

        raise RoleNotFound('Нет такой роли')

    def get_all_roles(self):
        """Вызывается для аудита и первичной синхронизации.
        Отдает всех пользователей, которые известны в системе, и их роли.
        """
        users = defaultdict(list)
        for role, cls in self._role_to_model.items():
            for user in cls.objects.filter():
                users[user.username].append(dict(role=role))

        return dict(
            self._successful_result,
            users=[
                dict(login=login, roles=roles)
                for login, roles in users.items()
            ],
        )
