# -*- coding: utf-8 -*-
import json
import logging

from django.db import DatabaseError
from django.http import HttpResponse
from django.shortcuts import get_object_or_404
from passport.backend.core.builders.passport import (
    BasePassportError,
    PassportTemporaryError,
)
from passport.backend.perimeter_api.perimeter_api.models import (
    MDMUser,
    TotpUser,
)
from passport.backend.perimeter_api.perimeter_api.roles_hooks import get_passport


log = logging.getLogger('perimeter_api.views')


def get_mdm_pass(request, username):
    password = get_object_or_404(MDMUser, username=username).password

    response = HttpResponse(content_type='text/plain')
    response.write(password)
    response['Content-Length'] = len(password)
    return response


def recreate_totp_secret(request, username):
    try:
        rv = get_passport().rfc_2fa_enable(login=username)
        if not TotpUser.objects.filter(username=username).exists():
            TotpUser.objects.create(username=username)
        response_values = {
            'status': 'ok',
            'secret': rv['secret'],
        }
    except PassportTemporaryError:
        response_values = {
            'status': 'error',
            'error': 'passport.failed',
        }
    except BasePassportError:
        response_values = {
            'status': 'error',
            'error': 'passport.permanent_error',
        }
    except DatabaseError as e:
        log.warning(e)
        response_values = {
            'status': 'error',
            'error': 'database.failed',
        }
    except Exception as e:
        log.error(e, exc_info=True)
        response_values = {
            'status': 'error',
            'error': 'exception.unhandled',
        }

    return HttpResponse(
        content=json.dumps(response_values),
        content_type='application/json',
    )
