package ADM::Core::GlobalSearcher;

use common::sense;

use Data::Dumper;
use List::Util 'first';
use List::MoreUtils qw/part uniq/;

use ADM::Utils;

use Class::XSAccessor {
    constructor => 'new',
    accessors   => [qw/acd filter limit accounts uids warning found_uids login_by_removed_uid/],
};

sub search {
    my $self = shift;

    $self->found_uids(undef);
    $self->accounts([]);
    $self->uids([]);

    my $rows_by_active_alias  = $self->search_by_active_alias;
    my $uids_by_active_alias  = [ map { $_->[0] } @$rows_by_active_alias ];

    my $rows_by_removed_alias = $self->search_by_removed_alias;
    my $uids_by_removed_alias = [ map { $_->[0] } @$rows_by_removed_alias ];
    my %login_by_removed_uid  = map +( $_->[0] => $_->[1] ), @$rows_by_removed_alias;
    $self->login_by_removed_uid(\%login_by_removed_uid);

    my $uids_by_aliases       = ADM::Utils::union($uids_by_active_alias, $uids_by_removed_alias);

    $self->add_found_uids($uids_by_aliases);
    return if $self->is_too_many_found_uids('aliases');

    my $uids_by_phone   = $self->search_by_phone;
    my $uids_by_email   = $self->search_by_email;
    my $uids_by_ip      = $self->search_by_registration_ip;

    $self->add_found_uids($uids_by_phone, $uids_by_email, $uids_by_ip);
    return if $self->is_too_many_found_uids('aliases/external');

    my $uids_by_attributes = $self->search_by_attributes;

    $self->add_found_uids($uids_by_attributes);
    return if $self->is_too_many_found_uids('aliases/external/attributes');

    return unless $self->found_uids_number;

    $self->order_desc;

    my $accounts = $self->load_accounts($self->found_uids);
    my $uids     = $self->filter_uids($self->found_uids, $accounts);

    splice @$uids,     $self->limit;
    splice @$accounts, $self->limit;

    $self->accounts($accounts);
    $self->uids($uids);

    return;
}

###

sub is_too_many_found_uids {
    my $self = shift;
    my ($type) = @_;

    my $max_limit = $ADM::Utils::FOUND_UIDS_MAX_LIMIT;
    my $number = $self->found_uids_number;

    if ($number >= $max_limit) {
        $self->warning("too many uids by $type were found ($number), you should obtain more specific filters");
        return 1;
    }
    else {
        return 0;
    }
}

sub add_found_uids {
    my $self = shift;

    my $old = $self->found_uids;
    my $new = ADM::Utils::intersection($old, @_);

    $self->found_uids($new);
    $self->exclude_pdd;

    return;
}

sub found_uids_number {
    my $self = shift;
    
    return 0 unless $self->found_uids;
    return scalar @{ $self->found_uids };
}

###

sub search_by_phone {
    my $self = shift;

    return unless my $phone = $self->filter->{phone};

    my $result = eval { ADM::Requester::FindAccountsByPhone($phone) };

    die 'search_by_phone has been failed'
      unless $result;

    return $result;
}

sub search_by_email {
    my $self = shift;

    return unless my $email = $self->filter->{email};

    my $result = eval { ADM::Requester::FindAccountsByEmail(lc $email) };

    die 'search_by_email has been failed'
      if $@;

    return $result;
}

sub search_by_registration_ip {
    my $self = shift;

    return unless $self->filter->{ip};

    my $registrations_by_ip = ADM::Requester::GetHistoryDb3RegistrationsByIp(
        ip   => $self->filter->{ip},
        from => $self->filter->{registered_after},
        to   => $self->filter->{registered_before},
    );

    die 'search_by_registration_ip has been failed'
      unless $registrations_by_ip;

    my $result = [ map { $_->{uid} } @$registrations_by_ip ];

    return $result;
}

sub search_by_active_alias {
    my $self = shift;

    my $result = eval { $self->acd->FindAccountsByActiveAliases(%{ $self->filter }) };

    die "search_by_active_aliases has been failed: $@"
      if $@;

    return $result;
}

sub search_by_removed_alias {
    my $self = shift;

    return if $self->filter->{only_active} or $self->filter->{only_enabled};

    my $result = eval { $self->acd->FindAccountsByRemovedAliases(%{ $self->filter }) };

    die 'search_by_removed_aliases has been failed'
      if $@;

    return $result;
}

sub search_by_attributes {
    my $self = shift;

    my %filter = %{ $self->filter };
    $filter{uids} = $self->found_uids;

    my $result = eval { $self->acd->FindAccountsByAttributes(%filter) };

    die 'search_by_attributes has been failed'
      if $@;

    return $result;
}

###

sub exclude_pdd {
    my $self = shift;

    my $old = $self->found_uids;

    my $new
      = $self->filter->{exclude_pdd}
      ? [ grep { $_ < $ADM::Utils::PDD_MIN_UID } @$old ]
      : $old;

    $new = undef
      unless @$new;

    $self->found_uids($new);

    return;
}

sub filter_accounts {
    my $self = shift;
    my ($accounts) = @_;

    my $filter = $self->filter;

    my @result;

    for my $account (@$accounts) {
        next if $filter->{only_enabled}      and not $account->is_enabled;
        next if $filter->{exclude_pdd}       and $account->is_pdd;
        next if $filter->{exclude_lite}      and $account->is_lite;
        next if $filter->{registered_after}  and $account->registration_ts < $filter->{registered_after};
        next if $filter->{registered_before} and $account->registration_ts > $filter->{registered_before};

        push @result, $account;
    }

    return \@result;
}

sub filter_uids {
    my $self = shift;
    my ($uids, $accounts) = @_;

    my $result
      = $self->filter->{only_active}
      ? [ map { $_->uid } @$accounts ]
      : $uids;

    return $result;
}

###

sub load_accounts {
    my $self = shift;
    my ($uids) = @_;

    my $groups = $self->split_uids_by_groups($uids);

    my $groups_number = scalar @$groups;
    my $uids_number   = scalar @$uids;

    my @result;

    for my $i (1 .. $groups_number) {
        my $group = $groups->[$i - 1];

        ADM::Logs::DeBug(
            sprintf 'get_accounts_by_uids: %d/%d (number=%d)',
            $i, $groups_number, $uids_number 
        );

        my $found_accounts    = $self->get_accounts_by_uids($group);
        my $filtered_accounts = $self->filter_accounts($found_accounts);

        push @result, @$filtered_accounts;

        last if @result >= $self->limit;
    }

    return \@result;
}

sub get_accounts_by_uids {
    my $self = shift;
    my ($uids) = @_;

    my $result = eval { $self->acd->dbaf->GetAccountsExceptionable($uids) };

    die 'get_accounts_by_uids has been failed'
      if $@;

    return $result;
}

###
sub split_uids_by_groups {
    my $self = shift;
    my ($uids) = @_;

    my $i;
    my $group_max_size = $admin::Conf->GetVal('blackbox_group_max_size');
    my @groups         = part { $i++ / $group_max_size } @$uids;

    return \@groups;
}

sub order_desc {
    my $self = shift;

    my $pdd_min_uid = $ADM::Utils::PDD_MIN_UID;

    my $old = $self->found_uids;

    # Бьём uid'ы на две группы: сначала обычные, потом ПДД-шные.
    # Внутри каждой группы сортируем по убыванию.
    my $new = [
        sort {
            return -1 if $b >= $pdd_min_uid and $a < $pdd_min_uid;
            return  1 if $a >= $pdd_min_uid and $b < $pdd_min_uid;
            return $b <=> $a;
        } @$old
    ];

    $self->found_uids($new);

    return;
}

1;
