package ADM::Core::Passport;

use strict;
use utf8;
use open qw(:std :utf8);

use Storable;
use Encode;
use Data::Dumper;
use MIME::Base64;
use Digest::MD5;
use Time::HiRes qw/gettimeofday tv_interval/;
use POSIX 'ceil';

use AIO::Requester;
use Common::Puny 'to_puny';
use Database::Converter;
use Global;

sub is_stop_word {
    my ($self, $login) = @_;

    my %params = (
        logins => [ $login ],
    );
    my $requester = AIO::Requester->new(1);
    my $statuses  = $requester->LoginOccupation(%params);

    return undef
      unless $statuses;

    my $status = $statuses->{$login};

    return $status eq 'stoplist' ? 1 : 0;
}

sub ListAllServices { $main::Conf->GetHCVal('sid2from') }

sub EnableAccounts {
    my ($self, $param) = @_;

    my $dbaf = $self->dbaf;

    my @uids = @{ $param->{uids} };

    return 1
      if not defined $param->{enable} and not defined $param->{karma};

    my $is_enabled
      = (not length $param->{enable}) ? undef
      : $param->{enable}              ? 1
      :                                 0;
    my $karma_prefix
      = (not length $param->{karma}) ? undef
      : $param->{karma}              ? 2
      :                                1;

    for my $uid (@uids) {
        if (defined $is_enabled) {
            unless ($dbaf->SetAccountIsEnabled($uid, $is_enabled)) {
                ADM::Logs::IntErr("can't change account.is_enabled");
                return undef;
            }
        }

        if (defined $karma_prefix) {
            unless ($dbaf->SetKarmaPrefix($uid, $karma_prefix)) {
                ADM::Logs::IntErr("can't change karma.prefix");
                return undef;
            }
        }

        my $opdata = {
            optype  => 'mod',
            ip_from => $ENV{REMOTE_ADDR},
            uid     => $uid,
        };

        my $changes = {
            admin_login   => $param->{admin_login},
            admin_comment => $param->{admin_comment},
        };
        $changes->{ena} = $is_enabled
          if defined $is_enabled;
        $changes->{prefix} = $karma_prefix
          if defined $karma_prefix;

        unless (ADM::Logs::LogChanges($opdata, $changes)) {
            ADM::Logs::IntErr("can't log changes in DisableLogins");
            return undef;
        }
    }

    return 1;
}

sub WarnUsers {
    my ($self, %args) = @_;

    my $dbaf = $self->dbaf;

    my $users = $args{users};
    my $is_logins = $args{is_logins};
    my $limit = $args{limit};
    if ($limit > scalar @$users) {
        $limit = scalar @$users;
    }

    my $results = [];

    for (0 .. $limit - 1) {
        my $user = shift @$users;
        
        # определим uid юзера
        my $account = $self->dbaf->GetAccount($user);

        unless ($account) {
            push @$results, "$user not found";
            next;
        }

        if (not $args{message_id} and $account->password->is_2fa_enabled) {
            push @$results, "$user with 2fa, ignored";
            next;
        }

        my $uid = $account->uid;
        ADM::Logs::Test("WarnUsers: uid - $uid");

        my $result
          = $args{message_id}
          ? $dbaf->InsertUserMessage($uid, $args{message_id}, 1)
          : $dbaf->SetNeedChangePassword($uid);

        return undef
          unless $result;

        my $opdata = {
            optype  => 'mod',
            ip_from => $ENV{REMOTE_ADDR},
            uid     => $uid,
        };
        my $changes = {};

        if (not $args{message_id}) {
            $changes = {
                sid         => 8,
                login_rule  => 5,
                admin_login => ADM::Utils::GetCurrentLogin(),
            };
        }

        my $time = time;

        unless ($dbaf->GlobalLogout($uid, $time)) {
            ADM::Logs::LogChanges($opdata, $changes);
            return;
        }

        $changes->{glogout} = $time;
        ADM::Logs::LogChanges($opdata, $changes);

        # собираем результаты
        push @$results, $user . ($is_logins ? " $uid" : '');
    }

    return $results;
}

sub SetIsVipAccount {
    my ($self, %param) = @_;

    my $sid = 671;
    my $uid = $param{uid};

    my $result
      = $param{flag}
      ? $self->dbaf->SubscribeAccountToSid($uid => $sid)
      : $self->dbaf->UnsubscribeAccountFromSid($uid => $sid);

    return
      unless $result;

    my $opdata = {
        optype  => 'mod',
        ip_from => $ENV{REMOTE_ADDR},
        uid     => $uid,
    };

    my $method = $param{flag} ? 'add_sid' : 'rm_sid';
    my $changes = {
        $method     => $sid,
        admin_login => $param{admin_login},
    };

    unless (ADM::Logs::LogChanges($opdata, $changes)) {
        ADM::Logs::IntErr("can't log changes in SetIsVipAccount");
        return;
    }

    return 1;
}

sub ConvertRequestParamsToFilter {
    my $self = shift;
    my ($params) = @_;

    my %filter;

    # Точные фильтры

    if (my $uid = $params->{uid}) {
        die 'uid must contain only digits'
          unless $uid =~ /^\d+$/;

        $filter{uid} = $uid;
    }

    if (my $suid = $params->{suid}) {
        die 'suid must contain only digits'
          unless $suid =~ /^\d+$/;

        $filter{suid} = $suid;
    }

    # Неточные фильтры

    if (my $login = $params->{login}) {
        $login = lc $login;
        $login =~ s/^\s+//;
        $login =~ s/\s+$//;

        $login =~ s{
            ^
            (?:https?://)?
            (?:www\.)?
            ([^.]+)                # login wo/ hyphen
            \.(?:narod2?|ya)\.ru
            (?:/.*)?
            $
        }{$1}ix;

            my ($username, $domain) = split /\@/, $login, 2;

            my $phone_number;
            my $external_domain_id = $main::Conf->GetHCVal('ExternalDomains')->{$domain};
            my $is_yandex_domain   = $main::Conf->GetHCVal('YandexDomains')->{$domain} ? 1 : 0;

            $domain = undef
              if $is_yandex_domain;

            if ($username =~ /[*?]/) {
                die 'login must start with minimum 1 character before "*"'
                  unless $username =~ /^ .{1,} [*?] /x;
            }

            # Если похож на телефонный номер, то сносим все ненужные символы, кроме цифр и символов маски для поиска
            if (not $username =~ /[a-z]/i) {
                $username =~ s/[^*?\d]//g;
            }

            $filter{username} = $username;

            if ($domain) {
                $filter{domain}      = $domain; # Нужен, чтобы отличать поиски с наличием домена от поисков без него
                $filter{lite_domain} = $domain;
                $filter{pdd_domain}  = $domain;
            }

            if ($external_domain_id) {
                $filter{external_domain_id} = $external_domain_id;
            }
    }

    if (my $name = $params->{fio}) {
        my ($lastname, $firstname) = split /\s+/, $name, 2;

        if ($lastname =~ /[*?]/) {
            die 'lastname must start with minimum 1 character before "*"'
              unless $lastname =~ /^ .{1,} [*?] /x;
        }

        if ($firstname =~ /[*?]/) {
            die 'firstname must start with minimum 1 character before "*"'
              unless $firstname =~ /^ .{1,} [*?] /x;
        }

        $filter{firstname} = $firstname
          if length $firstname;
        $filter{lastname}  = $lastname
          if length $lastname;
    }

    if (my $birthdate = $params->{birth_date}) {
        my $birthday;
        my $birthyear;

        if ($birthdate =~ /^(\d\d\d\d)-(\d\d)-(\d\d)$/) {
            $birthday  = "$2-$3";
            $birthyear = $1;
            if ($birthyear =~ /^0+$/) {
                $birthyear = undef;
            }
        }
        elsif ($birthdate =~ /^\d\d-\d\d$/) {
            $birthday = $birthdate;
        }
        else {
            die 'birthdate must be either in "yyyy-mm-dd" or "mm-dd" format';
        }

        $filter{birthday} = $birthday
          if $birthday;
        $filter{birthyear} = $birthyear
          if $birthyear;
    }

    if (my $registered_after = $params->{reg_date_from}) {
        die 'registered_after must be in "yyyy-mm-dd" format'
          unless $registered_after =~ /^\d\d\d\d-\d\d-\d\d$/;

        $filter{registered_after} = Global::DatetimeToTimestamp($registered_after);
    }

    if (my $registered_before = $params->{reg_date_to}) {
        die 'registered_before must be in "yyyy-mm-dd" format'
          unless $registered_before =~ /^\d\d\d\d-\d\d-\d\d$/;

        $filter{registered_before} = Global::DatetimeToTimestamp("$registered_before 23:59:59");
    }

    if (my $ip = $params->{ip_from}) {
        $filter{ip} = $ip;
    }

    if (my $email = $params->{email}) {
        $filter{email} = $email;
    }

    if (my $phone = $params->{phone}) {
        $phone =~ s/^\s+//g;
        $phone =~ s/^8/+7/g;

        if ($phone =~ /^\+/) {
            $phone =~ s/\D//g;
            $phone = "+$phone";
        }
        else {
            $phone =~ s/\D//g;
        }

        if ($phone =~ /^9\d{9}$/) {
            $phone = "+7$phone";
        }

        $filter{phone} = $phone;
    }

    if (my $karma = $params->{karma}) {
        $filter{karma} = $karma;
    }

    # Дополнительные флаги
    my $only_enabled = $params->{ena};
    my $only_active  = $params->{live};     # Только в актуальных аккаунтах, без поиска по удалённым
    my $exclude_pdd  = $params->{nopdd};
    my $exclude_lite = $params->{nolight};

    $filter{only_enabled} = 1 if $only_enabled;
    $filter{only_active}  = 1 if $only_active;
    $filter{exclude_pdd}  = 1 if $exclude_pdd;
    $filter{exclude_lite} = 1 if $exclude_lite;

    return %filter;
}

sub ExecuteCollectionQuery {
    my ($self, $name, $query, @values) = @_;

    my $uid_collection = $self->dbaf->uid_collection;
    my $slaves = $uid_collection->slaves;

    my $log_format = 'shards query: name=%s group=%s slave=%s';

    my @result;

    for my $slave (@$slaves) {
        my @log_values = ($name, $slave->group_name, $slave->get_dsn);

        my $started = [gettimeofday];
        ADM::Logs::DeBug(sprintf "$log_format started", @log_values);

        my $sth = eval { $slave->execute($query, @values) };
        if ($@) {
            ADM::Logs::DbErr(sprintf "$log_format failed error=%s", @log_values, $@);
            die $@;
        }

        push @result, $sth;

        my $elapsed = tv_interval $started;
        ADM::Logs::DeBug(sprintf "$log_format finished elapsed=%.2f", @log_values, $elapsed);
    }

    return \@result;
}

sub ExecuteCentralQuery {
    my ($self, $name, $query, @values) = @_;

    my $central_slave = $self->dbaf->central_group->slave;

    my $log_format = 'central query: name=%s';
    my @log_values = ($name);

    my $started = [gettimeofday];
    ADM::Logs::DeBug(sprintf "$log_format started", @log_values);

    my $sth = eval { $central_slave->execute($query, @values) };
    if ($@) {
        ADM::Logs::DbErr(sprintf "$log_format failed error=%s", @log_values, $@);
        die $@;
    }

    my $elapsed = tv_interval $started;
    ADM::Logs::DeBug(sprintf "$log_format finished elapsed=%.2f", @log_values, $elapsed);

    return $sth;
}

sub GetPeopleBorn {
    my ($self, $day) = @_;

    return $self->FindAccountsByAttributes(birthday => $day, no_limiting => 1);
}

sub GetHostMailboxes {
    my ($self, $hostid, $is_pdd) = @_;

    my %args = (
        mail_hostid => $hostid,
        no_limiting => 1,
    );

    if ($is_pdd) {
        $args{only_pdd} = 1;
    }
    else {
        $args{exclude_pdd} = 1;
    }

    my $uids = $self->FindAccountsByAttributes(%args);

    return $uids;
}

sub FindAccountsByAttributes {
    my ($self, %filter) = @_;

    my $shard_searcher = ADM::Core::ShardSearcher->new;
    $shard_searcher->compile(%filter);

    return
      unless $shard_searcher->query;

    my $sths = $self->ExecuteCollectionQuery('find_accounts_by_attributes', $shard_searcher->query, @{ $shard_searcher->values });
    my $result = $self->GetUidsFromSths($sths);

    return $result;
}

sub FindAccountsByActiveAliases  { shift->_FindAccountsByAliases('aliases',         @_) }
sub FindAccountsByRemovedAliases { shift->_FindAccountsByAliases('removed_aliases', @_) }

sub _FindAccountsByAliases {
    my ($self, $table, %filter) = @_;

    die "unknown table $table"
      unless $table =~ /^ (?:removed_)? aliases $/x;

    our $need_like = 0;

    my $username = $filter{username};
    if ($username =~ /\*/) {
        $username =~ s/[_%]/\\$1/g;
        $username =~ s/\*/%/g;
        $username =~ s/\?/_/g;
        $need_like = 1;
    }

    my $normalized_username = $username;
    $normalized_username =~ tr/./-/;

    my $pdd_domain_part;

    if ($filter{pdd_domain_id} or $filter{pdd_domain}) {
        $pdd_domain_part
          = $table eq 'aliases'         ?          $filter{pdd_domain_id}
          : $table eq 'removed_aliases' ? to_puny( $filter{pdd_domain}    )
          :                               die "unknown table $table";
    }

    our @where_conditions = ();
    our @values = ();

    sub add_row {
        my ($alias, $value) = @_;

        my $aliases
          = ref $alias eq 'ARRAY'
          ?  $alias
          : [$alias];

        my @types = map { Database::Converter->get_alias_type_by_name($_) } @$aliases; 
        my $types_number = scalar @types;

        my $where_condition;
        $where_condition .= '(';
        $where_condition .= 'type ';
        $where_condition
          .= $types_number == 1
           ? '= ?'
           : 'IN (' . join(',', ('?') x $types_number) . ')'
           ;
        $where_condition .= ' AND value ';
        $where_condition .= $need_like ? 'LIKE ?' : '= ?';
        $where_condition .= ')';

        push @where_conditions, $where_condition;
        push @values, @types, $value;
    };

    if (length $username) {

        # При наличии доменной части в логине, ищем только как pdd, lite и altdomain
        if ($filter{lite_domain} or $pdd_domain_part or $filter{external_domain_id}) {

            if ($pdd_domain_part and not $filter{exclude_pdd}) {
                add_row(['pdd', 'pddalias'], join '/', $pdd_domain_part, $username);
            }

            if ($filter{lite_domain} and not $filter{exclude_lite}) {
                add_row('lite', join '@', $username, $filter{lite_domain});
            }

            if ($filter{external_domain_id}) {
                add_row('altdomain', join '/', $filter{external_domain_id}, $normalized_username);
            }

        }

        # При отсутствии доменной части (в том числе, если логин ввели с доменом яндекса), ищем в обычных алиасах
        else {

            if ($normalized_username =~ /^uid-/) {
                add_row('social', $normalized_username);
            }

            elsif ($normalized_username =~ /^phne-/) {
                add_row('phonish', $normalized_username);
            }

            elsif ($normalized_username =~ /^[%_\d]+$/) {
                add_row('phonenumber', $normalized_username);
            }

            else {
                add_row([qw/portal mail narodmail narod/], $normalized_username);
            }
        }

    }
    else {

        # При наличии доменной части, но отсутствии username, ищем все аккаунты на указанном ПДД-домене
        if ($pdd_domain_part and not $filter{exclude_pdd}) {
            $need_like = 1;
            $username = '%';
            add_row(['pdd', 'pddalias'], join '/', $pdd_domain_part, $username);
        }
    }

    return
      unless @where_conditions;

    my $where_conditions = join "\n   OR ", @where_conditions;
    $where_conditions = "\n      $where_conditions";

    # Дополнительный фильтр по ПДД-шным uid'ам, т.к. в будущем у ПДД-шников могут быть всякие разные алиасы
    if ($filter{exclude_pdd}) {
        $where_conditions = "($where_conditions\n)\n  AND uid < 1130000000000000";
    }

    my $limit = $ADM::Utils::FOUND_UIDS_MAX_LIMIT + 1;
    my $query = "SELECT uid, value\nFROM $table\nWHERE $where_conditions\nGROUP BY uid LIMIT $limit";

    my $sth = $self->ExecuteCentralQuery("find_accounts_by_$table", $query, @values);
    my $result = $sth->fetchall_arrayref;

    return $result;
}

sub GetUidsFromSths {
    my ($self, $sths) = @_;

    my @result;

    for my $sth (@$sths) {
        push @result, map { $_->[0] } @{ $sth->fetchall_arrayref };
    }

    return \@result;
}

sub GetAccountsByGroups {
    my ($self, $uids) = @_;

    my $group_max_size = $admin::Conf->GetVal('blackbox_group_max_size');
    my $uids_number    = scalar @$uids;
    my $groups_number  = ceil($uids_number / $group_max_size);

    my $result = [];

    for my $i (0 .. $groups_number - 1) {
        my $first_index = $i * $group_max_size;
        my $last_index  = $first_index + $group_max_size - 1;
        $last_index = $uids_number - 1
          if $last_index >= $uids_number;

        ADM::Logs::DeBug(
            sprintf 'GetManyAccounts: %d/%d [%d..%d] (group_max_size=%d uids_number=%d)',
            $i + 1, $groups_number, $first_index, $last_index, $group_max_size, $uids_number,
        );

        my @group_uids    = @$uids[$first_index .. $last_index];
        my $accounts_uids = $self->dbaf->GetAccountsExceptionable(\@group_uids);

        push @$result, @$accounts_uids;
    }

    return $result;
}

1;
