package ADM::Core::ShardSearcher;

use common::sense;

use Data::Dumper;
use List::Util 'first';

use ADM::Utils;
use Database::Converter;

my %FILTER_TO_ATTRIBUTE = (
    only_enabled      => Database::Converter->get_attribute_type_by_name('account.is_disabled'),
    registered_after  => Database::Converter->get_attribute_type_by_name('account.registration_datetime'),
    registered_before => Database::Converter->get_attribute_type_by_name('account.registration_datetime'),
    firstname         => Database::Converter->get_attribute_type_by_name('person.firstname'),
    lastname          => Database::Converter->get_attribute_type_by_name('person.lastname'),
    mail_hostid       => Database::Converter->get_attribute_type_by_name('subscription.mail.host_id'),
    karma             => Database::Converter->get_attribute_type_by_name('karma.value'),
    birthday          => 1001,
    birthyear         => 1002,
);

my @PRIMARY_FILTERS = qw/firstname lastname mail_hostid birthday birthyear/;

my $TABLE        = 'searchable_attributes';
my $TABLE0_ALIAS = 't0';

use Class::XSAccessor {
    constructor => 'new',
    accessors   => [qw/factors query values/],
};

sub compile {
    my $self = shift;
    my %filter = @_;

    $self->factors([]);
    $self->build_factors(%filter);
    $self->build_query(%filter);

    return 1;
}

sub build_factors {
    my $self = shift;
    my %filter = @_;

    # Простые, основные фильтры
    for my $filter_name (@PRIMARY_FILTERS) {
        my $filter_value   = $filter{$filter_name};
        my $attribute_type = $self->get_attribute_type($filter_name);

        next unless length $filter_value;

        my $factor = $self->create_new_factor; 

        my $value_condition = '= ?';

        # Для имени и фамилии можно указывать маску
        if ($filter_name eq 'firstname' or $filter_name eq 'lastname') {
            if ($filter_value =~ /\*/) {
                $filter_value =~ s/[_%]/\\$1/g;
                $filter_value =~ s/\*/%/g;
                $filter_value =~ s/\?/_/g;
                $value_condition = 'LIKE ?';
            }
        }

        $factor->set_simple_type($value_condition, $attribute_type, $filter_value);
    }

    # Фильтр по массиву карм, может содержать значение 0, усложняющее запрос созданием LEFT JOIN фактора,
    # т.к. при нулевой карме атрибут просто отсутствует у аккаунта
    if ($filter{karma}) {
        my $attribute_type = $self->get_attribute_type('karma');

        my $karma         = $filter{karma};
        my @zero_karma    = grep { $_ == 0 } @$karma;
        my @nonzero_karma = grep { $_ != 0 } @$karma;

        if (@zero_karma) {
            my $factor = $self->create_new_factor; 

            $factor->set_left_join_type($attribute_type);

            $factor->add_where_conditions('uid IS NULL OR :table.value = ?');
            $factor->add_where_values(@zero_karma);
        }

        if (@nonzero_karma) {
            my $factor = $self->create_new_factor; 

            my $value_condition = $self->build_in_condition(\@nonzero_karma);

            $factor->set_simple_type($value_condition, $attribute_type, @nonzero_karma);
        }
    }

    # Фильтр по времени регистрации, может ограничивать сверху, снизу или с двух сторон
    if ($filter{registered_after} or $filter{registered_before}) {
        my $factor = $self->create_new_factor; 

        my $attribute_type = $self->get_attribute_type('registered_after');

        $factor->join_type('JOIN');
        $factor->add_where_conditions('type = ?');
        $factor->add_where_values($attribute_type);

        if ($filter{registered_after}) {
            $factor->add_where_conditions('value >= ?');
            $factor->add_where_values($filter{registered_after});
        }

        if ($filter{registered_before}) {
            $factor->add_where_conditions('value <= ?');
            $factor->add_where_values($filter{registered_before});
        }
    }

    # Фильтр по незаблокированности аккаунта, создаёт LEFT JOIN фактор, т.к. атрибут отсутствует, если аккаунт не заблокирован
    if ($filter{only_enabled}) {
        my $factor = $self->create_new_factor; 

        my $attribute_type = $self->get_attribute_type('only_enabled');

        $factor->set_left_join_type($attribute_type);

        $factor->add_where_conditions('uid IS NULL');
    }

    return;
}

sub build_query {
    my $self = shift;
    my %filter = @_;

    my $factors = $self->factors;

    return unless @$factors;

    # Делим факторы на два списка: основной список с обычным JOIN'ами и дополнительный с LEFT JOIN'ами
    my @inner_join_factors = grep { $_->join_type eq 'JOIN'      } @$factors;
    my @left_join_factors  = grep { $_->join_type eq 'LEFT JOIN' } @$factors;

    # Если нет ни одного фактора с обычным JOIN'ом, то все факторы с LEFT JOIN'ами не к чему присоединять.
    # Поэтому, создаём специальный null-фактор, к которому будут присоединятся все LEFT JOIN'ы.
    my $null_factor
      = @inner_join_factors
      ? undef
      : $self->create_new_factor;

    # Сортируем собранные факторы в определённом порядке:
    #   1. Если есть специальный null-фактор - он всегда идёт первым.
    #   2. Дальше имеющиеся обычные JOIN'ы.
    #   3. И завершаем LEFT JOIN'ами.
    # Это нужно и для консистентности текста запросов, и для удобства, и вообще - для их работоспособности.
    my @sorted_factors;
    if ($null_factor) {
        push @sorted_factors, $null_factor;
    }
    push @sorted_factors, @inner_join_factors, @left_join_factors;

    my @table_references;
    my @join_values;
    my @where_conditions;
    my @where_values;
    my $index = 0;

    # Дополнительный фильтр по ранее найденным uid'ам другими способами
    if ($filter{uids} and @{ $filter{uids} }) {
        my $first_factor = @sorted_factors[0];

        my $uid_condition = $self->build_in_condition($filter{uids});

        $first_factor->add_where_conditions("uid $uid_condition");
        $first_factor->add_where_values(@{ $filter{uids} });
    }

    # Теперь обрабатываем все собранные факторы по порядку
    for my $factor (@sorted_factors) {
        $factor->table_index($index++);

        if ($factor->join_type eq 'JOIN') {
            my $pdd_min_uid = $ADM::Utils::PDD_MIN_UID;

            # Для специальных ПДД/неПДД ограничений к каждому добавляем доп. условие
            $factor->add_where_conditions("uid >= $pdd_min_uid")
              if $filter{only_pdd};
            $factor->add_where_conditions("uid < $pdd_min_uid")
              if $filter{exclude_pdd};
        }

        # И собираем в нужном порядке все данные факторов для последующей сборки запроса
        push @table_references, $factor->compiled_table_reference;
        push @join_values,      @{ $factor->join_values };

        push @where_conditions, $factor->compiled_where_condition || ();
        push @where_values,     @{ $factor->where_values };
    }

    # При наличии null-фактора поиск может выдать неуникальные uid'ы, т.к. никакого ограничения/условия на
    # эту таблицу t0 не указывается. Поэтому добавляем модификатор DISTINCT для выборки.
    my $select_field     = $null_factor ? "DISTINCT($TABLE0_ALIAS.uid)" : "$TABLE0_ALIAS.uid";

    # Собираем запрос
    my @values           = (@join_values, @where_values);
    my $table_references = join "\n  ",     @table_references;
    my $where_conditions = join "\n  AND ", @where_conditions;
    my $query            = "SELECT $select_field\nFROM $table_references\nWHERE $where_conditions";
    
    unless ($filter{no_limiting}) {
        my $limit = $ADM::Utils::FOUND_UIDS_MAX_LIMIT + 1;
        $query .= "\nLIMIT $limit";
    }

    ADM::Logs::DeBug(Dumper($query, \@values));

    $self->query($query);
    $self->values(\@values);

    return;
}

sub create_new_factor {
    my $self = shift;

    my $factor = ADM::Core::ShardSearcher::Factor->new;
    $factor->table($TABLE);

    push @{ $self->factors }, $factor;

    return $factor;
}

sub get_attribute_type { $FILTER_TO_ATTRIBUTE{ $_[1] } }

sub build_in_condition {
    my $self = shift;
    my ($array) = @_;

    my $result
      = 'IN ('
      . join(',', ('?') x @$array)
      . ')';

    return $result;
}


package ADM::Core::ShardSearcher::Factor;

use common::sense;

my $TABLE_ALIAS_PREFIX = 't';
my $TABLE0_ALIAS       = $TABLE_ALIAS_PREFIX . '0';

use Class::XSAccessor {
    accessors   => [qw/table table_index join_type join_condition join_values where_conditions where_values/],
};

sub new {
    my $class = shift;

    my $self = {
        join_values      => [],
        where_conditions => [],
        where_values     => [],
    };

    bless $self, $class;

    return $self;
}

sub table_alias { $TABLE_ALIAS_PREFIX . shift->table_index }

sub compiled_table_reference {
    my $self = shift;

    my $result;

    # Самый первый фактор записывается без JOIN-выражения
    if ($self->table_index == 0) {
        $result = join ' ', $self->table, $self->table_alias;
    }

    # Все последующие - только с JOIN'ами разного типа
    else {
        my $join_condition
          = $self->join_condition
          ? sprintf("ON %s.uid = $TABLE0_ALIAS.uid AND %s.%s", $self->table_alias, $self->table_alias, $self->join_condition)
          : 'USING (uid)'
          ;
        $result = join ' ', $self->join_type, $self->table, $self->table_alias, $join_condition;
    }

    return $result;
}

sub compiled_where_condition {
    my $self = shift;

    my @items;

    for my $where_condition (@{ $self->where_conditions }) {
        my $item = sprintf '(%s.%s)', $self->table_alias, $where_condition;
        push @items, $item;
    }

    return '' unless @items;

    my $result = join ' AND ', @items;

    # Для сложных условий с OR'ами приходится указывать placeholder имени таблицы рядом с полем
    $result =~ s/:table/$self->table_alias/eg;

    return $result;
}

sub add_where_conditions {
    my $self = shift;
    my (@where_conditions) = @_;

    push @{ $self->where_conditions }, @where_conditions;

    return;
}

sub add_where_values {
    my $self = shift;
    my (@where_values) = @_;

    push @{ $self->where_values }, @where_values;

    return;
}

sub add_join_values {
    my $self = shift;
    my (@join_values) = @_;

    push @{ $self->join_values }, @join_values;

    return;
}

sub set_simple_type {
    my $self = shift;
    my ($value_condition, $attribute_type, @where_values) = @_;

    $self->join_type('JOIN');

    $self->add_where_conditions('type = ?', "value $value_condition");
    $self->add_where_values($attribute_type, @where_values);

    return;
}

sub set_left_join_type {
    my $self = shift;
    my ($attribute_type) = @_;

    $self->join_type('LEFT JOIN');
    $self->join_condition('type = ?');
    $self->add_join_values($attribute_type);

    return;
}

1;
