#!/usr/bin/env perl

use lib::abs '.';
use lib::abs '../..';
use lib::abs '../../../../cgi-bin';
use lib::abs '../../../../perl5/lib/perl5';

use Data::Dumper;
use JSON::XS;
use Test::More;

use_ok('ADM::Core::ShardSearcher');

sub build {
    my %filter = @_;

    my $searcher = ADM::Core::ShardSearcher->new;
    $searcher->compile(%filter);

    return $searcher;
}

sub check {
    my ($test_name, $filter, $expected_query, $expected_values) = @_;

    $expected_query =~ s/^\n//m;
    $expected_query =~ s/\n$//m;

    $filter->{no_limiting} = 1;

    my $searcher = build(%$filter);

    my $got_query  = $searcher->query;
    my $got_values = $searcher->values;

    my $filter_json = encode_json $filter;

    local $Test::Builder::Level = $Test::Builder::Level + 1;

    is        $got_query,  $expected_query,  "$test_name; $filter_json; query";
    is_deeply $got_values, $expected_values, "$test_name; $filter_json; values";
}



check(
    'simple inner filter',
    { firstname => 'qwe' },
'
SELECT t0.uid
FROM searchable_attributes t0
WHERE (t0.type = ?) AND (t0.value = ?)
',
    [ 27, 'qwe' ],
);


check(
    'simple like filter firstname',
    { firstname => 'qwe?asd*' },
'
SELECT t0.uid
FROM searchable_attributes t0
WHERE (t0.type = ?) AND (t0.value LIKE ?)
',
    [ 27, 'qwe_asd%' ],
);


check(
    'simple like filter lastname',
    { lastname => 'qwe?asd*' },
'
SELECT t0.uid
FROM searchable_attributes t0
WHERE (t0.type = ?) AND (t0.value LIKE ?)
',
    [ 28, 'qwe_asd%' ],
);


check(
    'simple doubled inner filters',
    { firstname => 'qwe', lastname => 'asd' },
'
SELECT t0.uid
FROM searchable_attributes t0
  JOIN searchable_attributes t1 USING (uid)
WHERE (t0.type = ?) AND (t0.value = ?)
  AND (t1.type = ?) AND (t1.value = ?)
',
    [ 27, 'qwe', 28, 'asd' ],
);


check(
    'karma with nonzero values',
    { karma => [ 85, 100 ] },
'
SELECT t0.uid
FROM searchable_attributes t0
WHERE (t0.type = ?) AND (t0.value IN (?,?))
',
    [ 17, '85', '100' ],
);



check(
    'karma with zero value, with null factor',
    { karma => [ 0 ] },
'
SELECT DISTINCT(t0.uid)
FROM searchable_attributes t0
  LEFT JOIN searchable_attributes t1 ON t1.uid = t0.uid AND t1.type = ?
WHERE (t1.uid IS NULL OR t1.value = ?)
',
    [ 17, '0' ],
);


check(
    'simple registration time filter',
    { registered_after => 1234567890, registered_before => 2345678901 },
'
SELECT t0.uid
FROM searchable_attributes t0
WHERE (t0.type = ?) AND (t0.value >= ?) AND (t0.value <= ?)
',
    [ 1, '1234567890', '2345678901' ],
);


check(
    'only enabled filter, with null factor',
    { only_enabled => 1 },
'
SELECT DISTINCT(t0.uid)
FROM searchable_attributes t0
  LEFT JOIN searchable_attributes t1 ON t1.uid = t0.uid AND t1.type = ?
WHERE (t1.uid IS NULL)
',
    [ 3 ],
);


check(
    'only enabled with simple filter, with no null factor',
    { firstname => 'qwe', only_enabled => 1 },
'
SELECT t0.uid
FROM searchable_attributes t0
  LEFT JOIN searchable_attributes t1 ON t1.uid = t0.uid AND t1.type = ?
WHERE (t0.type = ?) AND (t0.value = ?)
  AND (t1.uid IS NULL)
',
    [ 3, 27, 'qwe' ],
);


check(
    'only pdd additional filter',
    { firstname => 'qwe', only_pdd => 1 },
'
SELECT t0.uid
FROM searchable_attributes t0
WHERE (t0.type = ?) AND (t0.value = ?) AND (t0.uid >= 1130000000000000)
',
    [ 27, 'qwe' ],
);


check(
    'exclude pdd additional filter',
    { firstname => 'qwe', exclude_pdd => 1 },
'
SELECT t0.uid
FROM searchable_attributes t0
WHERE (t0.type = ?) AND (t0.value = ?) AND (t0.uid < 1130000000000000)
',
    [ 27, 'qwe' ],
);


check(
    'exclude pdd additional filter with several filters',
    { firstname => 'qwe', lastname => 'asd', only_enabled => 1, exclude_pdd => 1 },
'
SELECT t0.uid
FROM searchable_attributes t0
  JOIN searchable_attributes t1 USING (uid)
  LEFT JOIN searchable_attributes t2 ON t2.uid = t0.uid AND t2.type = ?
WHERE (t0.type = ?) AND (t0.value = ?) AND (t0.uid < 1130000000000000)
  AND (t1.type = ?) AND (t1.value = ?) AND (t1.uid < 1130000000000000)
  AND (t2.uid IS NULL)
',
    [ 3, 27, 'qwe', 28, 'asd' ],
);


check(
    'with no inner join, only left join, with null factor',
    { karma => [ 0 ], only_enabled => 1 },
'
SELECT DISTINCT(t0.uid)
FROM searchable_attributes t0
  LEFT JOIN searchable_attributes t1 ON t1.uid = t0.uid AND t1.type = ?
  LEFT JOIN searchable_attributes t2 ON t2.uid = t0.uid AND t2.type = ?
WHERE (t1.uid IS NULL OR t1.value = ?)
  AND (t2.uid IS NULL)
',
    [ 17, 3, '0' ],
);


check(
    'with no join and uids list',
    { karma => [ 85 ], uids => [ 123, 456 ], },
'
SELECT t0.uid
FROM searchable_attributes t0
WHERE (t0.type = ?) AND (t0.value IN (?)) AND (t0.uid IN (?,?))
',
    [ 17, '85', 123, 456 ],
);


check(
    'with inner join and uids list',
    { firstname => 'qwe', karma => [ 85 ], uids => [ 123, 456 ], },
'
SELECT t0.uid
FROM searchable_attributes t0
  JOIN searchable_attributes t1 USING (uid)
WHERE (t0.type = ?) AND (t0.value = ?) AND (t0.uid IN (?,?))
  AND (t1.type = ?) AND (t1.value IN (?))
',
    [ 27, 'qwe', 123, 456, 17, '85' ],
);


check(
    'with no inner join and uids list',
    { karma => [ 0 ], uids => [ 123, 456 ], },
'
SELECT DISTINCT(t0.uid)
FROM searchable_attributes t0
  LEFT JOIN searchable_attributes t1 ON t1.uid = t0.uid AND t1.type = ?
WHERE (t0.uid IN (?,?))
  AND (t1.uid IS NULL OR t1.value = ?)
',
    [ 17, 123, 456, '0' ],
);


check(
    'all the filters',
    { firstname => 'qwe', lastname => 'asd', mail_hostid => '1034', birthday => '02-29', birthyear => '1980', karma => [ 0, 85, 100 ], registered_after => 1234567890, registered_before => 2345678901, only_enabled => 1},
'
SELECT t0.uid
FROM searchable_attributes t0
  JOIN searchable_attributes t1 USING (uid)
  JOIN searchable_attributes t2 USING (uid)
  JOIN searchable_attributes t3 USING (uid)
  JOIN searchable_attributes t4 USING (uid)
  JOIN searchable_attributes t5 USING (uid)
  JOIN searchable_attributes t6 USING (uid)
  LEFT JOIN searchable_attributes t7 ON t7.uid = t0.uid AND t7.type = ?
  LEFT JOIN searchable_attributes t8 ON t8.uid = t0.uid AND t8.type = ?
WHERE (t0.type = ?) AND (t0.value = ?)
  AND (t1.type = ?) AND (t1.value = ?)
  AND (t2.type = ?) AND (t2.value = ?)
  AND (t3.type = ?) AND (t3.value = ?)
  AND (t4.type = ?) AND (t4.value = ?)
  AND (t5.type = ?) AND (t5.value IN (?,?))
  AND (t6.type = ?) AND (t6.value >= ?) AND (t6.value <= ?)
  AND (t7.uid IS NULL OR t7.value = ?)
  AND (t8.uid IS NULL)
',
    [ 17, 3, 27, 'qwe', 28, 'asd', 38, '1034', 1001 => '02-29', 1002 => '1980', 17, '85', '100', 1, '1234567890', '2345678901', '0'],
);


done_testing;

