package ADM::Engine::Api;

use common::sense;

use Data::Dumper;
use JSON::XS;
use Tie::IxHash;
use Time::HiRes;
use URI;
use URI::QueryParam;

use DBAF;
use Passport;
use IP;
use Global;
use Common::Puny 'to_puny';

my $JSON_ENCODER = JSON::XS->new->utf8(1)->pretty->canonical;
my $JSON_UGLY_ENCODER = JSON::XS->new->utf8(1)->canonical;
our $REQ = undef;
our $REAL_IP = undef;
our $URI = undef;

sub handler {
    my $req = shift;

    $REQ = $req;
    $REAL_IP = undef;
    $REAL_IP = $REQ->header_in('X-Real-IP') || $REQ->connection->remote_ip;
    $Input::Input = { real_ip => IP->new($REAL_IP) };

    my %controllers = (
        userinfo        => \&userinfo,
        userinfoft      => \&userinfoft,
        reguserinfo     => \&reguserinfo,
        auth            => \&auth,
        event           => \&event,
        domain          => \&domain,
        makesession     => \&makesession,
        born            => \&born,
        host_mailboxes  => \&host_mailboxes,
        mailbox_history => \&mailbox_history,
        create_restoration_link => \&create_restoration_link,
    );

    my $uri = $REQ->uri;    
    return 404 unless $uri =~ m{^/api/([^/]+)};
    if( $REQ->method()  eq 'POST' ) {
        $URI = URI->new($uri.'?'.$REQ->content);    
    } else {
        $URI = URI->new($uri.'?'.$REQ->query_string);
    }

    my $controller_name = $1;
    return 404
      unless exists $controllers{$controller_name};

    return 403
      unless ADM::Utils::TestIpAllowed("api$controller_name", IP->new($REAL_IP));

    return $controllers{$controller_name}->($REQ);
}

sub render {
    my ($data, $status) = @_;

    $status ||= 200;

    my $json = $JSON_ENCODER->encode($data);

    $REQ->content_type('application/json');
    $REQ->status($status);
    $REQ->send_http_header;
    $REQ->print($json, "\n");

    $REQ = undef;

    return 200;
}

sub render_ugly {
    my ($data, $status) = @_;

    $status ||= 200;

    my $json = $JSON_UGLY_ENCODER->encode($data);

    $REQ->content_type('application/json');
    $REQ->status($status);
    $REQ->send_http_header;
    $REQ->print($json, "\n");

    $REQ = undef;

    return 200;
}

sub render_error     { render({ error => shift }, shift || 400) }
sub render_exception { render_error('internal-exception', 500) }

sub get_uid {
    my %args = $REQ->args;

    my $uid = $args{uid};
    my $email = $args{email};

    return render_error('email-or-uid-required')
      unless $uid or $email;

    if ($uid) {
        return render_error('uid-invalid') unless $uid =~ /^\d+$/;
    } else {
        my $acd;
        my $fail = get_acd($acd);
        return $fail if $fail;
        my $account = eval { $acd->dbaf->GetAccountByUidExceptionable($email) };
        return render_exception() if $@;
        $uid = $account->uid
          if $account;
        return render_error('user-not-found', 404) unless $uid;
    }
    
    $_[0] = $uid;

    return 0;
}

sub get_acd {
    my $acd = ADM::Core::DB->new;
    return render_exception() unless $acd;
    $_[0] = $acd;
    return 0;
}


=head2 makesession

Изготовляем тестовые куки с заданными параметрами

=cut
sub makesession {
    my ($uidstr, $created, $age, $type, $policy, $ip, $tld, $betatester, $yandexoid, $get_safe) 
        = map { my $v=$URI->query_param($_); defined($v) ? $v : undef } 
            qw(uids created age type policy userip tld betatester yandexoid get_safe);

    return render_error('uids argument not set') unless defined($uidstr);
    my @uids = grep { $_ } split /\D+/, $uidstr;
    return render_error('uids argument not set') unless @uids;
    
    my $ret = { "status" => "OK" };

    my $acd = ADM::Core::DB->new;
    my $dbaf = $acd->dbaf;

    $ip = $REAL_IP unless defined($ip);

    $Input::AllVals = {};
    $Input::Input   = { real_ip => IP->new($ip) };

    my $key_space = "yandex." . ($tld || 'ru');

    my $authz = Passport::CreateAuthorization($dbaf);

    render_error('db error')
      unless $acd and $dbaf and $authz;

    for my $uid (@uids) {
        my $account = eval { $acd->dbaf->GetAccountByUidExceptionable($uid) };

        return render_exception()
          if $@;

        unless ($account and $account->uid) {
            $ret->{errors}->{$uid} = 'user-not-found';
            next;
        }

        my $sessionid = $authz->create_sessionid_safe($uid, key_space => $key_space);

        if (not $policy) {
            $sessionid->be_sessional;
        }
        elsif ($policy =~ /^(?:short|longer|sessional)$/) {
            $sessionid->be_sessional if $policy eq 'sessional';
            $sessionid->be_short     if $policy eq 'short';
            $sessionid->be_longer    if $policy eq 'longer';
        }
        else {
            return render_error("invalid policy: $policy; accepted: short|medium|long|longer|sessional");
        }

        if (not $type) {
            $sessionid->be_normal;
        }
        elsif ($type =~ /^(?:normal|lite)$/) {
            $sessionid->be_normal if $type eq 'normal';
            $sessionid->be_lite   if $type eq 'lite';
        }
        else {
            return render_error("invalid type: $type; accepted: normal|lite|pdd");
        }

        if ($created) {
            $sessionid->created($created);
        } elsif (defined $age) {
            $sessionid->age($age);
        }

        $sessionid->is_stressed(1);

        $sessionid->is_betatester($betatester ? 1 : 0) if defined $betatester;
        $sessionid->is_yandexoid ($yandexoid  ? 1 : 0) if defined $yandexoid;

        my $session = eval { $authz->serialize_sessionid };
        if ($@) {
            ADM::Logs::DeBug($@);
        }
        ADM::Logs::Test('makesession '.Dumper($sessionid) );
        $ret->{sessions}->{$uid} = $session;
        if ($get_safe) {
            $ret->{sslsessions}->{$uid} = $sessionid->cookie2->value;
        }
    }

    return render($ret);
}

sub userinfo {
    my ($uid, $acd);

    my $fail;
    $fail = get_uid($uid);
    return $fail if $fail;
    $fail = get_acd($acd);
    return $fail if $fail;

    my $account = $acd->dbaf->GetAccountWithPhones($uid);

    return render_error('user-not-found', 404)
      unless $account and $account->uid;

    my $phones_view = [];
    for my $phone (@{ $account->phones->list }) {
        push @$phones_view, {
            number => $phone->number,
            is_confirmed => \($phone->is_confirmed),
        };
    }

    my $emails_view = [];
    for my $email (@{ $account->emails->list }) {
        push @$emails_view, {
            address => $email->address,
            is_confirmed => \($email->is_confirmed),
            is_native => \($email->is_native),
        };
    }

    my $person = $account->person;

    my $result_view = { userinfo => {
        uid           => $account->uid + 0,
        login         => $account->machine_readable_login,
        registered_at => $account->registration_datetime,
        is_enabled    => \ $account->is_enabled,
        firstname     => $person->firstname,
        lastname      => $person->lastname,
        birthday      => $person->birthday,
        sex           => ( $person->sex ? $person->sex == 1 ? 'm' : $person->sex == 2 ? 'f' : undef : undef),
        city          => $person->city,
        country       => $person->country,
        phones        => $phones_view,
        emails        => $emails_view,
    }};

    return render($result_view);
}

sub userinfoft {
    my ($acd);

    my %args  = $REQ->args;
    my $login = $args{login};
    my $email = $args{email};
    my $input_uid = $args{uid};

    my $fail;
    $fail = get_acd($acd);
    return $fail if $fail;

    $login = $email || $login;

    my $uids;

    if ($input_uid) {
        $uids = [$input_uid];
    }
    else {
        my %filter = eval { $acd->ConvertRequestParamsToFilter({ login => $login }) };

        return render_exception()
          unless %filter;

        if ($filter{pdd_domain} and not $filter{exclude_pdd}) {
            my $domains = $acd->dbaf->GetHostedDomains(domain_name => to_puny($filter{pdd_domain}));

            return render_exception()
              unless $domains;

            $filter{pdd_domain_id} = $domains->[0]{domain_id}
              if $domains->[0];
        }

        my $active_rows = eval { $acd->FindAccountsByActiveAliases(%filter) };
        my $active_uids = [ map { $_->[0] } @$active_rows ];

        if ($@) {
            ADM::Logs::IntErr('FindAccountsByActiveAliases has been failed: ' . $@);
            return render_exception();
        }

        my $removed_rows = eval { $acd->FindAccountsByRemovedAliases(%filter) };
        my $removed_uids = [ map { $_->[0] } @$removed_rows ];

        if ($@) {
            ADM::Logs::IntErr('FindAccountsByRemovedAliases has been failed: ' . $@);
            return render_exception();
        }

        $uids = ADM::Utils::union($active_uids, $removed_uids);
    }

    my @userinfos_ft;

    for my $uid (@$uids) {
        my $userinfo_ft = ADM::Requester::GetHistoryDb3UserinfoFT(uid => $uid);

        unless ($userinfo_ft) {
            ADM::Logs::IntErr('GetHistoryDb3UserinfoFT has been failed' );
            return render_exception();
        }

        next unless %$userinfo_ft;

        push @userinfos_ft, $userinfo_ft;
    }

    return render_error('user-not-found', 404)
      unless @userinfos_ft;

    my $users_view = [];
    my $result_view = { users => $users_view };

    for my $userinfo_ft (@userinfos_ft) {
        push @$users_view, { userinfoft => {
            uid           => $userinfo_ft->{uid} + 0,
            login         => $userinfo_ft->{login},
            firstname     => $userinfo_ft->{firstname},
            lastname      => $userinfo_ft->{lastname},
            registered_at => $userinfo_ft->{dt},
            ip            => $userinfo_ft->{ip},
        }};
    }

    return render($result_view);
}

sub auth {
    my ($uid, $acd);

    my $fail;
    $fail = get_uid($uid);
    return $fail if $fail;
    $fail = get_acd($acd);
    return $fail if $fail;

    my %args = $REQ->args;
    my ($from, $to) = @args{qw(from to)};

    return render_error('from-invalid')
      if $from and not $from =~ /^\d\d\d\d-\d\d-\d\dT\d\d:\d\d:\d\d$/;
    return render_error('to-invalid')
      if $to and not $to =~ /^\d\d\d\d-\d\d-\d\dT\d\d:\d\d:\d\d$/;

    $from =~ s/T/ /;
    $to   =~ s/T/ /;

    my $begin = Global::DatetimeToTimestamp($from || $admin::Conf->GetVal('historydb2_start_date'));
    my $end   = $to ? Global::DatetimeToTimestamp($to) : time;

    my $statuses = $args{statuses};

    my %filter = (
        uid    => $uid,
        from   => $begin,
        to     => $end,
    );

    if ($statuses) {
        $filter{status} = [ split /[ ,]/, $statuses ];
    }
    else {
        $filter{source} = 'bb';
        $filter{status} = [qw/successful disabled failed blocked/];
    }

    my $historydb3_auths = ADM::Utils::GetHistoryDb3Auths(%filter);

    return render_exception()
      unless $historydb3_auths;

    my $type_name_to_id   = $admin::Conf->GetHCVal('HistoryAuthTypeNameToId');
    my $status_name_to_id = $admin::Conf->GetHCVal('HistoryAuthStatusNameToId');

    my $auths = [];

    for my $auth (@$historydb3_auths) {
        push @$auths, {
            time   => $auth->{dt},
            type   => $type_name_to_id->{ $auth->{type_name} } + 0,
            status => $status_name_to_id->{ $auth->{status_name} } + 0,
            ip     => $auth->{ip},
            proxy  => undef,
        };
    }

    my $result = { auth => $auths };

    return render($result);
}

sub event {
    my ($uid, $acd);

    my $fail;
    $fail = get_uid($uid);
    return $fail if $fail;
    $fail = get_acd($acd);
    return $fail if $fail;

    my %args = $REQ->args;
    my ($from, $to) = @args{qw(from to)};

    return render_error('from-invalid')
      if $from and not $from =~ /^\d\d\d\d-\d\d-\d\dT\d\d:\d\d:\d\d$/;
    return render_error('to-invalid')
      if $to and not $to =~ /^\d\d\d\d-\d\d-\d\dT\d\d:\d\d:\d\d$/;

    $from =~ s/T/ /;
    $to   =~ s/T/ /;

    my $begin = Global::DatetimeToTimestamp($from || $admin::Conf->GetVal('historydb2_start_date'));
    my $end   = $to ? Global::DatetimeToTimestamp($to) : time;

    my $historydb3_events = ADM::Requester::GetHistoryDb3Events(
        uid  => $uid,
        from => $begin,
        to   => $end,
    );

    return render_exception()
      unless $historydb3_events;

    my $events = [];

    for my $event (@$historydb3_events) {
        next if $event->{name} eq 'userinfo_ft';

        push @$events, {
            time  => $event->{dt},
            event => $event->{name},
            value => $event->{value},
            ip    => $event->{ip},
            proxy => undef,
        };
    }

    my $result = { event => $events };

    return render($result);
}

sub domain {
    my $acd;

    my $fail;
    $fail = get_acd($acd);
    return $fail if $fail;

    my %args = $REQ->args;
    my $domain_name = $args{domain};

    return render_error('domain-empty')
      unless $domain_name;

    my $domain_id = $acd->dbaf->GetDomainId($domain_name);

    return render_error('domain-not-found', 404)
      unless $domain_id;

    my $result = $acd->dbaf->FindAccounts(domain_id => $domain_id, all => 1);
    my $accounts = $result->{accounts};
    my @domain_users = map +{ uid => $_ . '' + 0 }, @$accounts;

    my $result = {
        domain => {
            domid => $domain_id,
        },
        users => \@domain_users,
    };

    return render($result);
}

sub born {
    my ($uid, $acd);

    my %args = $REQ->args;
    my $day  = $args{day};

    return render_error('day-empty')
      unless $day;

    return render_error('day-invalid')
      unless $day =~ /^\d\d-\d\d$/;

    my $fail;
    $fail = get_acd($acd);
    return $fail if $fail;

    my $started = Time::HiRes::time;
    my $uids;

    eval {
        $uids = $acd->GetPeopleBorn($day);
    };

    return render_exception() if $@;

    $_ = $_ . '' + 0
      for @$uids;

    my $finished = Time::HiRes::time;
    my $elapsed  = $finished - $started;

    my $result = {
        started  => $started,
        finished => $finished,
        elapsed  => $elapsed,

        day      => $day,
        number   => scalar @$uids,
        uids     => $uids,
    };

    return render_ugly($result);
}

my %MAILBOX_ACTION_BY_NAME = (
    'mail.add' => 'ins',
    'mail.upd' => 'upd',
    'mail.rm'  => 'del',
);

sub mailbox_history {
    my %args = $REQ->args;

    my $suid = $args{suid};

    return render_error('suid-invalid')
      unless $suid =~ /^\d+$/;

    my $acd;

    my $fail;
    $fail = get_acd($acd);
    return $fail if $fail;

    my $is_pdd = $suid >= 1130000000000000 ? 1 : 0;
    my $hosts = $acd->dbaf->GetMailHosts($is_pdd);

    return render_exception()
      unless $hosts;

    my $events = ADM::Requester::GetHistoryDb3MailboxHistory(suid => $suid);

    return render_exception()
      unless $events;

    my $result = {
        suid => $suid + 0,
    };

    my $prev_event;
    for my $event (@$events) {
        my $action = $MAILBOX_ACTION_BY_NAME{ $event->{name} };

        my %item;
        tie %item, 'Tie::IxHash';

        my $host  = $hosts->{by_host_id}{ $event->{mail_host_id} };

        %item = (
            ts     => $event->{timestamp} + 0,
            dt     => Global::TimestampToLocalDatetime($event->{timestamp}),
            action => $action,
        );

        if ($action eq 'upd' and $prev_event and $prev_event->{mail_host_id}) {
            my $old_host = $hosts->{by_host_id}{ $prev_event->{mail_host_id} };
            $item{old_hostid} = $prev_event->{mail_host_id} + 0;
            $item{old_dbid}   = $old_host ? $old_host->{db_id} : undef;
        }

        $item{hostid} = $event->{mail_host_id} + 0;
        $item{dbid}   = $host ? $host->{db_id} : undef;

        push @{ $result->{history} }, \%item;

        $prev_event = $event;
    }

    return render($result);
}

sub host_mailboxes {
    my $acd;

    my %args = $REQ->args;

    my $db_id = $args{db_id};
    my $scope = $args{scope};

    return render_error('db_id-empty')
      unless $db_id;

    return render_error('scope-invalid')
      unless $scope =~ /^(?:yandex|pdd)$/;

    my $fail;
    $fail = get_acd($acd);
    return $fail if $fail;

    my $is_pdd = $scope eq 'pdd' ? 1 : 0;
    my $hosts = $acd->dbaf->GetMailHosts($is_pdd);

    return render_exception()
      unless $hosts;

    my $hosts = $hosts->{hosts_by_db_id}{$db_id};

    return render_error('db_id-unknown')
      unless $hosts;

    my $host_ids = join ',', map { $_->{host_id} } @$hosts;

    my $time = [Time::HiRes::gettimeofday];
    my $task_id = sprintf '%s-%d-%06d', 'host-mailboxes', @$time;

    my $tasks_path = $admin::Conf->GetVal('background_tasks_path');
    my $task_input_filename = "$tasks_path/$task_id.input";
    my $task_log_filename   = "$tasks_path/$task_id.log";

    my ($task_input_file, $task_log_file);

    unless (open $task_input_file, '>', $task_input_filename) {
        ADM::Logs::IntErr($!);
        return render_exception();
    }

    unless (open $task_log_file, '>', $task_log_filename) {
        ADM::Logs::IntErr($!);
        return render_exception();
    }

    print $task_input_file $JSON_UGLY_ENCODER->encode({ is_pdd => $is_pdd, host_ids => $host_ids, db_id => $db_id, scope => $scope });
    print $task_log_file   scalar(localtime time), " Created\n";

    close $task_log_file;
    close $task_input_file;

    my $base_url = $admin::Conf->GetVal('base_url');
    my $this_host_id = $admin::Conf->GetVal('this_host_id');
    my $task_base_url = "$base_url/tasks/$this_host_id/$task_id";

    my $result = {
        created    => $time->[0],
        host_ids   => $host_ids,
        db_id      => $db_id,
        scope      => $scope,
        is_pdd     => $is_pdd,
        log_url    => "$task_base_url.log",
        result_url => "$task_base_url.result.tar.gz",
    };

    return render($result);
}

sub create_restoration_link {
    my $acd;
 
    my %args = $REQ->args;

    my $uid = $args{uid};

    my $fail;
    $fail = get_uid($uid);
    return $fail if $fail;
    $fail = get_acd($acd);
    return $fail if $fail;

    my $type        = $args{type} || 1;
    my $admin_login = $args{admin_login};
    my $ip          = $args{ip};

    return render_error('uid-invalid')
      unless $uid;

    return render_error('admin_login-invalid')
      unless $admin_login;

    return render_error('ip-invalid')
      unless $ip;

    return render_error('type-invalid')
      unless $type == 1;

    my $email = 'stub@yandex';

    my $account = $acd->dbaf->GetAccount($uid);

    return render_error('user-not-found')
      unless $account and $account->uid;

    return render_error('type-denied-if-2fa')
      if $type == 2 and $account->password->is_2fa_enabled;

    my $key = $acd->dbaf->CreateRestorationEmailKey($uid, $email, $type);
    my $passport_url = $admin::Conf->GetVal('passport_url');
    $passport_url =~ s/\.ru$/.com.tr/ if $account->person->country =~ /^tr$/i;
    $passport_url =~ s/\.ru$/.com/    if $account->person->country =~ /^u[sk]$/i;
    my $url = "$passport_url/passport?mode=restore&key=$key";

    return render_error("link-creating-failed")
      unless $key;

    my $changes = {
        key_generated => scalar localtime,
        admin_login   => $admin_login,
    };

    if ($type == 1) {
        return render_error("account-disabling-failed")
          unless $acd->dbaf->SetAccountIsEnabled($uid, 0);
        $changes->{ena} = 0;
    }

    my $opdata = {
        uid     => $uid,
        ip_from => $ip,
        optype  => 'mod',
    };
    ADM::Logs::LogChanges($opdata, $changes);

    my $result = {
        uid  => $uid,
        type => $type,
        url  => $url,
    };

    return render($result);
}

1;
