package ADM::Engine::Users;

use strict;
use utf8;
use open qw(:std :utf8);

use Apache::Cookie ();
use DateTime;
use Digest::SHA 'sha256';
use List::Util 'first';
use List::MoreUtils 'uniq';
use JSON::XS;
use POSIX 'ceil';

use Date::Calc;
use Data::Dumper;
use ADM::Logs;
use ADM::FakeInput;
use Global;
use Common::Puny 'to_puny';
use Common::Csrf;
use ADM::SodiumSeal;

sub RedirBack {
    my ($req, $url) = @_;

    my $referer = $req->header_in('Referer');
    $req->status_line('302 Found');
    $req->header_out("Location", ($url || $referer || '/users/'));
    $req->send_http_header();

    return 302;
}

sub GetAllVals {
    my $req = shift;
    my @invars = ($req->args(), $req->content());
    my $param = {};
    while (my ($k, $v) = splice(@invars,0,2)) {
        utf8::decode($v);
        $v =~ s/^\s+//g;
        $v =~ s/\s+$//g;
        if ($k =~ s/\[\]$//) {
            push @{$param->{$k}}, $v;
        } else {
            $param->{$k} = $v;
        }
    }
    return $param;
}

sub SetDefaultParams {
    my ($req, $AllVals) = @_;

    my %default = ();

    $default{nolight} = $AllVals->{nolight} || 0;
    $default{nopdd}   = $AllVals->{nopdd}   || 0;
    $default{limit}   = $AllVals->{limit}   || 0;

    Apache::Cookie->new($req,
        -name => 'settings',
        -value => \%default,
        -domain => $req->hostname,
        -path => '/',
        -expires => '+10y',
    )->bake;
}

sub GetDefaultParams {
    my ($req, $AllVals) = @_;

    my %cookies = Apache::Cookie->fetch;

    if (%cookies) {
        my $cookie = $cookies{settings};
        if ($cookie) {
            my %default = $cookie->value;

            $AllVals->{nolight} = $default{nolight} || 0;
            $AllVals->{nopdd}   = $default{nopdd}   || 0;
            $AllVals->{limit}   = $default{limit}   || 0;
        }
    } else {
        $AllVals->{nopdd}   = 1;
    }
}

sub handler {
    my $req = shift;

    ADM::Logs::DeBug("Adminko handler(): started");

    my $grants_filepath = $admin::Conf->GetVal('grants_filepath');
    my $guard = $admin::Guard;
    eval {
        if ($grants_filepath) {
            $guard->load_file($grants_filepath);
        }
    };
    if ($@) {
        ADM::Logs::IntErr("in handler: grants loading failed: $@; keep previous grants");
    }

    # создаём переменные пакета Input для совместимости с cgi-bin
    ADM::FakeInput::InitInput($req);

    my $AllVals = GetAllVals($req);

    my $uri = $req->uri();

    my $acd = ADM::Core::DB->new();
    unless ($acd) {
        return 200;
    }
    $acd->dbaf->FlushCache();

    my $tt = ADM::Template->new($admin::Template);
    unless($tt) {
        return 200;
    }

    my $request_uri = URI->new;
    $request_uri->scheme('https');
    $request_uri->host($req->hostname);
    $request_uri->path($req->uri);
    $request_uri->query_form(($req->args));

    my %cookies = Apache::Cookie->fetch;
    my $session;
    my $session_uid;
    my $session_login;
    my $session_status;
    my $session_authid;
    my $is_valid_session;
    my $is_rotten_session;

    if ($cookies{Session_id}) {
        $session = ADM::Requester::CheckTeamSessionId(
            host      => $req->hostname,
            ip        => $ENV{REMOTE_ADDR},
            sessionid => $cookies{Session_id}->value,
            queries   => [],
        );
        my $method = $req->method;
        $session_uid       = $session->{uid}{value}    || 0;
        $session_login     = lc($session->{login}      || '');
        $session_status    = $session->{status}{value} || '';
        $session_authid    = $session->{authid}{id}    || '';
        $is_valid_session  = ($session_status eq 'VALID' or ($session_status eq 'NEED_RESET' and not $method eq 'GET')) ? 1 : 0;
        $is_rotten_session = ($session_status eq 'NEED_RESET' and $method eq 'GET') ? 1 : 0;
        ADM::Logs::DeBug("teamauth: uid=$session_uid login=$session_login status=$session_status is_valid=$is_valid_session is_rotten=$is_rotten_session");
    }
    else {
        ADM::Logs::DeBug("teamauth: nosessionid");
    }

    if ($is_valid_session) {
        $ENV{REMOTE_USER}    = $session_login;
        $ENV{SESSION_UID}    = $session_uid;
        $ENV{SESSION_AUTHID} = $session_authid;
        $tt->Assign('AdminLogin', $session_login);
    }
    elsif ($is_rotten_session) {
        my $team_mda_url = $admin::Conf->GetVal('team_mda_url');
        my $location = URI->new($team_mda_url);
        $location->path('/auth/update');
        $location->query_form(
            retpath => $request_uri,
        );
        return RedirBack($req, $location);
    }
    else {
        my $team_passport_url = $admin::Conf->GetVal('team_passport_url');
        my $location = URI->new($team_passport_url);
        $location->path('/auth');
        $location->query_form(
            from    => 'adm',
            retpath => $request_uri,
        );
        return RedirBack($req, $location);
    }

    unless (ADM::Utils::IsUserAllowed()) {
        $tt->SetTemplate('deny.tt2');
        ADM::Logs::DeBug("guard: login=$session_login forbidden");
        return DoResponse($req, $tt, $AllVals);
    }

    ADM::Logs::DeBug("guard: login=$session_login roles=" . ADM::Utils::UserRolesString());

    AssignResultVars($tt, { request_uri => $request_uri });

    if ($uri =~ m{^/users/stopreg/?}) {
        if ($AllVals->{'login'}) {
            my $is_stop_word = $acd->is_stop_word($AllVals->{login});
            AssignHashVars($tt, {
                is_stop_word => $is_stop_word,
                login        => $AllVals->{login},
            });
        }
        $tt->SetTemplate('stopreg_info.tt2');
        return DoResponse($req, $tt, $AllVals);

    } elsif ($uri =~ m{^/users/padmin/?}) {
        return RedirBack($req);

    } elsif ($uri =~ m{^/users/jbstatchg/?}) {
        $tt->SetTemplate('disable.tt2');

        unless (ADM::Utils::HaveGrant('allow_jabber_change')) {
            return DoResponse($req, $tt, $AllVals);
        }

        my $uid = $AllVals->{uid};
        my $subscriptions = $acd->GetSubscriptionsInfo({'uid' => $uid});
        if ($subscriptions) {
            foreach my $subscr (@$subscriptions) {
                # Block subscription:
                if ( $subscr->{'sid'} == 27 and $subscr->{'login_rule'} ) {
                    ADM::Logs::DeBug("Subscription: Blocking   27 for uid: $uid;");
                    $acd->dbaf->UpdateLoginRule($uid, 27, 0);
                } elsif ( $subscr->{'sid'} == 27 and ! $subscr->{'login_rule'} ) {
                    ADM::Logs::DeBug("Subscription: UnBlocking 27 for uid: $uid;");
                    $acd->dbaf->UpdateLoginRule($uid, 27, 1);
                }
            }
            AssignResultVars($tt, {'subscriptions' => $subscriptions});

        } elsif ($acd->dberror()) {
            ADM::Logs::IntErr($acd->dberror());
        }

    } elsif ($uri =~ m{^/users/disable/?}) {
        $tt->SetTemplate('disable.tt2');

        unless (ADM::Utils::HaveAtLeastOneGrant('allow_karma_whiten', 'allow_karma_blacken', 'allow_account_enable', 'allow_account_disable')) {
            return DoResponse($req, $tt, $AllVals);
        }

        if (defined $AllVals->{enable}) {
            if ($AllVals->{enable} and not ADM::Utils::HaveGrant('allow_account_enable')) {
                AssignResultVars($tt, { error => "you don't have 'allow_account_enable' permission" });
                return DoResponse($req, $tt, $AllVals);
            }
            if (not $AllVals->{enable} and not ADM::Utils::HaveGrant('allow_account_disable')) {
                AssignResultVars($tt, { error => "you don't have 'allow_account_disable' permission" });
                return DoResponse($req, $tt, $AllVals);
            }
        }

        if (defined $AllVals->{karma}) {
            if ($AllVals->{karma} and not ADM::Utils::HaveGrant('allow_karma_whiten')) {
                AssignResultVars($tt, { error => "you don't have 'allow_karma_whiten' permission" });
                return DoResponse($req, $tt, $AllVals);
            }
            if (not $AllVals->{karma} and not ADM::Utils::HaveGrant('allow_karma_blacken')) {
                AssignResultVars($tt, { error => "you don't have 'allow_karma_blacken' permission" });
                return DoResponse($req, $tt, $AllVals);
            }
        }

        my $admin_login = $ENV{REMOTE_USER};
        AssignResultVars($tt, { admin_login => $admin_login });

        AssignResultVars($tt, { enable => $AllVals->{enable} })
          if defined $AllVals->{enable};

        my @logins;
        my $uids = $AllVals->{uids} || [];

        if (not UNIVERSAL::isa($uids, 'ARRAY')) {
            $uids = [ $uids ];
        }

        my @uids = uniq grep { !/[^\d]/ } grep { $_ } map { split /[\n\r\s\t]+/ } @$uids;

        if (@uids) {
            my $accounts = eval { $acd->GetAccountsByGroups(\@uids) };
            if ($@ or not $accounts) {
                AssignResultVars($tt, { error => "Internal error, try again later" });
                return DoResponse($req, $tt, $AllVals);
            }
            @logins = map +($_->machine_readable_login), @$accounts;
        } elsif ($AllVals->{logins}) {
            @logins = split(/[\n\r\s\t]+/,$AllVals->{'logins'});
            @logins = map (do {s/[@\.](yandex|narod|ya)\.(ru|com|com\.tr|ua|by|kz)$//;$_;}, @logins);
            @logins = grep {/.+/} @logins;

            for my $login (@logins) {
                my $account = $acd->dbaf->GetAccount($login);
                next unless $account;
                next unless $account->uid;
                push @uids, $account->uid;
            }
        }

        if (@logins) {
            AssignResultVars($tt, { disable_logins => \@logins });
        }

        if (@uids) {
            AssignResultVars($tt, { disable_uids => \@uids });
        }

        return DoResponse($req, $tt, $AllVals)
          unless $AllVals->{submit};

        unless ($AllVals->{admin_comment}) {
            AssignResultVars($tt, { error => "Specify comment" });
            return DoResponse($req, $tt, $AllVals);
        }

        unless (@uids) {
            AssignResultVars($tt, { error => "Specify either uid or login" });
            return DoResponse($req, $tt, $AllVals);
        }

        my $params = {
            uids   => \@uids,
            enable => $AllVals->{enable},
            karma  => $AllVals->{karma},
            admin_login => $admin_login,
            admin_comment => $AllVals->{admin_comment},
        };
        unless ($acd->EnableAccounts($params)) {
            AssignResultVars($tt, { error => "Internal error, try again later" });
            return DoResponse($req, $tt, $AllVals);
        }
        if ($acd->dberror()) {
            ADM::Logs::IntErr($acd->dberror());
            AssignResultVars($tt, { error => "Internal error, try again later" });
            return DoResponse($req, $tt, $AllVals);
        }

        AssignResultVars($tt, { status => "DONE!" });
        return DoResponse($req, $tt, $AllVals);

    } elsif ($uri =~ m{^/users/form/?}) {
        my $uid = $AllVals->{id_client} || $AllVals->{uid};
        return RedirBack($req, "/users/account/view?uid=$uid");

    } elsif ($uri =~ m{^/users/userdata/?}) {
        $tt->SetTemplate('userdata.tt2');

        unless (ADM::Utils::HaveGrant('show_history')) {
            return DoResponse($req, $tt, $AllVals);
        }

        my $uid = $AllVals->{'id_client'} || $AllVals->{'uid'};
        unless ($uid and $uid =~ /^\d+$/) {
            return RedirBack($req, '/users/');
        }

        my $acc_regdate;
        my $account = $acd->dbaf->GetAccount($uid);
        if ($account) {
            $acc_regdate = substr($account->registration_datetime, 0, 10);
            if ($acc_regdate eq '0000-00-00') {
                $acc_regdate = undef;
            }
        } elsif ($acd->dberror()) {
            ADM::Logs::IntErr($acd->dberror());
        } else {
            my $acc_ft = ADM::Requester::GetHistoryDb3UserinfoFT(uid => $uid);
            if ($acc_ft) {
                $acc_regdate = substr($acc_ft->{dt}, 0, 10);
                if ($acc_regdate eq '0000-00-00') {
                    $acc_regdate = undef;
                }
            } elsif ($acd->dberror()) {
                ADM::Logs::IntErr($acd->dberror());
            }
        }


        my ($year, $month, $day) = Date::Calc::Localtime();
        AssignResultVars($tt, {
            'ud_one_year'  => sprintf("%04d-%02d-%02d", Date::Calc::Add_Delta_YM($year, $month, $day, -1, 0)),
            'ud_half_year' => sprintf("%04d-%02d-%02d", Date::Calc::Add_Delta_YM($year, $month, $day, 0, -6)),
            'ud_one_month' => sprintf("%04d-%02d-%02d", Date::Calc::Add_Delta_YM($year, $month, $day, 0, -1)),
            'ud_one_week'  => sprintf("%04d-%02d-%02d", Date::Calc::Add_Delta_Days($year, $month, $day, -7)),
            'ud_one_day'   => sprintf("%04d-%02d-%02d", Date::Calc::Add_Delta_Days($year, $month, $day, -1)),
            'ud_now'       => sprintf("%04d-%02d-%02d", $year, $month, $day),
            'ud_reg_date'  => $acc_regdate,
        });

        my $last_auth = ADM::Requester::GetHistoryDb3Lastauth(uid => $uid);
        if ($last_auth) {
            $last_auth = Global::TimestampToLocalDatetime($last_auth);
            $last_auth = sprintf("%04d-%02d-%02d", split(/[-\s]/, $last_auth));

            AssignResultVars($tt, {
                'ud_last_auth'  => $last_auth,
                'ud_last_day'   => sprintf("%04d-%02d-%02d", Date::Calc::Add_Delta_Days(split('-', $last_auth), -1)),
                'ud_last_week'  => sprintf("%04d-%02d-%02d", Date::Calc::Add_Delta_Days(split('-', $last_auth), -7)),
                'ud_last_month' => sprintf("%04d-%02d-%02d", Date::Calc::Add_Delta_YM(split('-', $last_auth), 0, -1)),
                'ud_last_half'  => sprintf("%04d-%02d-%02d", Date::Calc::Add_Delta_YM(split('-', $last_auth), 0, -6)),
                'ud_last_year'  => sprintf("%04d-%02d-%02d", Date::Calc::Add_Delta_YM(split('-', $last_auth), -1, 0)),
            });
        }

        unless ($AllVals->{'Search'}) {
            # Делаем Changes History дефолтно включённым
            # Also sets template var

            for (qw(s_reg s_auth auth_type_all_default auth_status_all_default)) {
                $AllVals->{$_} = 1;
            }

            $AllVals->{'auth_start_date'}  = $tt->{'__VARS__'}{'UdLastYear'};
            $AllVals->{'auth_end_date'}    = $tt->{'__VARS__'}{'UdLastAuth'};
            $AllVals->{'event_start_date'} = $tt->{'__VARS__'}{'UdRegDate'};
            $AllVals->{'event_end_date'}   = $tt->{'__VARS__'}{'UdNow'};
        }

        DoUserDataSearch($tt, $acd, $AllVals);

    } elsif ($uri =~ m{^/users/warn/?}) {
        $tt->SetTemplate('warn.tt2');

        unless (ADM::Utils::HaveAtLeastOneGrant('allow_warning_send', 'allow_password_changing_require')) {
            return DoResponse($req, $tt, $AllVals);
        }

        if ($AllVals->{users}) {
            my $users = [];
            my $is_logins = $AllVals->{is_logins};
            for (split /[\n\r\s\t]+/, $AllVals->{users}) {
                s/^\s+//;
                s/\s+$//;
                if ($is_logins){
                    s/^<//;
                } else {
                    $_ = '' unless /^\d+$/;
                }
                next unless $_;
                push @$users, $_;
            }

            my $message_id = $AllVals->{message_id};
            if (not $message_id =~ /^\d+$/) {
                return RedirBack($req, '/users/');
            }

            if ($message_id and not ADM::Utils::HaveGrant('allow_warning_send')) {
                AssignResultVars($tt, { error => "you don't have 'allow_warning_send' permission" });
                return DoResponse($req, $tt, $AllVals);
            }
            if (not $message_id and not ADM::Utils::HaveGrant('allow_password_changing_require')) {
                AssignResultVars($tt, { error => "you don't have 'allow_password_changing_require' permission" });
                return DoResponse($req, $tt, $AllVals);
            }

            my $warn_results = $acd->WarnUsers(
                users => $users,
                is_logins => $is_logins,
                message_id => $message_id,
                limit => 200,
            );
            if (not $warn_results and $acd->dberror) {
                ADM::Logs::IntErr($acd->dberror);
            } else {
                ADM::Logs::DeBug("warn logins: message_id=$message_id results: " . join("\n", @$warn_results));
                $tt->Assign('WarnResults', $warn_results);
                $tt->Assign('Users', $users);
                $tt->Assign('MessageId', $message_id);
                $tt->Assign('IsLogins', $is_logins);
            }
        } else {
            $tt->Assign('IsLogins', 1);
        }

    } elsif ($uri =~ m{^/users/auths_statistics_extended/?}) {
        my $retpath = $AllVals->{retpath} || "/users/";

        my $uid = $AllVals->{uid};

        return RedirBack($req, $retpath)
          unless $uid =~ /^\d+/;

        my $to   = time;
        my $from = $to - 86400 * 365;

        my $auths_statistics_extended = ADM::Requester::GetHistoryDb3AuthsStatisticsExtended(
            uid  => $uid,
            ip   => $ENV{REMOTE_ADDR},
            from => $from,
            to   => $to,
        );

        $tt->Assign('AuthsStatisticsExtended' => $auths_statistics_extended);

        $tt->SetTemplate('auths_statistics_extended.tt2');

    } elsif ($uri =~ m{^/users/account/(.+)/?}) {
        return HandleAccountForms($req, $tt, $AllVals, $acd, $1);

    } elsif ($uri =~ m{^/users/key_diag/?}) {
        return HandleKeyDiagnosticForms($req, $tt, $AllVals, $acd);

    } elsif ($uri =~ m{^/users/sms_history/?}) {
        return HandleSmsHistoryForms($req, $tt, $AllVals, $acd);

    } elsif ($uri =~ m{^/users/sms_routing/?}) {
        return HandleSmsRoutingForms($req, $tt, $AllVals, $acd);

    } elsif ($uri =~ m{^/users/money/?}) {
        return RedirBack($req, '/users/');

    } else {
        my ($year, $month, $day) = Date::Calc::Localtime();
        AssignResultVars($tt, {
            'ud_one_year'  => sprintf("%04d-%02d-%02d", Date::Calc::Add_Delta_YM($year, $month, $day, -1, 0)),
            'ud_now'       => sprintf("%04d-%02d-%02d", $year, $month, $day),
        });

        if ($AllVals->{search}) {
            SetDefaultParams($req, $AllVals);
        } else {
            GetDefaultParams($req, $AllVals);
        }
        DoIndexSearches($tt, $acd, $AllVals);
        $tt->SetTemplate('index.tt2');
    }

    return DoResponse($req, $tt, $AllVals);
}

sub HandleAccountForms {
    our ($req, $tt, $AllVals, $acd, $handler) = @_;

    our $uid = 0;

    $uid = $AllVals->{uid};
    $uid =~ s/\D+//g;


    $tt->SetTemplate('account_view.tt2');
    $tt->Assign('show_hint_answer', $AllVals->{show_hint_answer});

    my $uid2shard = sub {
        my $u = shift;
        my $result = $acd->dbaf->uid_collection->get_shard_id_by_key($u);
        return $result;
    };

    my $is_stale_phone = sub {
        my $phone = shift;
        return 0 unless $phone->confirmation_ts;
        return (time - $phone->confirmation_ts > 4383 * 60 * 60) ? 1 : 0;
    };

    $tt->Assign('uid2shard', $uid2shard);
    $tt->Assign('is_stale_phone', $is_stale_phone);

    sub render_error {
        my $error = shift;
        $tt->Assign('message', { level => 'error', text => $error });
        return DoResponse($req, $tt, $AllVals);
    };

    sub render_message {
        my ($level, $text) = @_;
        Apache::Cookie->new($req,
            -name     => 'message',
            -value    => {
                level => $level,
                text  => $text,
            },
            -domain   => $req->hostname,
            -path     => '/',
            -expires  => '+10s',
        )->bake;
        return RedirBack($req, "/users/account/view?uid=$uid");
    };


    # Message displaying
    my %cookies = Apache::Cookie->fetch;
    if (%cookies) {
        my $cookie = $cookies{message};
        if ($cookie) {
            my %message = $cookie->value;
            Apache::Cookie->new($req,
                -name    => 'message',
                -value   => undef,
                -domain  => $req->hostname,
                -path    => '/',
                -expires => '-1y',
            )->bake;
            $tt->Assign('message', \%message);
        }
    }


    # Account loading
    return render_error("invalid uid=$uid")
      unless $uid;

    my $account = $acd->dbaf->GetAccountWithPhonesAndSocialProfiles($uid);
    return render_error("can't get account info for uid=$uid")
      unless $account;

    $tt->Assign('account', $account);

    my $support_notes = ADM::Requester::GetHistoryDb3SupportNotes(uid => $uid);
    return render_error("can't get support_notes for uid=$uid")
      unless $support_notes;

    $account->{support_notes} = $support_notes;


    my $admin_login = ADM::Utils::GetCurrentLogin();


    # CSRF-token processing
    my $expected_csrf_token = Common::Csrf::GenerateToken(fields => [
        $ENV{SESSION_UID},
        $ENV{SESSION_AUTHID},
        'account_form',
    ]);

    $tt->Assign('csrf_token', $expected_csrf_token);

    my $actual_csrf_token = $AllVals->{csrf_token};

    unless ($handler eq 'view') {
        return render_message(error => 'possible csrf-attack is catched (no token), try again')
          unless $actual_csrf_token;

        my $token_lifetime = Common::Csrf::GetTokenLifetime($actual_csrf_token);

        return render_message(error => 'possible csrf-attack is catched (expired token), try again')
          if $token_lifetime > 3 * 60 * 60;

        my $is_csrf_token_valid = Common::Csrf::ValidateToken(
            token => $actual_csrf_token,
            fields => [
                $ENV{SESSION_UID},
                $ENV{SESSION_AUTHID},
                'account_form',
            ],
        );

        return render_message(error => 'possible csrf-attack is catched (invalid token), try again')
          unless $is_csrf_token_valid;
    }


    # Views handling
    if ($handler eq 'view') {
        return DoResponse($req, $tt, $AllVals);
    }

    # Delete
    elsif ($handler eq 'delete') {
        return render_message(error => "you don't have 'allow_account_delete' permission")
          unless ADM::Utils::HaveGrant('allow_account_delete');

        my $dsr = $acd->dbaf->DeleteSubscriptions($uid, force_drop => 1);

        my $opdata = {
            uid     => $uid,
            ip_from => $ENV{REMOTE_ADDR},
            optype  => 'del',
        };
        my $changes = $dsr->{changes};

        $changes->{login} = $account->machine_readable_login;
        $changes->{admin_login} = $admin_login;

        if ($dsr->{error}) {
            if (%$changes) {
                Common::Logs::LogChanges($opdata, $changes);
            }
            return render_error("can't delete account's subscriptions");
        }

        unless ($acd->dbaf->DeleteUser($uid)) {
            if (%$changes) {
                Common::Logs::LogChanges($opdata, $changes);
            }
            return render_error("can't delete account");
        }

        ADM::Logs::LogChanges($opdata, $changes);

        return render_message(info => 'Account has been deleted');
    }

    # Refresh glogout
    elsif ($handler eq 'refresh_glogout') {
        return render_message(error => "you don't have 'allow_global_logout_reset' permission")
          unless ADM::Utils::HaveGrant('allow_global_logout_reset');

        my $time = time;

        return render_message(error => "can't refresh account's glogout")
          unless $acd->dbaf->GlobalLogout($uid, $time);

        my $opdata = {
            uid     => $uid,
            ip_from => $ENV{REMOTE_ADDR},
            optype  => 'mod',
        };
        my $changes = {
            admin_login => $admin_login,
            glogout     => $time,
        };
        ADM::Logs::LogChanges($opdata, $changes);

        return render_message(info => 'Glogout has been refreshed');
    }

    # Drop hint
    elsif ($handler eq 'drop_hint') {
        return render_message(error => "you don't have 'allow_hint_reset' permission")
          unless ADM::Utils::HaveGrant('allow_hint_reset');

        my $toupdate = {
            hintq => '0',
            hinta => '',
        };
        my $vals = { %$toupdate };

        return render_message(error => "can't drop account's hint")
          unless $acd->dbaf->UpdateAccountInfo($uid, $toupdate, $vals);

        my $opdata = {
            uid     => $uid,
            ip_from => $ENV{REMOTE_ADDR},
            optype  => 'mod',
        };
        my $changes = {
            %$toupdate,
            admin_login => $admin_login,
        };
        ADM::Logs::LogChanges($opdata, $changes);

        return render_message(info => 'Hint has been dropped');
    }

    # Restore mailbox
    elsif ($handler eq 'restore_mailbox') {
        return render_message(error => "you don't have 'allow_mail_restore' permission")
          unless ADM::Utils::HaveGrant('allow_mail_restore');

        my $mailbox_info = $AllVals->{mailbox_info};

        return render_message(error => 'empty mailbox_info')
          unless $mailbox_info;

        my ($sid, $olduid, $oldlogin, $suid, $hostid) = split /\|/, $mailbox_info;

        ADM::Logs::DeBug("adminmailsidrestore: adminlogin=$admin_login target=$uid olduid=$olduid oldlogin=$oldlogin suid=$suid hostid=$hostid");

        return render_message(error => 'invalid mailbox_info')
          unless $sid and $olduid and $oldlogin and $suid and $hostid;

        return render_message(error => 'alien mailbox_info')
          if $uid != $olduid or lc $account->machine_readable_login ne lc $oldlogin;

        return render_message(error => "can't restore account's mailbox")
          unless $acd->dbaf->RestoreMailInfo($uid, $oldlogin, $suid, $hostid);

        my $opdata = {
            uid     => $uid,
            ip_from => $ENV{REMOTE_ADDR},
            optype  => 'mod',
        };
        my $changes = {
            restore_subscr => $mailbox_info,
            admin_login    => $admin_login,
        };
        ADM::Logs::LogChanges($opdata, $changes);

        return render_message(info => 'Mailbox has been restored');
    }

    # Remove alias
    elsif ($handler eq 'remove_alias') {
        return render_message(error => "you don't have 'allow_service_login_delete' permission")
          unless ADM::Utils::HaveGrant('allow_service_login_delete');

        my $alias_name = $AllVals->{alias_name};

        return render_message(error => 'empty alias_name')
          unless $alias_name;

        my %removable_aliases = (
            mail      => 1,
            narodmail => 1,
            narod     => 1,
            lite      => 1,
        );

        return render_message(error => "invalid alias_name=$alias_name")
          unless $removable_aliases{$alias_name};

        return render_message(error => "account must have a portal alias")
          unless $account->aliases->portal;

        return render_message(error => "can't remove account's alias")
          unless $acd->dbaf->RemoveAlias($uid, $alias_name);

        return render_message(info => 'Alias has been removed');
    }

    # Create restoration link
    elsif ($handler eq 'create_restoration_link') {
        return render_message(error => "you don't have 'allow_restoration_link_create' permission")
          unless ADM::Utils::HaveGrant('allow_restoration_link_create');

        my $type = $AllVals->{type} || 1;

        return render_message(error => "type=2 isn't allowed if 2fa")
          if $type == 2 and $account->password->is_2fa_enabled;

        my $is_incomplete_social_account = (not $account->password->is_exists and $account->aliases->social and $account->aliases->portal) ? 1 : 0;
        return render_message(error => "type=4 is allowed only if password creating is required or social account with login has no password")
          if $type == 4
            and not $account->password->is_creating_required
            and not $is_incomplete_social_account;

        my $host = $main::Conf->GetVal('pass_domain');
        $host .= 'yandex.ru'
          unless $host =~ /yandex\.ru$/;

        my $url = ADM::Requester::CreateRestorationEmailUrl(
            uid   => $uid,
            type  => $type,
            admin => $admin_login,
            ip    => $ENV{REMOTE_ADDR},
            host  => $host,
        );

        return render_message(error => "can't generate restoration key")
          unless $url;

        $tt->Assign('restoration_url',  $url);
        $tt->Assign('restoration_type', $type);

        $tt->SetTemplate('restoration_link.tt2');

        return DoResponse($req, $tt, $AllVals);
    }

    # Add support note
    elsif ($handler eq 'add_support_note') {
        # TODO нужен грант?

        my $text = $AllVals->{text};

        return render_message(error => 'empty text')
          unless length $text;

        my $opdata = {
            uid     => $uid,
            ip_from => $ENV{REMOTE_ADDR},
            optype  => 'mod',
        };
        my $changes = {
            support_note => $text,
            admin_login  => $admin_login,
        };
        ADM::Logs::LogChanges($opdata, $changes);

        return render_message(info => 'Support note has been added');
    }

    # Change VIP subscription
    elsif ($handler eq 'change_vip_subscription') {
        return render_message(error => "you don't have 'allow_vip_change' permission")
          unless ADM::Utils::HaveGrant('allow_vip_change');

        my $value = $AllVals->{value};

        return render_message(error => 'empty value')
          unless defined $value;

        $acd->SetIsVipAccount(
            uid         => $uid,
            flag        => $value ? 1 : 0,
            admin_login => $admin_login,
        );

        return render_message(info => 'VIP subscription has been changed');
    }

    # Change password changing requirement
    elsif ($handler eq 'change_password_changing_requirement') {
        return render_message(error => "you don't have 'allow_password_changing_require' permission")
          unless ADM::Utils::HaveGrant('allow_password_changing_require');

        my $value = $AllVals->{value};
        my $reason = $AllVals->{reason};

        return render_message(error => 'empty value')
          unless defined $value;

        return render_message(error => 'empty reason')
          unless length $reason;

        my $result = ADM::Requester::UpdatePasswordOptions(
            uid   => $uid,
            admin => $admin_login,
            comment => $reason,
            ip    => $ENV{REMOTE_ADDR},
            is_changing_required => $value,
        );

        return render_message(error => "can't change password changing requirement")
          unless $result;

        return render_message(info => 'Password changing requirement has been changed');
    }

    # Change shared status
    elsif ($handler eq 'change_shared_status') {
        return render_message(error => "you don't have 'allow_shared_change' permission")
          unless ADM::Utils::HaveGrant('allow_shared_change');

        my $value = $AllVals->{value};
        my $reason = $AllVals->{reason};

        return render_message(error => 'empty value')
          unless defined $value;

        return render_message(error => 'empty reason')
          unless length $reason;

        my $result = ADM::Requester::UpdateAccountOptions(
            uid   => $uid,
            admin => $admin_login,
            comment => $reason,
            ip    => $ENV{REMOTE_ADDR},
            is_shared => $value,
        );

        return render_message(error => "can't change shared status")
          unless $result;

        return render_message(info => 'Shared status has been changed');
    }

    # Drop phone
    elsif ($handler eq 'drop_phone') {
        return render_message(error => "you don't have 'allow_phone_delete' permission")
          unless ADM::Utils::HaveGrant('allow_phone_delete');

        my $phoneid = $AllVals->{phoneid};

        return render_message(error => 'empty phoneid')
          unless $phoneid;

        return render_message(error => "can't drop account's phone")
          unless ADM::Requester::DropPhone($uid, $phoneid);

        return render_message(info => 'Phone has been deleted');
    }

    # Defaultify phone
    elsif ($handler eq 'defaultify_phone') {
        return render_message(error => "you don't have 'allow_default_phone_change' permission")
          unless ADM::Utils::HaveGrant('allow_default_phone_change');

        my $phoneid = $AllVals->{phoneid};

        return render_message(error => 'empty phoneid')
          unless $phoneid;

        return render_message(error => "can't set account's phone as default")
          unless ADM::Requester::SetDefaultPhone(uid => $uid, phone_id => $phoneid, ip => $ENV{REMOTE_ADDR});

        return render_message(info => 'Phone has been set as default');
    }

    # Drop email
    elsif ($handler eq 'drop_email') {
        return render_message(error => "you don't have 'allow_email_delete' permission")
          unless ADM::Utils::HaveGrant('allow_email_delete');

        my $email = $AllVals->{email};

        return render_message(error => 'empty email')
          unless $email;

        return render_message(error => "can't drop account's email")
          unless ADM::Requester::DropEmail(
              uid     => $uid,
              email   => $email,
              admin   => $admin_login,
              comment => 'removed by button',
              ip      => $ENV{REMOTE_ADDR},
          );

        return render_message(info => 'Email has been deleted');
    }


    else {
        return DoResponse($req, $tt, $AllVals);
    }
}

sub HandleSmsHistoryForms {
    our ($req, $tt, $AllVals, $acd) = @_;

    our $sms_id       = $AllVals->{sms_id};
    our $phone_number = $AllVals->{phone_number};
    our $from         = $AllVals->{from};
    our $to           = $AllVals->{to};

    sub render_error {
        my $error = shift;
        $tt->Assign('message', { level => 'error', text => $error });
        return DoResponse($req, $tt, $AllVals);
    };

    $tt->SetTemplate('sms_history.tt2');

    return render_error("you don't have 'allow_sms_search' permission")
      unless ADM::Utils::HaveGrant('allow_sms_search');

    $phone_number =~ s/^\s+//g;
    $phone_number =~ s/^8/+7/g;

    if ($phone_number =~ /^\+/) {
        $phone_number =~ s/\D//g;
        $phone_number = "+$phone_number";
    }
    else {
        $phone_number =~ s/\D//g;
    }

    $sms_id =~ s/\D//g;

    $tt->Assign('phone_number', $phone_number) if $phone_number;
    $tt->Assign('sms_id',       $sms_id)       if $sms_id;
    $tt->Assign('history_from', $from)         if $from;
    $tt->Assign('history_to',   $to)           if $to;

    my ($year, $month, $day) = Date::Calc::Localtime();
    $tt->Assign('one_day_ago',   sprintf '%04d-%02d-%02d', Date::Calc::Add_Delta_Days($year, $month, $day, -1));
    $tt->Assign('one_week_ago',  sprintf '%04d-%02d-%02d', Date::Calc::Add_Delta_Days($year, $month, $day, -7));
    $tt->Assign('one_month_ago', sprintf '%04d-%02d-%02d', Date::Calc::Add_Delta_YM($year, $month, $day, 0, -1));
    $tt->Assign('six_month_ago', sprintf '%04d-%02d-%02d', Date::Calc::Add_Delta_YM($year, $month, $day, 0, -6));
    $tt->Assign('one_year_ago',  sprintf '%04d-%02d-%02d', Date::Calc::Add_Delta_YM($year, $month, $day, -1, 0));
    $tt->Assign('today',         sprintf '%04d-%02d-%02d', $year, $month, $day);

    return DoResponse($req, $tt, $AllVals)
      unless $AllVals->{search};

    my $raw_sms_history;

    my %filter = (
        from => Global::DatetimeToTimestamp($from),
        to   => Global::DatetimeToTimestamp("$to 23:59:59"),
    );

    if ($phone_number) {
        $phone_number =~ s/^\+//g;

        return render_error("invalid phone number")
          unless $phone_number;

        $raw_sms_history = ADM::Requester::GetSmsHistory(%filter, phone_number => $phone_number);
    }
    elsif ($sms_id) {
        return render_error("invalid sms id")
          unless $sms_id;

        $raw_sms_history = ADM::Requester::GetSmsHistory(%filter, sms_id => $sms_id);
    }
    else {
        return render_error("filter isn't specified");
    }

    return render_error("can't search in sms history")
      unless $raw_sms_history;

    my $routing_dump = ADM::Requester::SmsRouting(action => 'dump');
    my $gates  = $routing_dump->{gates};
    my %gates  = map { $_->{gateid} => $_ } @$gates;

    my $normalized_sms_history = ADM::Utils::NormalizeSmsHistory($raw_sms_history);
    $tt->Assign('sms_history', $normalized_sms_history);

    $tt->Assign('sms_gates', \%gates);

    return DoResponse($req, $tt, $AllVals);
}

sub HandleSmsRoutingForms {
    our ($req, $tt, $AllVals, $acd) = @_;

    sub render_error {
        my $error = shift;
        $tt->Assign('message', { level => 'error', text => $error });
        return DoResponse($req, $tt, $AllVals);
    };

    $tt->SetTemplate('sms_routing.tt2');

    return render_error("you don't have 'allow_sms_routing' permission")
      unless ADM::Utils::HaveGrant('allow_sms_routing');

    my $action       = $AllVals->{action};
    my $phone_number = $AllVals->{phone_number};
    my $mode         = $AllVals->{mode};
    my $gate_id      = $AllVals->{gate_id};
    my $text         = $AllVals->{text};

    $tt->Assign('phone_number', $phone_number);

    my $routing_dump = ADM::Requester::SmsRouting(action => 'dump');
    my $routes = $routing_dump->{routes};
    my $gates  = $routing_dump->{gates};
    my %gates  = map { $_->{gateid} => $_ } @$gates;

    for my $route (@$routes) {
        $route->{gate} = $gates{ $route->{gateid} } || {};
    }

    if ($action eq 'route') {
        my ($message, $color);

        $tt->SetTemplate('sms_routing_inner.tt2');

        unless ($phone_number) {
            $message = 'Phone Number is required';
            $color = 'red';
        }

        unless ($mode) {
            $message = 'Gate is required';
            $color = 'red';
        }

        if ($phone_number and $mode) {
            my $routing = ADM::Requester::SmsRouting(
                action => 'route',
                number => $phone_number,
                route  => $mode,
            );

            return render_error("can't get sms routing dump")
              unless $routing;

            my $possible_gate_ids = $routing->{possible_gates};

            $gate_id
              = (@$possible_gate_ids and $gates{ $possible_gate_ids->[0] })
              ? $possible_gate_ids->[0]
              : 0;

            $message
              = join ' OR ',
                map { $gates{$_}{description} || "unknown (#$_)" }
                @$possible_gate_ids;

            $color = @$possible_gate_ids ? 'green' : 'red';
        }

        $tt->Assign('gate_id', $gate_id);
        $tt->Assign('message', $message);
        $tt->Assign('color',   $color);
    }

    elsif ($action eq 'send') {
        my ($message, $color);

        $tt->SetTemplate('sms_routing_inner.tt2');

        unless ($phone_number) {
            $message = 'Phone Number is required';
            $color = 'red';
        }

        unless ($gate_id) {
            $message = 'Gate is required';
            $color = 'red';
        }

        unless (length $text) {
            $message = 'Text is required';
            $color = 'red';
        }

        if ($phone_number and $gate_id and length $text) {
            my $smsid = ADM::Requester::SendSms(
                number  => $phone_number,
                gate_id => $gate_id,
                text    => $text,
            );

            if ($smsid) {
                $message = qq{Message is sent #<a href="/users/sms_history?search=1&phone_number=&sms_id=$smsid">$smsid</a>};
                $color = 'green';
            } else {
                $message = 'Sending has failed';
                $color = 'red';
            }
        }

        $tt->Assign('gate_id', $gate_id);
        $tt->Assign('message', $message);
        $tt->Assign('color',   $color);
    }

    else {
        my @sorted_routes = sort {
                (  ($a->{mode} eq 'default' and $b->{mode} eq 'default') ?  0   # Наибольший приоритет у роутов для default
                  : $a->{mode} eq 'default'                              ? -1
                  : $b->{mode} eq 'default'                              ?  1
                  :                                                         0)
             or $a->{mode}               cmp $b->{mode}                         # Собираем в группы по режиму
             or length $b->{destination} <=> length $a->{destination}           # Длинные префиксы приоритетнее
             or $a->{destination}        cmp $b->{destination}                  # И только потом сортируем по самому значению префикса
        } @$routes;

        my @sorted_gates = sort {
                (  ($a->{fromname} eq 'Yandex' and $b->{fromname} eq 'Yandex') ?  0   # Наибольший приоритет у гейтов с альфа-именем Yandex
                  : $a->{fromname} eq 'Yandex'                                 ? -1
                  : $b->{fromname} eq 'Yandex'                                 ?  1
                  :                                                               0)
            or $a->{fromname}    cmp $b->{fromname}                                   # Собираем в группы по альфа-имени
            or $a->{description} cmp $b->{description}                                # И в них сортируем по описанию
        } @$gates;

        my @modes = uniq map { $_->{mode} } @$routes;

        $tt->Assign('routes', \@sorted_routes);
        $tt->Assign('modes',  \@modes);
        $tt->Assign('gates',  \@sorted_gates);
    }

    return DoResponse($req, $tt, $AllVals);
}

sub HandleKeyDiagnosticForms {
    our ($req, $tt, $AllVals, $acd) = @_;

    our $uid         = $AllVals->{uid};
    our $raw_strings = $AllVals->{raw_strings};

    sub render_error {
        my $error = shift;
        $tt->Assign('message', { level => 'error', text => $error });
        return DoResponse($req, $tt, $AllVals);
    };

    $tt->SetTemplate('key_diag.tt2');

    return render_error("you don't have 'allow_key_diagnostic' permission")
      unless ADM::Utils::HaveGrant('allow_key_diagnostic');

    # Исправляем скопированный текст из письем, в которых строки с диагностикой
    # могут быть разбиты лишними переводами строк
    $raw_strings =~ s/\R//g;
    $raw_strings =~ s/([0-9]:[0-9]:)/\n$1/g;

    $tt->Assign('uid', $uid) if $uid;
    $tt->Assign('raw_strings', $raw_strings) if $raw_strings;

    return DoResponse($req, $tt, $AllVals)
      unless $AllVals->{search};

    my $public_key_hex  = $admin::Conf->GetVal('key_diag_public_key');
    my $private_key_hex = $admin::Conf->GetVal('key_diag_private_key');

    my $public_key_str  = ADM::Utils::hex2str($public_key_hex);
    my $private_key_str = ADM::Utils::hex2str($private_key_hex);

    my @raw_strings = split /\n/, $raw_strings;

    my @diags;

    for my $raw_string (@raw_strings) {
        $raw_string =~ s/^\s+//;
        $raw_string =~ s/\s+$//;

        next unless length $raw_string;

        my $diag = {
            raw_string => $raw_string,
        };
        push @diags, $diag;

        my ($diag_version, $diag_keys_id, $encoded_string, $encoded_string2) = split /:/, $raw_string, 4;

        # Костыль! Ключ иногда пишет строки не по формату, поэтому вырезаем лишнее сами.
        $encoded_string =~ s/^Optional\("//;
        $encoded_string =~ s/"\)$//;

        unless ($diag_version and $diag_keys_id and $encoded_string) {
            $diag->{error} = 'invalid raw string',
            next;
        }

        $diag->{version} = $diag_version;

        my $decoded_string   = ADM::Utils::b642str($encoded_string);
        my $decrypted_string = ADM::SodiumSeal::crypto_box_seal_open($decoded_string, $public_key_str, $private_key_str);
        my $decrypted_string2;

        unless ($decrypted_string) {
            $diag->{error} = 'invalid encrypted string';
            next;
        }

        utf8::upgrade $decrypted_string;

        $diag->{decrypted_string} = $decrypted_string;

        if ($encoded_string2) {
            my $decoded_string2 = ADM::Utils::b642str($encoded_string2);
            $decrypted_string2 = ADM::SodiumSeal::crypto_box_seal_open($decoded_string2, $public_key_str, $private_key_str);

            unless ($decrypted_string2) {
                $diag->{error} = 'invalid encrypted2 string';
                next;
            }

            utf8::upgrade $decrypted_string2;

            $diag->{decrypted_string2} = $decrypted_string2;
        }

        my $data;
        eval {
            $data = decode_json $decrypted_string;

            if ($decrypted_string2) {
                my $data2 = decode_json $decrypted_string2;
                $data->{diag} = { %{ $data->{diag} }, %{ $data2->{diag} } };
            }
        };
        if ($@) {
            $diag->{error} = 'invalid decrypted json';
            next;
        }

        $diag->{data} = $data;
        $diag->{pretty_string} = JSON::XS->new->pretty(1)->canonical(1)->encode($data);

        my $hardware_model
          = $data->{device}{manufacturer} eq 'Apple'
          ? ADM::Utils::GetAppleHumanModelByHardwareModel($data->{device}{hardware_model})
          : $data->{device}{hardware_model};

        $diag->{device_model} = join ' ', $data->{device}{manufacturer}, $hardware_model;

        my $device_os = $data->{device}{os_id};
        $device_os =~ s/iP(?:hone|ad) OS/iOS/;
        $diag->{device_os} = $device_os;

        $diag->{device_utctime} = ADM::Utils::TimestampToUtcDatetime($data->{diag}{ts});
        $diag->{device_dt_with_tz_msk} = DateTime->from_epoch(epoch => $data->{diag}{ts}, time_zone => 'Europe/Moscow');
        $diag->{server_utctime} = ADM::Utils::TimestampToUtcDatetime($data->{diag}{ts} + $data->{diag}{crr});

        if ($data->{device}{tzname}) {
            eval {
                my $tz = DateTime::TimeZone->new(name => $data->{device}{tzname});
                my $dt = DateTime->from_epoch(epoch => $data->{diag}{ts}, time_zone => $tz);
                $diag->{device_tz_by_name} = $tz;
                $diag->{device_dt_with_tz_by_name} = $dt;
            };
        }

        if ($data->{device}{tzvalue}) {
            eval {
                my $tz = DateTime::TimeZone->new(name => $data->{device}{tzvalue});
                my $dt = DateTime->from_epoch(epoch => $data->{diag}{ts}, time_zone => $tz);
                $diag->{device_tz_by_offset} = $tz;
                $diag->{device_dt_with_tz_by_offset} = $dt;
            };
        }

        $diag->{magic_offset}
          = $data->{diag}{mgcts}
          ? $data->{diag}{mgcts} - $data->{diag}{ts}
          : 0;

        $diag->{correction_hms} = ADM::Utils::s2hms($data->{diag}{crr});

        if ($@) {
            ADM::Logs::DeBug("cant create DateTime(epoch=$data->{diag}{ts}, tz=$data->{device}{tzname}): $@");
        }

        if ($uid) {
            if ($data->{diag}{uid} and $data->{diag}{uk}) {
                my $expected_uid_id_b64 = ADM::Utils::str2b64(sha256(ADM::Utils::b642str($data->{diag}{uk}) . $uid));
                $diag->{is_uid_id_correct} = $expected_uid_id_b64 eq $data->{diag}{uid} ? 1 : 0;
            }

            my $account = $acd->dbaf->GetAccount($uid);
            unless ($account) {
                $diag->{error} = 'cant get account in blackbox';
                next;
            }

            my $login = lc $account->machine_readable_login;
            $login =~ s/\./-/g;

            my $expected_login_id_b64 = ADM::Utils::str2b64(sha256(ADM::Utils::b642str($data->{diag}{lk}) . $login));
            $diag->{is_login_id_correct} = $expected_login_id_b64 eq $data->{diag}{lid} ? 1 : 0;

            my $prove = ADM::Requester::ProveKeyDiag(
                uid             => $uid,
                secret_salt     => $data->{diag}{sk},
                secret_id       => $data->{diag}{sid},
                pin_secret_salt => $data->{diag}{psk},
                pin_secret_id   => $data->{diag}{psid},
                totp_salt       => $data->{diag}{tk},
                totp_id         => $data->{diag}{tid},
                timestamp       => $data->{diag}{ts},
                skew            => 13 * 3600,
            );

            unless ($prove) {
                $diag->{error} = 'cant prove diag in blackbox';
                next;
            }

            $diag->{prove} = $prove;
            if ($prove->{totp_id}{correct_timestamp}) {
                $prove->{totp_id}{correct_utctime} = ADM::Utils::TimestampToUtcDatetime($prove->{totp_id}{correct_timestamp});
            }
        }
    }

    @diags = sort { $b->{data}{diag}{ts} <=> $a->{data}{diag}{ts} } @diags;

    $tt->Assign('diags', \@diags);

    return DoResponse($req, $tt, $AllVals);
}

sub DoResponse {
    my ($req, $tt, $AllVals) = @_;

    AssignIncomeVars($tt, $AllVals);
    AssignCommonVars($tt, $AllVals);

    $req->content_type('text/html; charset=utf-8');
    $req->send_http_header();

    $tt->Print;
    $tt->Dispose;
    return 200;
}

my @ALLOWED_AUTH_STATUSES = qw/successful disabled failed blocked ses_kill ses_create ses_update bruteforce challenge_shown/;
my @ALLOWED_AUTH_TYPES    = qw/unknown web autologin token xmpp calendar oauth oauthcheck oauthcreate/;

my %ALLOWED_AUTH_VAR_TO_TYPES = (
    unknown     => [qw/unknown/],
    web         => [qw/web/],
    autologin   => [qw/autologin/],
    token       => [qw/token/],
    xmpp        => [qw/xmpp/],
    mail        => [qw/smtp pop imap/],
    calendar    => [qw/calendar/],
    oauth       => [qw/oauth/],
    oauthtokens => [qw/oauthcheck oauthcreate/],
);

my @ALLOWED_AUTH_TYPES = map { @$_ } values %ALLOWED_AUTH_VAR_TO_TYPES;

sub DoUserDataSearch {
    my ($tt, $acd, $AllVals) = @_;

    my $uid = $AllVals->{'id_client'} || $AllVals->{'uid'};

    AssignResultVars($tt, {'show_search_results' => 1});

    my $account = eval { $acd->dbaf->GetAccountByUidExceptionable($uid) };

    if ($@) {
        return ShowError($tt, 'GetAccountByUid has been failed. Try again later.', $@);
    }

    if ($account) {
        AssignResultVars($tt, { account => $account });
    } elsif ($acd->dberror()) {
        ADM::Logs::IntErr($acd->dberror());
    }

    my $is_uid_long_removed = 0;
    if (not ADM::Utils::HaveGrant('show_deleted_users')) {
        eval {
            $is_uid_long_removed = CheckIfUidLongRemoved(time, $uid, $account, undef);
        };
        if ($@) {
            my $message = $@;
            $message =~ s/ at.*$//;
            return ShowError($tt, $message, $@);
        }
    }

    my $account_ft;
    if (not $is_uid_long_removed) {
        $account_ft = ADM::Requester::GetHistoryDb3UserinfoFT(uid => $uid);
    }
    if ($account_ft) {
        AssignResultVars($tt, { account_ft => $account_ft });
    } elsif ($acd->dberror()) {
        ADM::Logs::IntErr($acd->dberror());
    }

    my $event_from = $AllVals->{event_start_date} ? Global::DatetimeToTimestamp($AllVals->{event_start_date})          : undef;
    my $event_to   = $AllVals->{event_end_date}   ? Global::DatetimeToTimestamp("$AllVals->{event_end_date} 23:59:59") : undef;

    my $auth_start_ts = $AllVals->{auth_start_date} ? Global::DatetimeToTimestamp($AllVals->{auth_start_date})           : undef;
    my $auth_end_ts   = $AllVals->{auth_end_date}   ? Global::DatetimeToTimestamp("$AllVals->{auth_end_date} 23:59:59")  : undef;

    my ($auth_from, $auth_to);

    if (ADM::Utils::HaveGrant('show_full_auth_history')) {
        $auth_from = $auth_start_ts;
        $auth_to = $auth_end_ts;
    } else {
        my @today = Date::Calc::Today();

        my $one_year_ago = sprintf("%04d-%02d-%02d", Date::Calc::Add_Delta_YM(@today, -1, 0));
        my $one_year_ago_ts = Global::DatetimeToTimestamp($one_year_ago);
        my $end_of_today_ts = Global::DatetimeToTimestamp(sprintf("%04d-%02d-%02d 23:59:59", @today));

        $auth_from = $auth_start_ts > $one_year_ago_ts ? $auth_start_ts : $one_year_ago_ts;
        $auth_to   = $auth_end_ts   > $one_year_ago_ts ? $auth_end_ts   : $end_of_today_ts;
    }

    $AllVals->{auth_start_date} = Global::TimestampToLocalDate($auth_from);
    $AllVals->{auth_end_date} = Global::TimestampToLocalDate($auth_to);

    my $events = [];

    if ($AllVals->{s_reg} and $event_from and $event_to) {
        my $limit = 10000;
        $tt->Assign('EventsLimit', $limit);

        my $raw_events = [];

        if (not $is_uid_long_removed) {
            $raw_events = ADM::Requester::GetHistoryDb3Events(
                uid    => $uid,
                from   => $event_from,
                to     => $event_to,
                limit  => $limit,
            );
        }

        $events = ADM::Utils::JoinEventsById($raw_events);
        ADM::Utils::HideDeniedEvents($events);
        ADM::Utils::CopyAdminLoginAndCommentAsEvent($events);

        for my $item (@$events) {
            $item->{is_event} = 1;
            $item->{group}{ts} = $item->{real_ts};
            $item->{ip} = IpToIpRecord($item->{ip});
        }

        $tt->Assign('EventsNumber', scalar @$events);
    }

    my (@status_filter, %show_auth_status);
    for my $status (@ALLOWED_AUTH_STATUSES) {
        next if not $AllVals->{"auth_status_$status"} and not $AllVals->{auth_status_all_default};
        next if $AllVals->{auth_type_all_default} and $status eq 'failed';
        push @status_filter, $status;
        $show_auth_status{$status} = 1;
    }
    $show_auth_status{all} = 1
      if scalar @status_filter == scalar @ALLOWED_AUTH_STATUSES;

    my (@type_filter, %show_auth_type);
    while (my ($var, $types) = each %ALLOWED_AUTH_VAR_TO_TYPES) {
        next if not $AllVals->{"auth_type_$var"} and not $AllVals->{auth_type_all_default};
        for my $type (@$types) {
            push @type_filter, $type;
        }
        $show_auth_type{$var} = 1;
    }

    $show_auth_type{all} = 1
      if scalar @type_filter == scalar @ALLOWED_AUTH_TYPES;

    $tt->Assign('ShowAuthStatus', \%show_auth_status);
    $tt->Assign('ShowAuthType',   \%show_auth_type);

    $tt->Assign('GroupAuths',   $AllVals->{group_auths} || 'day');
    $tt->Assign('MergeHistory', $AllVals->{merge_history});

    my $auths = [];
    my $auths_summary;

    if (@status_filter and @type_filter and $auth_from and $auth_to) {
        my $limit
          = $AllVals->{group_auths} ne 'item'
          ? 5000
          : 1000
          ;
        $tt->Assign('AuthsLimit', $limit);

        if (not $is_uid_long_removed) {
            $auths = ADM::Utils::GetHistoryDb3Auths(
                uid    => $uid,
                from   => $auth_from,
                to     => $auth_to,
                status => [@status_filter],
                type   => [@type_filter],
                limit  => $limit,
            );
        }

        for my $item (@$auths) {
            $auths_summary->{ $item->{ip} }++;

            $item->{is_auth}       = 1;
            $item->{ip}            = IpToIpRecord($item->{ip});
            $item->{is_successful} = $item->{status_name} =~ m/^(?:successful|ses_create|ses_update)$/;
            $item->{is_money}      = $item->{hostid} =~ m/^04|05|0B|0C|24|25|2B|2C$/;
        }

        $tt->Assign('AuthsNumber', scalar @$auths);
    }

    if ($AllVals->{group_auths} ne 'item') {
        my $auths_by_key;
        my $group_id;

        for my $item (@$auths) {
            my $group;

            my ($year, $month, $day) = split /[-:\sT]+/, $item->{dt};
            if ($AllVals->{group_auths} eq 'week') {
                my $week_beginning     = sprintf '%04d-%02d-%02d',                            Date::Calc::Monday_of_Week(Date::Calc::Week_of_Year($year, $month, $day));
                my $week_ending        = sprintf '%04d-%02d-%02d', Date::Calc::Add_Delta_Days(Date::Calc::Monday_of_Week(Date::Calc::Week_of_Year($year, $month, $day)), 6);
                $group->{display_time} = join ' – ', $week_beginning, $week_ending;
                $group->{ts}           = Global::DatetimeToTimestamp($week_beginning);
            }
            else {
                $group->{display_time} = sprintf '%04d-%02d-%02d', $year, $month, $day;
                $group->{ts}           = Global::DatetimeToTimestamp($group->{display_time});
            }

            my $ip = $item->{ip}{ip};
            my $key = join '-', $group->{ts}, $ip, @$item{qw/type_name status_name/};
            my $parent = $auths_by_key->{$key} ||= { %$item };

            unless ($parent->{group}) {
                $parent->{group} = $group;
                $group->{id} = ++$group_id;
            }

            push @{ $parent->{children} }, $item;
        }

        my @auths_parents = values %$auths_by_key;
        for my $parent (@auths_parents) {
            my $children        = $parent->{children};
            my @sorted_children = sort { $a->{real_ts} <=> $b->{real_ts} } @$children;
            $parent->{children} = \@sorted_children;
        }

        $auths = \@auths_parents;
    }

    if ($AllVals->{merge_history}) {
        my $merged_history = [ sort { $a->{real_ts} <=> $b->{real_ts} } @$auths, @$events ];
        $tt->Assign('MergedHistory', $merged_history);
    }
    else {
        if (@$auths) {
            $auths  = [ sort { $a->{group}{ts} <=> $b->{group}{ts} or $a->{real_ts} <=> $b->{real_ts} } @$auths  ];
            AssignResultVars($tt, { historydb3_auths => $auths });
        }
        if (@$events) {
            $events = [ sort { $a->{group}{ts} <=> $b->{group}{ts} or $a->{real_ts} <=> $b->{real_ts} } @$events ];
            AssignResultVars($tt, { historydb3_events => $events });
        }
    }

    if ($auths_summary) {
        AssignResultVars($tt, { auth_list_summary => $auths_summary });
    }
}

sub IpToIpRecord {
    my $ip = shift;
    return undef unless $ip;
    return {
        ip => $ip,
        is_yandex => ADM::Utils::YandexNetwork($ip),
    };
}

sub DoIndexSearches {
    my ($tt, $acd, $AllVals) = @_;

    AssignResultVars($tt, { services => $acd->ListAllServices() });

    my $limit = ADM::Utils::GetLimit($AllVals->{limit});
    my $karma = ADM::Utils::GetKarma($AllVals->{karma});

    if (not ADM::Utils::HaveGrant('allow_search')) {
        return ShowError($tt, "you don't have 'allow_search' permission");
    }

    if ($AllVals->{phone} and not ADM::Utils::HaveGrant('allow_phone_search')) {
        return ShowError($tt, "you don't have 'allow_phone_search' permission");
    }

    if ($AllVals->{email} and not ADM::Utils::HaveGrant('allow_email_search')) {
        return ShowError($tt, "you don't have 'allow_email_search' permission");
    }

    my %filter = eval { $acd->ConvertRequestParamsToFilter({ %$AllVals, karma => $karma }) };

    if ($@) {
        my $message = $@;
        $message =~ s/ at.*$//;
        return ShowError($tt, $message);
    }

    return
      unless %filter;

    my $filter_dump = JSON::XS->new->utf8(0)->encode(\%filter);
    ADM::Logs::DeBug("Filter: $filter_dump");

    my $accounts = [];
    my $uids = [];
    my $login_by_removed_uid = {};
    my $found_number = 0;
    my $exact_search = 0;
    my $accounts_number = 0;

    my $uid  = $filter{uid};
    my $suid = $filter{suid};

    $uid  =~ s/\D//g;
    $suid =~ s/\D//g;

    # Точный поиск
    if ($uid or $suid) {
        my %args
          = $uid
          ? ($uid)
          : (undef, suid => $suid);

        my $account = eval { $acd->dbaf->GetAccountByUidExceptionable(%args) };

        return ShowError($tt, 'GetAccountByUid has been failed. Try again later.', $@)
          if $@;

        if ($uid) {
            $uids = [$uid];
        }
        elsif ($account and $account->uid) {
            $uids = [$account->uid];
        }

        if ($account) {
            $accounts     = [$account];
            $found_number = 1;
        }

        $exact_search = 1;
    }

    # Неточный поиск
    else {
        # Получаем идентификатор ПДД-домена по его имени, если в фильтрах есть ПДД-домен (нужно для поиска по текущим алиасам)
        if ($filter{pdd_domain} and not $filter{exclude_pdd}) {
            my $domains = $acd->dbaf->GetHostedDomains(domain_name => to_puny($filter{pdd_domain}));

            return ShowError($tt, 'GetHostedDomains has been failed. Try again later.')
              unless $domains;

            $filter{pdd_domain_id} = $domains->[0]{domain_id}
              if $domains->[0];
        }

        my $global_searcher = ADM::Core::GlobalSearcher->new;
        $global_searcher->acd($acd);
        $global_searcher->limit($limit);
        $global_searcher->filter(\%filter);

        eval {
            $global_searcher->search;
        };

        if ($@) {
            my $message = $@;
            $message =~ s/ at.*$//;
            return ShowError($tt, $message . "; try again later", $@);
        }

        if ($global_searcher->warning) {
            return ShowError($tt, $global_searcher->warning);
        }

        $accounts = $global_searcher->accounts;
        $uids     = $global_searcher->uids;
        $login_by_removed_uid = $global_searcher->login_by_removed_uid || {};
    }

    my %accounts_by_uid = map +( $_->uid => $_ ), @$accounts;

    my %events_by_uid;

    if (scalar @$uids <= 300) {
        for my $uid (@$uids) {
            my $events = ADM::Requester::GetHistoryDb3Events(
                uid => $uid,
                to  => time,
            );

            return ShowError($tt, 'GetHistoryDb3Events has been failed. Try again later.')
              unless $events;

            ADM::Utils::HideDeniedEvents($events);

            $events_by_uid{$uid} = $events;
        }
    }

    if (not ADM::Utils::HaveGrant('show_deleted_users')) {
        my $now = time;
        my %uids_to_hide;

        for my $uid (@$uids) {
            eval {
                $uids_to_hide{$uid} = 1
                  if CheckIfUidLongRemoved($now, $uid, $accounts_by_uid{$uid}, $events_by_uid{$uid});
            };
            if ($@) {
                my $message = $@;
                $message =~ s/ at.*$//;
                return ShowError($tt, $message, $@);
            }
        }

        $accounts = [ grep { not $uids_to_hide{$_->uid} } @$accounts ];
        $uids     = [ grep { not $uids_to_hide{$_}      } @$uids ];
    }

    # Получаем дополнительную информацию для найденных аккаунтов
    for my $account (@$accounts) {
        my $uid = $account->uid;

        my $subscription_list = $account->subscriptions->list;

        if ($account->is_pdd) {
            for my $pdd_alias (@{ $account->aliases->pddalias }) {
                my $subscription104 = Model::Subscription->new(
                    sid        => 105,
                    login      => $pdd_alias,
                    login_rule => 1,
                );
                push @$subscription_list, $subscription104;
            }
        }

        $subscription_list = [ sort { $a->sid <=> $b->sid or $a->login cmp $b->login } @$subscription_list ];

        $account->{subscription_list} = $subscription_list;

        my $events = $events_by_uid{$uid};

        if ($events and @$events) {
            my $subscription_created = ADM::Utils::ExtractSubscriptionsCreatedFromEvents($events);

            $subscription_created->{8}  ||= $account->registration_datetime;
            $subscription_created->{33} ||= $account->registration_datetime;
            $subscription_created->{58} ||= $account->registration_datetime;

            $account->{subscription_created} = $subscription_created;

            $account->{support_notes}        = ADM::Utils::ExtractSupportNotesFromEvents($events);
        }
    }

    if (scalar @$accounts < 50) {
        for my $account (@$accounts) {
            my $uid = $account->uid;

            my $friends_subscription = $account->subscriptions->get(26);

            if ($friends_subscription->is_exists) {
                $friends_subscription->{status} = ADM::Requester::GetFriendsServiceStatus($uid);
            }
        }
    }

    # Получаем регистрационную информацию по найденным uid'ам
    my $userinfos_ft = [];

    for my $uid (@$uids) {
        my $account     = $accounts_by_uid{$uid} || undef;
        my $events      = $events_by_uid{$uid}   || undef;

        my %userinfo_ft = (
            uid   => $uid,
            login => $account ? $account->human_readable_login : $login_by_removed_uid->{$uid},
        );

        push @$userinfos_ft, \%userinfo_ft;

        next unless $events;

        my $extracted_userinfo_ft = ADM::Utils::ExtractUserinfoFtFromEvents($events, $uid);

        next unless $extracted_userinfo_ft;

        %userinfo_ft = (%userinfo_ft, %$extracted_userinfo_ft);

        unless ($exact_search) {
            next if $filter{only_active}       and not $account;
            next if $filter{only_enabled}      and (not $account or not $account->is_enabled);
            next if $filter{exclude_pdd}       and $uid >= 1130000000000000;
            next if $filter{exclude_lite}      and $uid < 1130000000000000 and $userinfo_ft{login} =~ /\@/;
            next if $filter{registered_after}  and ($userinfo_ft{real_ts} / 1_000_000) < $filter{registered_after};
            next if $filter{registered_before} and ($userinfo_ft{real_ts} / 1_000_000) > $filter{registered_before};
        }

        $userinfo_ft{account} = $account;
    }

    # Сортировка от более свежих к старым
    $accounts     = [ sort { $b->uid   <=> $a->uid   } @$accounts     ];
    $userinfos_ft = [ sort { $b->{uid} <=> $a->{uid} } @$userinfos_ft ];

    AssignResultVars($tt, {
        accounts_count => $found_number,
        accounts       => $accounts,
        accounts_ft    => $userinfos_ft,
    });

    # TODO subscription.host_name =~ s/^([^.]+).*$/$1/

    # Если найден только один uid - получаем информацию по его событиям
    my $single_uid;

    my $accounts_number     = scalar @$accounts;
    my $userinfos_ft_number = scalar @$userinfos_ft;

    if (
           ($accounts_number == 1 and $userinfos_ft_number == 1 and $accounts->[0]->uid == $userinfos_ft->[0]->{uid}) # Найден один актуальный аккаунт и регистрационная информация для одного и того же uid'а
        or ($accounts_number == 1 and $userinfos_ft_number == 0)                                                      # Найден один актуальный аккаунт без регистрационной информации
        or ($accounts_number == 0 and $userinfos_ft_number == 1)                                                      # Найден регистрационная информация по одному uid'у без актуального аккаунта
    ) {
        $single_uid = $accounts_number ? $accounts->[0]->uid : $userinfos_ft->[0]->{uid};
    }

    if ($single_uid) {
        my $events = $events_by_uid{$single_uid};

        my %events_by_ts;
        my $user_adm_actions;

        for my $event (@$events) {
            next if $event->{name} eq 'userinfo_ft';

            push @{ $events_by_ts{ $event->{real_ts} } }, $event;
            Encode::_utf8_off($event->{value});
            utf8::decode($event->{value});
            if ($event->{admin}) {
                $user_adm_actions->{$event->{uid}} = 'admin_login';
            }
        }

        $events = ADM::Utils::JoinEventsById($events);
        ADM::Utils::CopyAdminLoginAndCommentAsEvent($events);

        $events = [ sort { $a->{ts} <=> $b->{ts} } @$events ];

        AssignResultVars($tt, { useradmactions => $user_adm_actions });
        AssignResultVars($tt, { admoperations  => $events           });
    }

}

sub CheckIfUidLongRemoved {
    my ($now, $uid, $account, $events) = @_;

    return 0
      if $account;

    my $one_year_ago = $now - 86400 * 365;

    unless ($events) {
        $events = ADM::Requester::GetHistoryDb3Events(
            uid  => $uid,
            from => $one_year_ago,
            to   => $now,
        );

        die "GetHistoryDb3Events has been failed. Try again later."
          unless $events;
    }

    my $delete_action = ADM::Utils::ExtractDeleteActionFromEvents($events);

    if (not $delete_action) {
        return 1;
    } elsif ($delete_action->{ts} < $one_year_ago) {
        return 1;
    } else {
        return 0;
    }
}

sub AssignIncomeVars {
    my $tt = shift;
    my $AllVals = shift;

    my %tt_vars = (
        Login       => 'login',
        Ena         => 'ena',
        Live        => 'live',
        NoLight     => 'nolight',
        NoPdd       => 'nopdd',

        IpFrom      => 'ip_from',
        Email       => 'email',
        IpProx      => 'ip_prox',
        Nickname    => 'nickname',
        BirthDate   => 'birth_date',
        RegDateFrom => 'reg_date_from',
        RegDateTo   => 'reg_date_to',
        Fio         => 'fio',
        Suid        => 'suid',
        Phone       => 'phone',
        Limit       => 'limit',
        Passwd      => 'passwd',
        SReg        => 's_reg',
        SAuth       => 's_auth',
        AllAuth     => 'all_auth',

        AuthStartDate => 'auth_start_date',
        AuthEndDate   => 'auth_end_date',
        EventStartDate => 'event_start_date',
        EventEndDate   => 'event_end_date',

        ShowFailedAuth => 'show_failed_auth',
    );

    while (my ($key, $value) = each(%tt_vars)) {
        $tt->Assign($key, $AllVals->{ $value });
    }

    $tt->Assign('Uid', $AllVals->{'uid'} || $AllVals->{'id_client'});

    $tt->Assign('AdminComment', $AllVals->{admin_comment});
    $tt->Assign('AdminLogin', $AllVals->{admin_login}) if $AllVals->{admin_login};
    $tt->Assign('Enable', $AllVals->{enable}) if defined $AllVals->{enable};
    $tt->Assign('Karma', $AllVals->{karma}) if defined $AllVals->{karma};
    $tt->Assign('FocusOnComment', $AllVals->{focus_on_comment}) if $AllVals->{focus_on_comment};
}

sub AssignResultVars {
    my $tt = shift;
    my $params = shift;

    my %tt_vars = (
        ShowSearchResults => 'show_search_results',
        AllServices       => 'services',

        Account           => 'account',
        Accounts          => 'accounts',
        AccountsCount     => 'accounts_count',

        AccountsFt        => 'accounts_ft',
        AccountsFtCount   => 'accounts_ft_count',

        Subscriptions     => 'subscriptions',

        UserAdmActions   => 'useradmactions',
        AdmOperations    => 'admoperations',
        Logins           => 'logins',
        AdminLogin       => 'admin_login',
        AdminComment     => 'admin_comment',

        Error            => 'error',
        ErrorMessage     => 'error_message',
        Status           => 'status',

        AccountFt       => 'account_ft',

        UdHalfYear      => 'ud_half_year',
        UdOneMonth      => 'ud_one_month',
        UdOneWeek       => 'ud_one_week',
        UdOneDay        => 'ud_one_day',
        UdOneYear       => 'ud_one_year',
        UdNow           => 'ud_now',
        UdLastAuth      => 'ud_last_auth',  # день последней авторизации
        UdLastWeek      => 'ud_last_week',  # за неделю до last auth
        UdLastDay       => 'ud_last_day',   # за день до last auth
        UdLastMonth     => 'ud_last_month', # месяц до последней авторизации
        UdLastHalf      => 'ud_last_half',  # 6 месяцев до последней авторизации
        UdLastYear      => 'ud_last_year',
        UdRegDate       => 'ud_reg_date',
        Changes         => 'changes',
        AuthList        => 'auth_list',
        AuthListSummary => 'auth_list_summary',

        NewChanges      => 'new_changes',
        NewAuthList     => 'new_auth_list',

        HistoryDb3Auths  => 'historydb3_auths',
        HistoryDb3Events => 'historydb3_events',
        HistoryDb3Merged => 'historydb3_merged',

        SReg  => 's_reg',
        SAuth => 's_auth',

        Logins => 'disable_logins',
        Uids   => 'disable_uids',
        RequestUri => 'request_uri',
    );

    while (my ($key, $value) = each(%tt_vars)) {
        $tt->Assign($key, $params->{ $value })
            if $params->{ $value };
    }

    $tt->Assign('Enable', $params->{enable}) if defined $params->{enable};
    $tt->Assign('Karma',  $params->{karma} ) if defined $params->{karma};
}

sub AssignCommonVars {
    my $tt = shift;
    my $params = shift;

    $tt->Assign('HaveRole',            sub { ADM::Utils::HaveRole(@_)            });
    $tt->Assign('HaveGrant',           sub { ADM::Utils::HaveGrant(@_)           });
    $tt->Assign('HaveAtLeastOneGrant', sub { ADM::Utils::HaveAtLeastOneGrant(@_) });

    my $l10n_global = sub {
        $Global::I18n->l10n('en', @_);
    };
    $tt->Assign('l10n', $l10n_global);
    $tt->Assign('L',    $l10n_global);

    my $is_protected_sid = sub {
        my $sid = shift;
        my $protected = $admin::Conf->GetHCVal('Protected');
        return $protected->{$sid} ? 1 : 0;
    };

    my $was_protected_sid = sub {
        my $sid = shift;
        my $protected = $admin::Conf->GetHCVal('Protected');
        return (not $protected->{$sid} and defined $protected->{$sid}) ? 1 : 0;
    };

    my $is_money_allowed_sid = sub {
        my $sid = shift;
        return $sid =~ /^ (?: 2|8|14|15|16|20 ) $/x ? 1 : 0;
    };

    my $ts2localdt = sub {
        my $ts = shift;
        my $result = Global::TimestampToLocalDatetime($ts);
        return $result;
    };

    my $ts2localt = sub {
        my $ts = shift;
        my $result = Global::TimestampToLocalDatetime($ts);
        $result =~ s/^.*? //;
        return $result;
    };

    my $s2hms = sub {
        my $s = shift;

        my $hh = int($s / 3600);
        my $mm = int(($s % 3600) / 60);
        my $ss = int($s % 60);

        my $result;
        $result .= "${hh}h" if $hh;
        $result .= "${mm}m" if $mm;

        if ($result) {
            $result .= "${ss}s";
        }
        else {
            $result = sprintf '%.1fs', $s;
        }

        return $result;
    };

    $tt->Assign('IsProtectedSid',    $is_protected_sid);
    $tt->Assign('WasProtectedSid',   $was_protected_sid);
    $tt->Assign('IsMoneyAllowedSid', $is_money_allowed_sid);
    $tt->Assign('ts2localdt',        $ts2localdt);
    $tt->Assign('ts2localt',         $ts2localt);
    $tt->Assign('s2hms',             $s2hms);
}

sub AssignHashVars {
    my $tt = shift;
    my $params = shift;

    foreach my $k (keys %$params) {
        $tt->Assign($k, $params->{$k});
    }
}

sub ShowError {
    my ($tt, $external, $internal) = @_;

    $internal ||= '';

    AssignResultVars($tt, { error_message => $external });
    ADM::Logs::IntErr("$external; $internal")
      if $internal;

    return;
}

1;
