package ADM::Guard;

use common::sense;

use Data::Dumper;
use JSON::XS;

my $MIN_SECONDS_BETWEEN_STAT_CHECK = 0;

sub new { bless {}, shift }

sub load_file {
    my $self = shift;
    my ($filepath) = @_;

    my $time       = time;
    my $last_mtime = $self->{last_mtime} || 0;
    my $last_check = $self->{last_check} || 0;
    my $delta      = $time - $last_check;

    return if $delta <= $MIN_SECONDS_BETWEEN_STAT_CHECK;

    ADM::Logs::DeBug("guard: filepath=$filepath time=$time last_mtime=$last_mtime last_check=$last_check delta=$delta>$MIN_SECONDS_BETWEEN_STAT_CHECK");

    die "filepath $filepath doesn't exist"
      unless -e $filepath;

    my $current_mtime = (stat $filepath)[9];

    ADM::Logs::DeBug("guard: current_mtime=$current_mtime");

    $self->{last_check} = $time;

    return if $last_mtime == $current_mtime;

    $self->{last_mtime} = $current_mtime;

    ADM::Logs::DeBug("guard: current_mtime != last_mtime; load content");

    my $content = do { local(@ARGV, $/) = $filepath; <> };
    my $data    = decode_json $content;

    $self->load_data($data);

    return 1;
}

sub load_data {
    my $self = shift;
    my ($data) = @_;

    my $users  = $data->{users};
    my $roles  = $data->{roles};
    my $grants = $data->{grants};

    die "no users"  unless @$users;
    die "no roles"  unless @$roles;
    die "no grants" unless @$grants;

    my %users  = map +( lc $_->{username} => $_ ), @$users;
    my %roles  = map +( $_->{id}          => $_ ), @$roles;
    my %grants = map +( $_->{id}          => $_ ), @$grants;

    my %grant_id_by_name = map +( $_->{name} => $_->{id} ), @$grants;
    my %role_id_by_name  = map +( $_->{name} => $_->{id} ), @$roles;

    while (my ($login, $user_data) = each %users) {
        my $user_roles = $user_data->{roles};
        my %user_roles_hash = map { $_ => 1 } @$user_roles;
        my %user_grants_hash;

        for my $role (@$user_roles) {
            my $role_data = $roles{$role} || {};
            die "unknown role $role for user $login"
              unless %$role_data;

            my $role_grants_list = $role_data->{grants} || [];
            die "no grants for role $role"
              unless @$role_grants_list;

            my $role_grants_hash;

            for my $grant_id (@$role_grants_list) {
                die "unknown grant $grant_id in role $role for user $login"
                  unless $grants{$grant_id};
                $role_grants_hash->{$grant_id} = 1;
            }

            %user_grants_hash = ( %user_grants_hash, %$role_grants_hash );
        }

        $user_data->{grants} = \%user_grants_hash;
        $user_data->{roles}  = \%user_roles_hash;
        $user_data->{roles_string} = join ',', map { $roles{$_}->{name} } @$user_roles;
    }

    $self->{users}  = \%users;
    $self->{roles}  = \%roles;
    $self->{grant}  = \%grants;

    $self->{grant_id_by_name} = \%grant_id_by_name;
    $self->{role_id_by_name}  = \%role_id_by_name;
}

sub is_user_have_grant {
    my $self = shift;
    my ($login, $grant) = @_;

    die "unknown grant $grant"
      unless my $grant_id = $self->{grant_id_by_name}{$grant};

    my $user_data = $self->{users}{$login} || undef;

    return 0 unless $user_data;
    return 1 if $user_data->{grants}{$grant_id};
    return 0;
}

sub is_user_have_role {
    my $self = shift;
    my ($login, $role) = @_;

    die "unknown role $role"
      unless my $role_id = $self->{role_id_by_name}{$role};

    my $user_data = $self->{users}{$login} || undef;

    return 0 unless $user_data;
    return 1 if $user_data->{roles}{$role_id};
    return 0;
}

sub is_user_allowed {
    my $self = shift;
    my ($login) = @_;

    return 1 if $self->{users}{$login};
    return 0;
}

sub get_user_roles_string {
    my $self = shift;
    my ($login) = @_;

    my $user_data = $self->{users}{$login} || undef;

    return '' unless $user_data;
    return $user_data->{roles_string};
}

1;
