package ADM::Requester;

use strict;
use utf8;
use open qw(:std :utf8);

use Encode qw(encode decode);
use JSON::XS;
use LWP::UserAgent;
use XML::Bare;
use URI;
use URI::QueryParam;
use Tie::IxHash;

my $ua = LWP::UserAgent->new;
$ua->timeout(5);

sub _GetContentByUrl {
    my $url = shift;

    my $response = $ua->get($url);
    unless ($response->is_success) {
        my $error_details = $response->status_line || 'Unrecognized HTTP error';
        Common::Logs::IntErr("Http error while requesting $url: $error_details");
        return;
    }

    my $result = $response->content;
    unless ($result) {
        Common::Logs::IntErr("Empty response from $url");
        return;
    }

    return $result;
}

sub _GetContentFromUrl {
    my ($url, $post, %headers) = @_;

    my $response
      = $post
      ? $ua->post($url, $post, %headers)
      : $ua->get($url, %headers);

    unless ($response->is_success) {
        my $error_details = $response->status_line || 'Unrecognized HTTP error';
        Common::Logs::IntErr("Http error while requesting $url: $error_details");
        return;
    }

    my $result = $response->content;
    unless ($result) {
        Common::Logs::IntErr("Empty response from $url");
        return;
    }

    return $result;
}

sub DownloadUrl {
    my ($url, %args) = @_;

    my $retries = $args{retries} || 0;

    my $result;

    for my $attempt (1 .. $retries + 1) {
        $result = _GetContentByUrl($url);
        last if $result;
    }

    return $result;
}

sub DropPhone {
    my ($uid, $phoneid) = @_;

    my $url = URI->new($admin::Conf->GetVal('phone_validator_url'));
    $url->path('/api/dropphone');
    $url->query_form(
        sender  => 'adminka',
        uid     => $uid,
        phoneid => $phoneid,
    );

    my $response = $ua->get($url);

    unless ($response->is_success) {
        Common::Logs::IntErr("DropPhone: invalid json content on $url");
        return;
    }

    my $result = eval { decode_json $response->content };
    if ($@) {
        Common::Logs::IntErr("DropPhone: invalid json content on $url");
        return;
    }

    unless ($result->{status} eq 'OK') {
        Common::Logs::DeBug("DropPhone: invalid status $result->{status} on $url");
        return 0;
    }

    return 1;
}

sub SmsRouting {
    my (%args) = @_;

    my %query = (
        sender  => 'adminka',
        action  => $args{action},
    );

    if ($args{number}) {
        $query{number} = $args{number};
        $query{route}  = $args{route};
    }

    my $url = URI->new($admin::Conf->GetVal('phone_validator_url'));
    $url->path('/routing');
    $url->query_form(%query);

    my $response = $ua->get($url);

    unless ($response->is_success) {
        Common::Logs::IntErr("SmsRouting: bad response $url: ". $response->status_line);
        return;
    }

    my $result = eval { decode_json $response->content };
    if ($@) {
        Common::Logs::IntErr("SmsRouting: invalid json content on $url");
        return;
    }

    if ($result->{error}) {
        Common::Logs::DeBug("SmsRouting: error $result->{error} on $url");
        return 0;
    }

    return $result;
}

sub SendSms {
    my (%args) = @_;

    my %query = (
        sender  => 'adminka',
        number  => $args{number},
        gate_id => $args{gate_id},
        text    => $args{text},
    );

    my $url = URI->new($admin::Conf->GetVal('phone_validator_url'));
    $url->path('/sendsms');
    $url->query_form(%query);

    my $response = $ua->get($url);

    unless ($response->is_success) {
        Common::Logs::IntErr("SendSms: invalid json content on $url");
        return;
    }

    my $xml;
    eval {
        $xml = XML::Bare->new(text => $response->content)->parse;
    };

    if ($@) {
        Common::Logs::IntErr("SendSms: invalid xml content on $url");
        return;
    }

    if ($xml->{doc}{error}) {
        Common::Logs::IntErr("SendSms: error=$xml->{doc}{errorcode}{value} on $url: $xml->{doc}{error}{value}");
        return;
    }

    my $smsid = $xml->{doc}{'message-sent'}{id}{value};

    unless ($smsid) {
        Common::Logs::IntErr("SendSms: no smsid on $url");
        return;
    }

    return $smsid;
}

sub SetDefaultPhone {
    my (%args) = @_;

    my %query = (
        consumer => 'adminka',
    );

    my $post = {
        uid      => $args{uid},
        phone_id => $args{phone_id},
    };

    my %headers = (
        'Ya-Consumer-Client-Ip' => $args{ip},
        'Ya-Consumer-Client-Scheme' => 'https',
    );

    my $url = URI->new('http://' . $admin::Conf->GetVal('passport_backend_host'));
    $url->path('/1/bundle/phone/manage/set_default/');
    $url->query_form(%query);

    my $content = eval { _GetContentFromUrl($url, $post, %headers) };

    if ($@) {
        Common::Logs::IntErr("SetDefaultPhone: cant get content: $@");
        return;
    }

    my $data = eval { decode_json $content };
    if ($@) {
        Common::Logs::IntErr("SetDefaultPhone: invalid json content on $url");
        return;
    }

    my $error = $data->{errors}[0] || '';
    unless ($data->{status} eq 'ok') {
        Common::Logs::IntErr("SetDefaultPhone: not ok. error=$error");
        return;
    }

    unless ($data->{status} eq 'ok') {
        Common::Logs::DeBug("SetDefaultPhone: invalid status $data->{status} on $url");
        return 0;
    }

    return 1;
}

sub CheckTeamSessionId {
    my (%args) = @_;

    my %query;
    tie %query, 'Tie::IxHash';

    $query{method}    = 'sessionid';
    $query{format}    = 'json';
    $query{host}      = $args{host};
    $query{sessionid} = $args{sessionid};
    $query{userip}    = $args{ip};

    %query = (%query, @{ $args{queries} });

    my $url = URI->new($admin::Conf->GetVal('team_blackbox_url'));
    $url->query_form(%query);

    return
      unless my $content = DownloadUrl($url, retries => 2);

    my $decoded = eval { decode_json $content };
    if ($@) {
        Common::Logs::IntErr("CheckTeamSessionId: invalid json content on $url");
        return;
    }

    my $error = $decoded->{error};
    unless ($decoded->{status}{value}) {
        Common::Logs::IntErr("CheckTeamSessionId: error response. error=$error");
        return;
    }

    my $result = $decoded;

    return $result;
}

sub ProveKeyDiag {
    my (%args) = @_;

    my %query;
    tie %query, 'Tie::IxHash';

    $query{method}          = 'prove_key_diag';
    $query{format}          = 'json';
    $query{uid}             = $args{uid};
    $query{secret_salt}     = $args{secret_salt};
    $query{secret_id}       = $args{secret_id};
    $query{pin_secret_salt} = $args{pin_secret_salt};
    $query{pin_secret_id}   = $args{pin_secret_id};
    $query{totp_salt}       = $args{totp_salt};
    $query{totp_id}         = $args{totp_id};
    $query{timestamp}       = $args{timestamp};
    $query{skew}            = $args{skew};

    my $url = URI->new($admin::Conf->GetVal('blackbox_url'));
    $url->query_form(%query);

    return
      unless my $content = DownloadUrl($url, retries => 2);

    my $decoded = eval { decode_json $content };
    if ($@) {
        Common::Logs::IntErr("ProveKeyDiag: invalid json content on $url");
        return;
    }

    my $error = $decoded->{error};
    if ($error) {
        Common::Logs::IntErr("ProveKeyDiag: error response. error=$error");
        return;
    }

    my $result = $decoded;

    $result->{totp_id}{is_correct} = delete $result->{is_totp_id_correct}
      if exists $result->{is_totp_id_correct};
    $result->{totp_id}{correct_timestamp} = delete $result->{correct_totp_id_timestamp}
      if exists $result->{correct_totp_id_timestamp};

    $result->{pin_secret_id}{is_correct} = delete $result->{is_pin_secret_id_correct}
      if exists $result->{is_pin_secret_id_correct};

    $result->{secret_id}{is_correct} = delete $result->{is_secret_id_correct}
      if exists $result->{is_secret_id_correct};

    return $result;
}

sub FindAccountsByPhone {
    my ($number) = @_;

    my %query = (
        method  => 'phone_bindings',
        format  => 'json',
        type    => 'all',
        numbers => $number,
    );

    my $url = URI->new($admin::Conf->GetVal('blackbox_url'));
    $url->query_form(%query);

    return
      unless my $content = DownloadUrl($url, retries => 2);

    my $decoded = eval { decode_json $content };
    if ($@) {
        Common::Logs::IntErr("FindAccountsByPhone: invalid json content on $url");
        return;
    }

    my $error = $decoded->{error};
    if ($error) {
        Common::Logs::IntErr("FindAccountsByPhone: error response. error=$error");
        return;
    }

    # Нужны только активные, текущие привязки (type=current), а также отвязанные телефоны, но пока что присутствующее на аккаунте (type=unbound).
    # Удалённые привязки (type=history) исключаем.
    my @current_and_unbound = grep { $_->{type} =~ /^(?:current|unbound)$/ } @{ $decoded->{phone_bindings} };
    my @uids = map { $_->{uid} } @current_and_unbound;

    return \@uids;
}

sub FindAccountsByEmail {
    my ($email) = @_;

    my %query = (
        method => 'email_bindings',
        format => 'json',
        email  => $email,
    );

    my $url = URI->new($admin::Conf->GetVal('blackbox_url'));
    $url->query_form(%query);

    return
      unless my $content = DownloadUrl($url, retries => 2);

    my $decoded = eval { decode_json $content };
    if ($@) {
        Common::Logs::IntErr("FindAccountsByEmail: invalid json content on $url");
        return;
    }

    my $error = $decoded->{error};
    if ($error) {
        Common::Logs::IntErr("FindAccountsByEmail: error response. error=$error");
        return;
    }

    return $decoded->{uids};
}

sub GetFriendsServiceStatus {
    my $uid = shift;

    return unless $uid;

    my $url = $admin::Conf->GetVal('friends_service_status_url');
    $url =~ s/%uid%/$uid/;

    return
      unless my $content = _GetContentByUrl($url);

    return 'unknown' unless $content;

    my $xml = XML::Bare->new(text => $content)->parse;

    return $xml->{user}->{source}->{status}->{value} || 'unknown';
}

sub CreateRestorationEmailUrl {
    my %args = @_;

    my %query = (
        consumer => 'adminka',
    );

    my $post = {
        uid        => $args{uid},
        link_type  => $args{type},
        admin_name => $args{admin},
    };

    my %headers = (
        'Ya-Consumer-Client-Ip' => $args{ip},
        'Ya-Consumer-Client-Scheme' => 'https',
        'Ya-Client-Host'       => $args{host},
    );

    my $url = URI->new('http://' . $admin::Conf->GetVal('passport_backend_host'));
    $url->path('/1/bundle/restore/create_link/');
    $url->query_form(%query);

    my $content = eval { _GetContentFromUrl($url, $post, %headers) };

    if ($@) {
        Common::Logs::IntErr("CreateRestorationEmailUrl: cant get content: $@");
        return;
    }

    my $data = eval { decode_json $content };
    if ($@) {
        Common::Logs::IntErr("CreateRestorationEmailUrl: invalid json. content=$content");
        return;
    }

    my $error = $data->{errors}[0] || '';
    unless ($data->{status} eq 'ok') {
        Common::Logs::IntErr("CreateRestorationEmailUrl: not ok. error=$error");
        return;
    }

    unless ($data->{secret_link}) {
        Common::Logs::IntErr("CreateRestorationEmailUrl: no secret_link. content=$content");
        return;
    }

    return $data->{secret_link};
}

sub UpdatePasswordOptions {
    my %args = @_;

    my %query = (
        consumer => 'adminka',
    );

    my $post = {
        admin_name => $args{admin},
        comment    => $args{comment},
    };

    for my $option (qw/is_changing_required/) {
        next unless defined $args{$option};
        $post->{$option} = $args{$option};
    }

    my %headers = (
        'Ya-Consumer-Client-Ip' => $args{ip},
        'Ya-Consumer-Client-Scheme' => 'https',
    );

    my $url = URI->new('http://' . $admin::Conf->GetVal('passport_backend_host'));
    $url->path("/2/account/$args{uid}/password_options/");
    $url->query_form(%query);

    my $content = eval { _GetContentFromUrl($url, $post, %headers) };

    if ($@) {
        Common::Logs::IntErr("UpdatePasswordOptions: cant get content: $@");
        return;
    }

    my $data = eval { decode_json $content };
    if ($@) {
        Common::Logs::IntErr("UpdatePasswordOptions: invalid json. content=$content");
        return;
    }

    my $error = $data->{errors}[0] || '';
    unless ($data->{status} eq 'ok') {
        Common::Logs::IntErr("UpdatePasswordOptions: not ok. error=$error");
        return;
    }

    return 1;
}

sub DropEmail {
    my %args = @_;

    my %query = (
        consumer => 'adminka',
    );

    my $post = {
        uid        => $args{uid},
        email      => $args{email},
        admin_name => $args{admin},
        comment    => $args{comment},
    };

    my %headers = (
        'Ya-Consumer-Client-Ip' => $args{ip},
        'Ya-Consumer-Client-Scheme' => 'https',
    );

    my $url = URI->new('http://' . $admin::Conf->GetVal('passport_backend_host'));
    $url->path("/1/bundle/email/delete_by_admin/");
    $url->query_form(%query);

    my $content = eval { _GetContentFromUrl($url, $post, %headers) };

    if ($@) {
        Common::Logs::IntErr("DropEmail: cant get content: $@");
        return;
    }

    my $data = eval { decode_json $content };
    if ($@) {
        Common::Logs::IntErr("DropEmail: invalid json. content=$content");
        return;
    }

    my $error = $data->{errors}[0] || '';
    unless ($data->{status} eq 'ok') {
        Common::Logs::IntErr("DropEmail: not ok. error=$error");
        return;
    }

    return 1;
}

sub UpdateAccountOptions {
    my %args = @_;

    my %query = (
        consumer => 'adminka',
    );

    my $post = {
        admin_name => $args{admin},
        comment    => $args{comment},
    };

    for my $option (qw/is_shared/) {
        next unless defined $args{$option};
        $post->{$option} = $args{$option};
    }

    my %headers = (
        'Ya-Consumer-Client-Ip' => $args{ip},
        'Ya-Consumer-Client-Scheme' => 'https',
    );

    my $url = URI->new('http://' . $admin::Conf->GetVal('passport_backend_host'));
    $url->path("/1/account/$args{uid}/options/");
    $url->query_form(%query);

    my $content = eval { _GetContentFromUrl($url, $post, %headers) };

    if ($@) {
        Common::Logs::IntErr("UpdateAccountOptions: cant get content: $@");
        return;
    }

    my $data = eval { decode_json $content };
    if ($@) {
        Common::Logs::IntErr("UpdateAccountOptions: invalid json. content=$content");
        return;
    }

    my $error = $data->{errors}[0] || '';
    unless ($data->{status} eq 'ok') {
        Common::Logs::IntErr("UpdateAccountOptions: not ok. error=$error");
        return;
    }

    return 1;
}

sub GetHistoryDb3Auths {
    my (%args) = @_;

    my $url = URI->new('http://' . $admin::Conf->GetVal('historydb3_proxy'));

    if ($args{good}) {
        $url->path('/2/auths/');
    }
    elsif ($args{bad}) {
        $url->path('/2/auths/failed/');
    }

    $url->query_form(
        uid     => $args{uid},
        from_ts => $args{from},
        to_ts   => $args{to},
        order_by => 'asc',
        consumer => 'adminka',
    );

    $url->query_param(status => join ',', @{ $args{status} })
      if $args{status};

    $url->query_param(type => join ',', @{ $args{type} })
      if $args{type};

    $url->query_param(client_name => $args{source})
      if $args{source};

    $url->query_param(limit => $args{limit})
      if $args{limit};

    return
      unless my $content = _GetContentByUrl($url);

    my $data = eval { decode_json $content };
    if ($@) {
        Common::Logs::IntErr("GetHistoryDb3Auths: invalid json content on $url");
        return;
    }

    unless ($data->{status} eq 'ok') {
        Common::Logs::IntErr("GetHistoryDb3Auths: error=$data->{errors}[0]{message} on url=$url");
        return;
    }

    my $result = $data->{auths};

    for my $item (@$result) {
        $item->{type_name}   = delete $item->{type};
        $item->{status_name} = delete $item->{status};
        $item->{hostid}      = sprintf '%02X', delete $item->{host_id};
        $item->{ts}          = delete $item->{timestamp};
        $item->{real_ts}     = sprintf '%d', $item->{ts} * 1000000;
        $item->{dt}          = Global::TimestampToLocalDatetime($item->{ts});
        $item->{ip}          = delete $item->{user_ip};
    }

    return $result;
}

sub GetHistoryDb3Events {
    my (%args) = @_;

    my $url = URI->new('http://' . $admin::Conf->GetVal('historydb3_proxy'));
    $url->path('/2/events/');
    $url->query_form(
        consumer => 'adminka',
        uid     => $args{uid},
        from_ts => $args{from} || 0,
        to_ts   => $args{to}   || 0xFFFFFFFF,
        limit   => $args{limit} || 10_000,
        order_by => 'asc',
    );

    $url->query_param(name => $args{name})
      if $args{name};

    return
      unless my $content = _GetContentByUrl($url);

    my $data = eval { decode_json $content };
    if ($@) {
        Common::Logs::IntErr("GetHistoryDb3Events: invalid json content on $url");
        return;
    }

    unless ($data->{status} eq 'ok') {
        Common::Logs::IntErr("GetHistoryDb3Events: error=$data->{errors}[0]{message} on url=$url");
        return;
    }

    my $result = $data->{events};

    for my $item (@$result) {
        $item->{hostid}      = sprintf '%02X', $item->{host_id};
        $item->{ts}          = delete $item->{timestamp};
        $item->{real_ts}     = sprintf '%d', $item->{ts} * 1000000;
        $item->{dt}          = Global::TimestampToLocalDatetime($item->{ts});
        $item->{ip}          = delete $item->{user_ip};
        $item->{id}          = join '-', @$item{qw/hostid real_ts/};
    }

    return $result;
}

sub GetHistoryDb3UserinfoFT {
    my (%args) = @_;

    # TODO ограничивать область поиска времени регистрации +- 5 секунд
    my $events = GetHistoryDb3Events(
        uid => $args{uid},
    );

    return ADM::Utils::ExtractUserinfoFtFromEvents($events, $args{uid});
}

sub GetHistoryDb3SupportNotes {
    my (%args) = @_;

    my $events = GetHistoryDb3Events(
        %args,
        name => 'support_note',
    );

    return unless $events;
    return $events;
}

sub GetHistoryDb3MailboxHistory {
    my (%args) = @_;

    my $url = URI->new('http://' . $admin::Conf->GetVal('historydb3_proxy'));
    $url->path('/2/events/by_suid/');
    $url->query_form(
        consumer => 'adminka',
        suid     => $args{suid},
        from_ts  => $args{from} || 0,
        to_ts    => $args{to}   || 0xFFFFFFFF,
    );

    return
      unless my $content = _GetContentByUrl($url);

    my $data = eval { decode_json $content };
    if ($@) {
        Common::Logs::IntErr("GetHistoryDb3MailboxHistory: invalid json content on $url");
        return;
    }

    unless ($data->{status} eq 'ok') {
        Common::Logs::IntErr("GetHistoryDb3MailboxHistory: error=$data->{errors}[0]{message} on url=$url");
        return;
    }

=pod
{
        "events": [
        {
            "client_name": "passport", 
            "date": "2014-03-14 13:07:08", 
            "host_id": "127", 
            "mail_host_id": "99", 
            "name": "mail.add", 
            "timestamp": 1394788028.641917, 
            "user_ip": "8.8.8.8"
        }, 
        {
            "client_name": "passport", 
            "date": "2014-03-14 13:08:08", 
            "host_id": "127", 
            "mail_host_id": "1000", 
            "name": "mail.upd", 
            "timestamp": 1394788088.641917, 
            "user_ip": "8.8.8.8"
        }, 
        {
            "client_name": "passport", 
            "date": "2014-03-14 13:10:08", 
            "host_id": "127", 
            "mail_host_id": "1000", 
            "name": "mail.rm", 
            "timestamp": 1394788208.641917, 
            "user_ip": "8.8.8.8"
        }
            ], 
    "status": "ok", 
    "suid": 6664455
}
=cut

    my $result = $data->{events};

    $result = [ sort { $a->{timestamp} <=> $b->{timestamp} } @$result ];

    return $result;
}

sub GetHistoryDb3Lastauth {
    my (%args) = @_;

    my $url = URI->new('http://' . $admin::Conf->GetVal('historydb3_proxy'));
    $url->path('/2/lastauth/');
    $url->query_form(
        consumer => 'adminka',
        uid      => $args{uid},
    );

    return
      unless my $content = _GetContentByUrl($url);

    my $data = eval { decode_json $content };
    if ($@) {
        Common::Logs::IntErr("GetHistoryDb3Lastauth: invalid json content on $url");
        return;
    }

    unless ($data->{status} eq 'ok') {
        Common::Logs::IntErr("GetHistoryDb3Lastauth: error=$data->{errors}[0]{message} on url=$url");
        return;
    }

    my $result = $data->{lastauth}{timestamp};

    return $result;
}

sub GetHistoryDb3RegistrationsByIp {
    my (%args) = @_;

    $args{ip} =~ s{\.\*$}{.0\/24}g;

    my $url = URI->new('http://' . $admin::Conf->GetVal('historydb3_proxy'));
    $url->path('/2/events/by_ip/registrations/');
    $url->query_form(
        consumer => 'adminka',
        subnet   => $args{ip},
        from_ts  => $args{from} || 0,
        to_ts    => $args{to}   || 0xFFFFFFFF,
    );

    return
      unless my $content = _GetContentByUrl($url);

=pod
      {
          "ip": "77.75.158.121", 
          "registrations": [
                               {
                                       'uid' => '3000455898',
                                       'host_id' => '126',
                                       'date' => '2014-04-11 12:07:22',
                                       'value' => 'account_register_admimportreg',
                                       'name' => 'action',
                                       'yandexuid' => '2429369391397203642',
                                       'timestamp' => '1397203642.27312',
                                       'client_name' => 'passport'
                                     },
          ], 
          "status": "ok"
      }
=cut

    my $data = eval { decode_json $content };
    if ($@) {
        Common::Logs::IntErr("GetHistoryDb3RegistrationsByIp: invalid json content on $url");
        return;
    }

    unless ($data->{status} eq 'ok') {
        Common::Logs::IntErr("GetHistoryDb3RegistrationsByIp: error=$data->{errors}[0]{message} on url=$url");
        return;
    }

    my $result = $data->{registrations} || [];

    return $result;
}

sub GetHistoryDb3AuthsStatisticsExtended {
    my (%args) = @_;

    my $url = URI->new('http://' . $admin::Conf->GetVal('historydb3_proxy'));
    $url->path('/1/auths/statistics/extended');
    $url->query_form(
        consumer => 'adminka',
        uid      => $args{uid},
        user_ip  => $args{ip},
        from_ts  => $args{from} || 0,
        to_ts    => $args{to}   || 0xFFFFFFFF,
    );

    return
      unless my $content = _GetContentByUrl($url);

    unless ($content) {
        Common::Logs::IntErr("GetHistoryDb3AuthsStatisticsExtended: empty content url=$url");
        return;
    }

    my $result = $content;

    return $result;
}

sub GetSmsHistory {
    my (%args) = @_;

    my $url = URI->new('http://' . $admin::Conf->GetVal('historydb3_proxy'));

    my %query = (
        consumer => 'adminka',
        from_ts  => $args{from} || 0,
        to_ts    => $args{to}   || 0xFFFFFFFF,
    );

    if ($args{phone_number}) {
        $url->path('/yasms/2/sms/by_phone/');
        $query{phone} = $args{phone_number};
    }

    if ($args{sms_id}) {
        $url->path('/yasms/2/sms/by_globalid/');
        $query{global_smsid} = $args{sms_id};
    }

    $url->query_form(%query);

    return
      unless my $content = _GetContentByUrl($url);

    my $data = eval { decode_json $content };
    if ($@) {
        Common::Logs::IntErr("GetSmsHistory: invalid json content on $url");
        return;
    }

    unless ($data->{status} eq 'ok') {
        Common::Logs::IntErr("GetSmsHistory: error=$data->{errors}[0]{message} on url=$url");
        return;
    }

    my $result = $data;

    return $result;
}

# events?uid=11804317
# events?uid=1130000000030053
# userinfo_ft?uid=3000459118 (with account_created)
# userinfo_ft?uid=11804317
# support_notes?uid=39581
# auth?uid=1130000000030053
# auth?uid=1130000000017705
# suid_history?suid=6664455

1;

