package ADM::SodiumSeal;

use common::sense;

use Crypt::NaCl::Sodium ':utils'; # https://github.com/jedisct1/libsodium/

my $crypto_box         = Crypt::NaCl::Sodium->box;
my $crypto_hash        = Crypt::NaCl::Sodium->hash;
my $crypto_generichash = Crypt::NaCl::Sodium->generichash;
my $crypto_shorthash   = Crypt::NaCl::Sodium->shorthash;

my $SEALBYTES = $crypto_box->PUBLICKEYBYTES + $crypto_box->MACBYTES;

sub _crypto_box_seal_nonce {
    my ($pk1, $pk2) = @_;

    $pk1 = substr $pk1, 0, $crypto_box->PUBLICKEYBYTES;
    $pk2 = substr $pk2, 0, $crypto_box->PUBLICKEYBYTES;

    my $st = $crypto_generichash->init(bytes => $crypto_box->NONCEBYTES);
    $st->update($pk1, $pk2);
    my $nonce = $st->final(bytes => $crypto_box->NONCEBYTES);

    return $nonce;
}

sub crypto_box_detached {
    my ($m, $n, $pk, $sk) = @_;

    my $k         = $crypto_box->beforenm($pk, $sk);
    my ($mac, $c) = $crypto_box->encrypt_afternm($m, $n, $k);
    $c            = $mac . $c;

    return $c;
}

sub crypto_box_easy {
    my ($m, $n, $pk, $sk) = @_;

    return crypto_box_detached($m, $n, $pk, $sk);
}

sub crypto_box_seal {
    my ($in, $pk) = @_;

    my ($epk, $esk) = $crypto_box->keypair;
    my $nonce       = _crypto_box_seal_nonce($epk, $pk);
    my $c           = crypto_box_easy($in, $nonce, $pk, $esk);
    my $out         = $epk . $c;

    return $out;
}

sub crypto_box_open_detached {
    my ($c, $mac, $n, $pk, $sk) = @_;

    my $k = $crypto_box->beforenm($pk, $sk);
    my $m = $crypto_box->decrypt_detached_afternm($mac, $c, $n, $k);

    return $m;
}

sub crypto_box_open_easy {
    my ($c, $n, $pk, $sk) = @_;

    my $mac = substr $c, 0, $crypto_box->MACBYTES;
       $c   = substr $c,    $crypto_box->MACBYTES;
    my $m   = crypto_box_open_detached($c, $mac, $n, $pk, $sk);

    return $m;
}

sub crypto_box_seal_open {
    my ($in, $pk, $sk) = @_;

    return
      if length $in < $SEALBYTES;

    my $epk   = substr $in, 0, $crypto_box->PUBLICKEYBYTES;
    my $c     = substr $in,    $crypto_box->PUBLICKEYBYTES;
    my $nonce = _crypto_box_seal_nonce($in, $pk);
    my $m     = undef;

    eval {
        $m = crypto_box_open_easy($c, $nonce, $epk, $sk); 
    };

    return $m;
}

1;
