package ADM::Utils;

use strict;
use utf8;
use open qw(:std :utf8);

use DateTime;
use Digest::SHA 'sha256';
use List::MoreUtils 'uniq';
use List::Util qw/min max first/;
use MIME::Base64 qw/encode_base64url decode_base64url/;
use Net::Netmask;

use ADM::Requester;
use IP;
use Database::Converter;

my @YandexNetworks = (
    # _YANDEXNETS4_
    Net::Netmask->new('5.45.192.0/18'),
    Net::Netmask->new('5.255.192.0/18'),
    Net::Netmask->new('37.9.64.0/18'),
    Net::Netmask->new('37.140.128.0/18'),
    Net::Netmask->new('77.88.0.0/18'),
    Net::Netmask->new('84.201.128.0/18'),
    Net::Netmask->new('87.250.224.0/19'),
    Net::Netmask->new('93.158.128.0/18'),
    Net::Netmask->new('95.108.128.0/17'),
    Net::Netmask->new('100.43.64.0/19'),
    Net::Netmask->new('130.193.32.0/19'),
    Net::Netmask->new('141.8.128.0/18'),
    Net::Netmask->new('178.154.128.0/17'),
    Net::Netmask->new('185.32.184.0/22'),
    Net::Netmask->new('199.21.96.0/22'),
    Net::Netmask->new('199.36.240.0/22'),
    Net::Netmask->new('213.180.192.0/19'),

    # _YAMONEYNETS_
    Net::Netmask->new('77.75.152.0/21'),
    Net::Netmask->new('109.235.160.0/21'),
);

our $PDD_MIN_UID = 1130000000000000;
our $FOUND_UIDS_MAX_LIMIT = 30_000;

sub Now {
    my ($sec,$min,$hour,$mday,$mon,$year,undef,undef) = localtime;
    my $now = sprintf("%4d-%02d-%02d %02d:%02d:%02d",
                      $year + 1900, $mon + 1, $mday, $hour, $min, $sec);
    return $now;
}

sub HaveGrant {
    my $grant = shift;

    my $login  = GetCurrentLogin();
    my $guard  = $admin::Guard;
    my $result = $guard->is_user_have_grant($login, $grant);

    return $result;
}

sub HaveAtLeastOneGrant {
    my @grants = @_;

    for my $grant (@grants) {
        return 1 if HaveGrant($grant);
    }

    return 0;
}

sub HaveRole {
    my $role = shift;

    my $login  = GetCurrentLogin();
    my $guard  = $admin::Guard;
    my $result = $guard->is_user_have_role($login, $role);

    return $result;
}

sub IsUserAllowed   { $admin::Guard->is_user_allowed(       GetCurrentLogin() ) }
sub UserRolesString { $admin::Guard->get_user_roles_string( GetCurrentLogin() ) }

sub GetCurrentLogin {
    return  $ENV{'REMOTE_USER'} || '';
}

sub YandexNetwork($){
    my ($ip) = @_;

    unless ($ip){
        ADM::Logs::IntErr("no IP passed");
        return undef;
    }
    unless ($ip =~ m,^(\d+\.){3}\d+$,){
#        ADM::Logs::DeBug("bad IP '$ip' passed");
        return undef;
    }

    for (@YandexNetworks) {
        return 1 if ($_->match($ip));
    }

    return undef;
}

sub GetLimit {
    my ($l) = @_;
    $l ||= 2;

    if ($l == 2) {
        return 50;
    } elsif ($l == 3) {
        return 100;
    } elsif ($l == 4) {
        return 300;
    } elsif ($l == 5) {
        return 1000;
    } elsif ($l == 6) {
        return 3000;
    } elsif ($l == 7) {
        return 5000;
#    } elsif ($l == 99) {
#        return 0;
    } else {
        return 20;
    }
}

my %KARMA_FILTER = (
    1 => [    0           ],
    2 => [        85      ],
    3 => [             100],
    4 => [        85,  100],
    5 => [1000, 1085, 1100],
    6 => [2000, 2085, 2100],
);

sub GetKarma {
    my $index = shift;

    my $result = $KARMA_FILTER{$index};

    return undef unless $result;
    return $result;
}

sub TestIpAllowed {
    my ($mode, $ip) = @_;
    $ip ||= '';
    $ip =~ s/^\s+//;
    $ip =~ s/\s+$//;
    chomp($ip);
    return 0 unless $ip; # IP not defined, access denided

    my $ipo = IP->new($ip);
    unless ($ipo->is_valid){
        ADM::Logs::AdMin("Access denied: invalid ip; mode=$mode ip=$ip");
        return 0;
    }

    my $allow_from = $admin::Conf->GetVal('allow_from');
    return -1 unless defined($allow_from->{$mode}); # Rule not defined, access granted

    # Search in IP list
    if (defined($allow_from->{$mode}->{'allow-ip'}) &&
        defined($allow_from->{$mode}->{'allow-ip'}->{$ip}) &&
        $allow_from->{$mode}->{'allow-ip'}){
        return 1;
    }

#    # copied from Common::Tests just in case
#    if ($ipo->is_v6) {
#        my $ipv6_as_bytes = inet_pton(AF_INET6, $ipo);
#        my $project_id = substr $ipv6_as_bytes, 8, 4;
#        $project_id = unpack 'H*', $project_id;
#        $project_id = lc $project_id;
#        $project_id =~ s/^0+//;

#        my $patricia_trypo = $allow_from->{$mode}{'allow-trypo'}{$project_id} || undef;

#        if ($patricia_trypo and $patricia_trypo->match_string($ipo)) {
#            return 1;
#        }
#    }

    if ($ipo->is_v6) {
        my $patricia6 = $allow_from->{$mode}{'allow-patricia6'};
        if (my $granted_subnet = $patricia6->match_string($ipo)) {
            return 1;
        }
        Common::Logs::AdMin("Access denied: not found in ip-list and patricia6; mode=$mode ip=$ipo");
        return 0;
    }

    # Search in SUBNET list
    if(!defined($allow_from->{$mode}->{'allow-subnet'})) {
        ADM::Logs::DeBug("Access denied mode($mode): has no subnet list");
        return 0;
    }
    foreach my $subnet (keys %{$allow_from->{$mode}->{'allow-subnet'}}) {
        my $data = $allow_from->{$mode}->{'allow-subnet'}->{$subnet};
        my $block = $data->{'block'};
        if ($block->match($ip)) {
            return 1;
        }
    }
    return 0;
}

my $DM_DOMAIN;
my $DM_MODE;

sub DMDomain {
    return $DM_DOMAIN unless defined $_[0];
    $DM_DOMAIN = $_[0];
}

sub DMMode {
    return $DM_MODE   unless defined $_[0];
    $DM_MODE = $_[0];
}

sub GetLoginFromNativeEmail {
    my $email = shift;

    $email = lc $email;
    my $yandex_domains = $admin::Conf->GetHCVal('YandexDomains');
    my ($name, $domain) = split '@', $email;

    return $name if exists $yandex_domains->{$domain};
    return $email;
}

sub intersection {
    my (@arrays) = @_;

    @arrays = grep { defined } @arrays;

    return undef
      unless @arrays;

    my %counter;

    for my $array (@arrays) {
        ++$counter{$_} for @$array;
    }

    my $number = scalar @arrays;
    my @result = grep { $counter{$_} == $number } keys %counter;

    return \@result;
}

sub union {
    my (@arrays) = @_;

    @arrays = grep { defined } @arrays;

    return undef
      unless @arrays;

    my @result = uniq map { @$_ } @arrays;

    return \@result;
}

sub ExtractSupportNotesFromEvents {
    my $events = shift;

=pod
        [
          {
            'real_ts' => '1396524494049098',
            'admin' => 'spleenjack',
            'event_name' => 'support_note',
            'ip' => '2a02:6b8:0:107:e1e6:7a27:2511:aa62',
            'ts' => '2014-04-03 15:28:14',
            'host_id' => 127,
            'value' => "\x{442}\x{435}\x{441}\x{442}",
            'hostid' => '7F',
            'client_name' => 'admin'
          }
        ]
=cut

    my @result = grep { $_->{name} eq 'support_note' } @$events;

    return \@result;
}

sub ExtractUserinfoFtFromEvents {
    my ($events, $uid) = @_;

=pod
      {
        'real_ts' => '1270742097000000',
        'source' => 'userinfo_ft',
        'firstname' => 'a',
        'event_name' => 'userinfo_ft',
        'ip' => '95.108.193.106',
        'ts' => '2010-04-08 19:54:57',
        'hinta' => 'aa',
        'hostid' => '00',
        'lastname' => 'b',
        'login' => 'spleenjacktest2',
        'hintq' => "1:\x{414}\x{435}\x{432}\x{438}\x{447}\x{44c}\x{44f} \x{444}\x{430}\x{43c}\x{438}\x{43b}\x{438}\x{44f} \x{43c}\x{430}\x{442}\x{435}\x{440}\x{438}",
        'client_name' => 'passport'
      }


    [
      {
        'real_ts' => '1397728522511882',
        'event_name' => 'action',
        'ip' => '127.0.0.1',
        'ts' => '2014-04-17 13:55:22',
        'host_id' => 126,
        'value' => 'account_register_admimportreg',
        'hostid' => '7E',
        'yandexuid' => '6225545071397728522',
        'client_name' => 'passport'
      },
      {
        'real_ts' => '1397728522511882',
        'event_name' => 'info.firstname',
        'ip' => '127.0.0.1',
        'ts' => '2014-04-17 13:55:22',
        'host_id' => 126,
        'value' => 'oleg',
        'hostid' => '7E',
        'yandexuid' => '6225545071397728522',
        'client_name' => 'passport'
      },
    ]
=cut

    return {} unless $events;

    my $userinfo_ft = first { $_->{name} eq 'userinfo_ft' } @$events;
    my $action_account
      = first {
            $_->{name} eq 'action'
        and $_->{value} =~ /^account_(?:register|create).*/
      } @$events;

    my $result = {};

    if ($userinfo_ft) {
        $result = {
            dt        => $userinfo_ft->{dt},
            real_ts   => $userinfo_ft->{real_ts},
            ip        => $userinfo_ft->{ip},
            login     => $userinfo_ft->{login_wanted} || $userinfo_ft->{login},
            firstname => $userinfo_ft->{firstname},
            lastname  => $userinfo_ft->{lastname},
        };
    }
    elsif ($action_account) {
        my @adjacent
          = grep {
                $_->{host_id} eq $action_account->{host_id}
            and $_->{real_ts} eq $action_account->{real_ts}
          } @$events;

        my %event_to_value = map +( $_->{name} => $_->{value} ), @adjacent;

        $result = {
            dt        => $action_account->{dt},
            real_ts   => $action_account->{real_ts},
            ip        => $action_account->{ip},
            login     => $event_to_value{'info.login_wanted'} || $event_to_value{'info.login'},
            firstname => $event_to_value{'info.firstname'},
            lastname  => $event_to_value{'info.lastname'},
        };
    }

    if (%$result) {
        $result->{uid}  = $uid;
        $result->{name} = join '  ', $result->{lastname}, $result->{firstname};
    }

    return $result;
}

sub ExtractDeleteActionFromEvents {
    my $events = shift;

    return unless $events;

    my $action_delete
      = first {
            $_->{name} eq 'action'
        and $_->{value} eq 'account_delete'
      } @$events;

    return $action_delete;
}

my %EVENT_NAME_TO_SID = (
    'sid.restore'                => 2,
    'phone.add'                  => 36,
    'sid.yasms_number'           => 36,
    'sid.wwwdgt_wmode'           => 65,
    'subscription.yastaff.login' => 669,
);

sub ExtractSubscriptionsCreatedFromEvents {
    my $events = shift;

    my %sid_by_alias = Database::Converter->sid_by_alias;

    my %result;

    for my $event (@$events) {
        my $name  = $event->{name};
        my $value = $event->{value};

        my @sids;

        if ($name eq 'sid.add') {
            my @records = split /,/, $value;
            @sids = map { (split /\|/)[0] } @records;
        }

        elsif ($name eq 'alias.add') {
            my ($alias_name) = split /\|/, $value;
            $alias_name =~ s/_alias$//;

            my $sid = $sid_by_alias{$alias_name};
            @sids = ($sid)
              if $sid;
        }

        elsif ($EVENT_NAME_TO_SID{$name}) {
            @sids = ($EVENT_NAME_TO_SID{$name});
        }

        for my $sid (@sids) {
            next if $sid == 105; # Множественные pddalias
            $result{$sid} = $event->{dt};
        }
    }

    return \%result;
}

my %GRANT_TO_DENIED_EVENTS = (
    'show_person'             => [qw/info.firstname info.lastname info.sex info.birthday info.country info.lang info.city info.hintq info.used_question/],
    'show_encrypted_password' => [qw/info.password/],
    'show_hint_answer'        => [qw/info.hinta/],
    'show_phones'             => [qw/sid.yasms_number phone.add phone.rm phones.add phones.rm yasms.phone.number info.used_phone/],
    'show_emails'             => [qw/info.email email.invalidate email.delete email.getkey email.confirm email.rpop email.unsafe email.default email.send info.used_email/],
);

my %DENIED_EVENT_TO_GRANT;

while (my ($grant, $event_names) = each %GRANT_TO_DENIED_EVENTS) {
    for my $event_name (@$event_names) {
        $DENIED_EVENT_TO_GRANT{$event_name} = $grant;
    }
}

sub HideDeniedEvents {
    my $events = shift;

    return unless $events;

    for my $event (@$events) {
        my $items = $event->{items} || [ $event ];

        for my $item (@$items) {
            my $name  = $item->{name};
            my $value = $item->{value};

            my $grant = $DENIED_EVENT_TO_GRANT{$name} || undef;
            $grant = 'show_phones'
              if not $grant and $name =~ /^phone\.\d+?\.number$/;
            $grant = 'show_phones'
              if not $grant and $name =~ /^phone\.\d+?\.operation\.\d+?\.security_identity$/ and $value ne '1';

            next unless $grant;
            next if HaveGrant($grant);

            $item->{is_hidden} = 1;
        }
    }

    return $events;
}

sub JoinEventsById {
    my $items = shift;

    my $events_by_id;
    for my $item (@$items) {
        my $event = $events_by_id->{ $item->{id} } ||= $item;
        my $event_copy = { %$item };
        push @{ $event->{items} }, $event_copy;
    }

    my $result = [ values %$events_by_id ];

    return $result;
}

sub CopyAdminLoginAndCommentAsEvent {
    my $events = shift;

    return unless $events;

    for my $event (@$events) {
        my $items = $event->{items};

        if ($items) {
            my $item = first { $_->{admin} or $_->{comment} } @$items;

            if ($item->{admin}) {
                my %additional = %$item;
                $additional{name} = 'admin_login';
                $additional{value} = $additional{admin};
                $additional{is_hidden} = 0;
                push @$items, \%additional;
            }
            if ($item->{comment}) {
                my %additional = %$item;
                $additional{name} = 'admin_comment';
                $additional{value} = $additional{comment};
                $additional{is_hidden} = 0;
                utf8::decode($additional{value});
                push @$items, \%additional;
            }
        }
    }
}

my %SMS_ACTION_TO_STATE = (
    enqueued           => 'enqueued',
    senttosmsc         => 'sent',
    deliveredtosmsc    => 'accepted',
    notdeliveredtosmsc => 'rejected',
    queuedonsmsc       => 'queued',
    delivered          => 'delivered',
    notdelivered       => 'failed',
    'antifraud:check'  => 'antifraud_check',
    'antifraud:allow'  => 'antifraud_allow',
    'antifraud:deny'   => 'antifraud_deny',
    'antifraud:retry'  => 'antifraud_retry',
);

sub NormalizeSmsHistory {
    my ($sms_history) = @_;

    my $items_list
      = $sms_history->{global_smsid}
      ? [ $sms_history->{items} ]
      : [ values %{ $sms_history->{items} } ];

    my $result = [];

    for my $items (@$items_list) {
        my $sms;

        for my $item (@$items) {
            my $action = $item->{action};
            my $state  = $SMS_ACTION_TO_STATE{$action};

            $sms->{$state} = $item->{timestamp};

            if ($state eq 'enqueued') {
                $sms = { %$sms, %$item };

                delete @$sms{qw/action timestamp sms/};
            }
        }

        if ($sms->{enqueued}) {
            push @$result, $sms;
        }
    }

    $result = [ sort { $a->{enqueued} <=> $b->{enqueued} } @$result ];

    return $result;
}

sub str2hex ($) { unpack 'H*', $_[0] }
sub hex2str ($) {   pack 'H*', $_[0] }
sub str2b64 ($) { encode_base64url $_[0], '' }
sub b642str ($) { decode_base64url $_[0]     }

sub s2hms {
    my $seconds = shift;

    my $p
      = $seconds > 0 ? '+'
      : $seconds < 0 ? '-'
      :                '';

    my $h = int(abs($seconds) / 3600);
    my $m = int(abs($seconds) /   60) % 60;
    my $s =     abs($seconds)         % 60;

    my $result
      = $p
      . ($h ? $h . 'h' : '')
      . ($m ? $m . 'm' : '')
      .       $s . 's'
      ;

    return $result;
}

sub TimestampToUtcDatetime {
    my $ts = shift;

    my ($sec, $min, $hour, $day, $mon, $year) = gmtime $ts;
    my $result = sprintf '%04d-%02d-%02d %02d:%02d:%02d', $year + 1900, $mon + 1, $day, $hour, $min, $sec;

    return $result;
}

my %APPLE_HARDWARE_MODEL_TO_HUMAN_MODEL = (
    'iPhone1,2' => 'iPhone 3G',
    'iPhone2,1' => 'iPhone 3GS',
    'iPhone3,1' => 'iPhone 4',
    'iPhone3,3' => 'Verizon iPhone 4',
    'iPhone4,1' => 'iPhone 4S',
    'iPhone5,1' => 'iPhone 5 (GSM)',
    'iPhone5,2' => 'iPhone 5 (GSM+CDMA)',
    'iPhone5,3' => 'iPhone 5c (GSM)',
    'iPhone5,4' => 'iPhone 5c (GSM+CDMA)',
    'iPhone6,1' => 'iPhone 5s (GSM)',
    'iPhone6,2' => 'iPhone 5s (GSM+CDMA)',
    'iPhone7,1' => 'iPhone 6 Plus',
    'iPhone7,2' => 'iPhone 6',
    'iPod1,1' => 'iPod Touch 1G',
    'iPod2,1' => 'iPod Touch 2G',
    'iPod3,1' => 'iPod Touch 3G',
    'iPod4,1' => 'iPod Touch 4G',
    'iPod5,1' => 'iPod Touch 5G',
    'iPad1,1' => 'iPad',
    'iPad2,1' => 'iPad 2 (WiFi)',
    'iPad2,2' => 'iPad 2 (GSM)',
    'iPad2,3' => 'iPad 2 (CDMA)',
    'iPad2,4' => 'iPad 2 (WiFi)',
    'iPad2,5' => 'iPad Mini (WiFi)',
    'iPad2,6' => 'iPad Mini (GSM)',
    'iPad2,7' => 'iPad Mini (GSM+CDMA)',
    'iPad3,1' => 'iPad 3 (WiFi)',
    'iPad3,2' => 'iPad 3 (GSM+CDMA)',
    'iPad3,3' => 'iPad 3 (GSM)',
    'iPad3,4' => 'iPad 4 (WiFi)',
    'iPad3,5' => 'iPad 4 (GSM)',
    'iPad3,6' => 'iPad 4 (GSM+CDMA)',
    'iPad4,1' => 'iPad Air (WiFi)',
    'iPad4,2' => 'iPad Air (Cellular)',
    'iPad4,4' => 'iPad mini 2G (WiFi)',
    'iPad4,5' => 'iPad mini 2G (Cellular)',
    'iPad4,7' => 'iPad mini 3 (WiFi)',
    'iPad4,8' => 'iPad mini 3 (Cellular)',
    'iPad4,9' => 'iPad mini 3 (China Model)',
    'iPad5,3' => 'iPad Air 2 (WiFi)',
    'iPad5,4' => 'iPad Air 2 (Cellular)',
    'i386'    => 'Simulator',
    'x86_64'  => 'Simulator',
);

sub GetAppleHumanModelByHardwareModel { $APPLE_HARDWARE_MODEL_TO_HUMAN_MODEL{ $_[0] } || $_[0] }

my %GOOD_STATUSES = map { $_ => 1 } qw/successful ses_kill ses_create ses_update/;
my %BAD_STATUSES  = map { $_ => 1 } qw/disabled failed blocked bruteforce challenge_shown/;

sub GetHistoryDb3Auths {
    my (%args) = @_;

    my @good_statuses = grep { $GOOD_STATUSES{$_} } @{ $args{status} };
    my @bad_statuses  = grep { $BAD_STATUSES{$_}  } @{ $args{status} };

    my ($good_auths, $bad_auths) = ([], []);

    if (@good_statuses) {
        $good_auths = ADM::Requester::GetHistoryDb3Auths(%args, statuses => [@good_statuses, @bad_statuses], good => 1);
        return
          unless $good_auths;
    }

    if (@bad_statuses) {
        $bad_auths  = ADM::Requester::GetHistoryDb3Auths(%args, statuses => [@bad_statuses], bad => 1);
        return
          unless $good_auths;
    }

    my %index;
    my @result
      = grep { not $index{ join '-', @{$_}{qw/hostid real_ts/} }++ }
        sort { $a->{real_ts} <=> $b->{real_ts} }
        (@$good_auths, @$bad_auths);

    return \@result;
}

1;
