package Authorization;

use common::sense;

use Carp;
use Log::Any '$log';
use Time::HiRes;

use IP;
use Model::Builder;
use SessionId;

our $STATUS_ROTTEN = 'NEED_RESET';
our $STATUS_VALID  = 'VALID';

use Class::XSAccessor
    constructor => 'new',
    accessors   => [qw/
        passport_host current_domain
        is_pdd_mode_code_ref
        dbaf params input cookies
        sessionid account status cookie_status
    /]
;

sub create_sessionid_safe {
    my $self = shift;

    my $result = eval { $self->create_sessionid(@_) };

    return $result unless $@;

    $log->debug("create_sessionid_safe died - $@");
    return;
}

sub create_sessionid {
    my ($self, $uid, %args) = @_;

    my $now = Time::HiRes::time;

    my $sessionid = SessionId->new(uid => $uid);
    $self->sessionid($sessionid);

    $self->_populate_sessionid_by_account;

    $sessionid->created       (int $now);
    $sessionid->started       ($now);
    $sessionid->ip            ($self->input->{real_ip}->as_string);
    $sessionid->passport_host ($self->passport_host);
    $sessionid->key_space_name($args{key_space} || $self->_key_space);

    my $params = $self->params;
    my $delta
      = $params->{'timestamp-delta'} > 0 ? $params->{'timestamp-delta'}
      : $params->{timestamp}         > 0 ? $params->{timestamp} - int($now * 1000)
      :                                    0;
    $sessionid->delta(int $delta);

    $params->{twoweeks} =~ /^(?:yes|1)$/
      ? $sessionid->be_longer
      : $sessionid->be_sessional;

    return $sessionid;
}

sub serialize_sessionid {
    my $self = shift;
    my %args = @_;

    my $sessionid = $self->sessionid;

    my %blackbox_args = (
        host          => $self->_key_space,
        passport_host => $self->passport_host,
        sessionid     => $sessionid,
        get_safe      => 1,
        uid           => $sessionid->uid,
    );

    my $input_authz   = $Input::authz;
    my $old_sessionid = $input_authz && $input_authz->is_cookie_valid ? $input_authz->sessionid : undef;

    my $blackbox
      = $old_sessionid
      ? AIO::Requester::EditSessionId(%blackbox_args, old_sessionid => $old_sessionid, ip => $self->input->{real_ip}->as_string, operation => 'add')
      : AIO::Requester::CreateSessionId(%blackbox_args);

    $sessionid->tag($blackbox->{authid}{id});

    my $values1 = $blackbox->{'new-session'};
    my $cookie1 = Cookie->new(
        name     => 'Session_id',
        value    => $values1->{value},
        expires  => $values1->{expires},
        httponly => $values1->{'http-only'} ? 1 : 0,
        secure   => $values1->{secure}      ? 1 : 0,
    );
    $sessionid->cookie1($cookie1);

    my $values2 = $blackbox->{'new-sslsession'};
    if ($values2) {
        my $cookie2 = Cookie->new(
            name     => 'sessionid2',
            value    => $values2->{value},
            expires  => $values2->{expires},
            httponly => $values2->{'http-only'} ? 1 : 0,
            secure   => $values2->{secure}      ? 1 : 0,
        );
        $sessionid->cookie2($cookie2);
    }

    return $cookie1->value;
}

sub load_sessionid_safe {
    my $self = shift;

    my $result = eval { $self->load_sessionid(@_) };

    if ($@) {
        $log->debugf('load_sessionid_safe has been failed: %s', $@);
        return;
    }

    return $result;
}

sub load_sessionid_safe {
    my $self = shift;

    my $result = eval { $self->load_sessionid(@_) };

    if ($@) {
        $log->debugf('load_sessionid_safe has been failed: %s', $@);
        return;
    }

    return $result;
}

sub load_sessionid {
    my $self = shift;
    my %args = @_;

    return $self->sessionid if $self->sessionid;
    return if $self->status;

    return
      unless my $string = $args{string} || $self->_cookie;

    my $sslstring = $args{sslstring} || $self->_sslcookie;

    my $real_ip   = $self->input->{real_ip};
    my $key_space
      = ($self->is_pdd_mode_code_ref->() && $self->cookies && $self->cookies->{Eda_id})
      ? 'pochta'
      : $args{key_space} || $self->_key_space;

    my $blackbox = AIO::Requester::CheckSessionId(
        host          => $key_space,
        passport_host => $self->passport_host,
        sessionid     => $string,
        sslsessionid  => $sslstring,
        ip            => $real_ip->as_string,
        queries       => Model::Builder->required_blackbox_queries,
    );

    unless ($blackbox) {
        $log->debug("load_sessionid has been failed on string='$string': $@");
        return;
    }

    my $time = time;

    $self->cookie_status($blackbox->{status}{value});

    return unless $self->is_cookie_valid;

    my $users        = $blackbox->{users};
    my %users_by_uid = map +( $_->{id} => $_ ), @$users;

    my $default_uid  = $blackbox->{default_uid};
    my $default_user = $users_by_uid{$default_uid};

    $self->status($default_user->{status}{value});

    my $ttl = $blackbox->{ttl} || 0;
    $ttl = 0
      if $ttl == 2;

    my $sessionid = SessionId->new;
    $self->sessionid($sessionid);

    $sessionid->string              ($string);
    $sessionid->sslstring           ($sslstring);
    $sessionid->policy_id           ($ttl);
    $sessionid->passport_host       ($blackbox->{authid}{host});
    $sessionid->ip                  (IP->new($blackbox->{authid}{ip})->as_string);
    $sessionid->started             ($blackbox->{authid}{time} ? $blackbox->{authid}{time} / 1000 : 0);
    $sessionid->tag                 ($blackbox->{authid}{id});
    $sessionid->age                 ($blackbox->{age});

    my (%uids_validity, %uids_secure);
    while (my ($uid, $user) = each %users_by_uid) {
        $uids_validity{$uid} = $user->{status}{value} =~ /^ (?: $STATUS_VALID | $STATUS_ROTTEN ) $/ox ? 1 : 0;
        $uids_secure{$uid}   = $user->{auth}{secure}                                                  ? 1 : 0;
        $self->dbaf->GetAccountFromBlackboxResponse($uid, $user);
    }
    $sessionid->uids_validity(\%uids_validity);
    $sessionid->uids_secure(\%uids_secure);
    $sessionid->uids_number(scalar @$users);

    return unless $self->is_authorized;

    $sessionid->uid                 ($default_uid);
    $sessionid->social_profile_id   ($default_user->{auth}{social}{profile_id});
    $sessionid->password_verification_age($default_user->{auth}{password_verification_age});
    $sessionid->is_stressed         (1) if $default_user->{special}{id} == 2;

    my $account = $self->dbaf->GetAccount($default_uid);
    $self->account($account);
    $self->_populate_sessionid_by_account;

    $sessionid->key_space_name($key_space);

    $sessionid->be_lite
      if $default_user->{uid}{lite};

    return $sessionid;
}

sub transfer_sessionid_safe {
    my $self = shift;

    my $result = eval { $self->transfer_sessionid(@_) };

    return $result unless $@;

    $log->debug("transfer_sessionid_safe died - $@");
    return;
}

sub transfer_sessionid {
    my $self   = shift;
    my $source = shift;
    my %args   = @_;

    die "source is required"
      unless $source;

    my $sessionid = $source->clone;
    $self->sessionid($sessionid);

    $self->_populate_sessionid_by_account;

    unless ($sessionid->is_medium) {
        $sessionid->created(time);
        $sessionid->be_longer
          if $sessionid->is_long;
    }

    return $sessionid;
}

sub transfer_tag {
    my $self   = shift;
    my $source = shift;

    return unless $source;

    die "destination sessionid is required"
      unless my $destination = $self->sessionid;

    return unless $source->uid eq $destination->uid;
    return unless $source->tag;

    $destination->tag($source->tag);

    return 1;
}

sub is_authorized { shift->status =~ /^ (?: $STATUS_VALID | $STATUS_ROTTEN ) $/ox ? 1 : 0 }
sub is_anonymous  { shift->is_authorized            ? 0 : 1 }
sub is_rotten     { shift->status eq $STATUS_ROTTEN ? 1 : 0 }
sub is_valid      { shift->status eq $STATUS_VALID  ? 1 : 0 }

sub is_cookie_valid { shift->cookie_status =~ /^ (?: $STATUS_VALID | $STATUS_ROTTEN ) $/ox ? 1 : 0 }

sub _key_space {
    my $self = shift;

    my $result = $self->current_domain;
    $result =~ s/^\.+//;

    return $result;
}

sub _cookie {
    my $self = shift;

    return '' unless $self->cookies;

    if ($self->is_pdd_mode_code_ref->()) {
        return $self->cookies->{Eda_id} || $self->cookies->{Session_id} || '';
    }
    else {
        return $self->cookies->{Session_id} || '';
    }
}

sub _sslcookie {
    my $self = shift;

    return '' unless $Input::secure_mode;
    return '' unless $self->cookies;

    if ($self->is_pdd_mode_code_ref->()) {
        return $self->cookies->{edaid2} || $self->cookies->{sessionid2} || '';
    }
    else {
        return $self->cookies->{sessionid2} || '';
    }
}

sub _populate_sessionid_by_account {
    my $self = shift;

    my $sessionid = $self->sessionid;
    my $dbaf      = $self->dbaf;
    my $uid       = $sessionid->uid;
    my $account   = $dbaf->GetAccount($uid);

    $sessionid->user_have_password($account->have_password);
    $sessionid->is_betatester     ($account->is_betatester);
    $sessionid->is_yandexoid      ($account->is_yandexoid);

    $sessionid->be_normal;
    $sessionid->be_lite           (1) if $account->is_lite;

    return $sessionid;
}

1;
