#
# Package for various debug functions
#
# $Id: Debug.pm,v 2.2 2006-03-30 14:46:55 shurukhin Exp $
package Common::Debug;
use strict;
use utf8;
use open qw(:std :utf8);

use Exporter;

my %ref_handler = (
    HASH => \&deep_print_hash,
    ARRAY => \&deep_print_array,
    SCALAR => sub {}
);

BEGIN {
    use vars qw(@ISA @EXPORT);
    @ISA = qw(Exporter);
    @EXPORT = qw(print_hash print_array serialize_all);
}

sub print_hash {
    my ($hash_name, $ref, $checked, $FH) = @_;
    $FH = \*STDERR unless(defined($FH) and ref($FH) eq 'GLOB');
    if ($checked and not defined($ref) and (ref($ref) ne 'HASH')) {
        print $FH "[" . localtime() . "] [debug] The argument $hash_name undefined or isn't a hash\n";
        return;
    }
    my ($k, $v);
    print $FH "[" . localtime() . "] [debug] printing hash $hash_name ---- \n" if (defined($hash_name));
    while (($k, $v) = each %$ref) {
        print $FH "$k => $v\n"; 
    }
    print $FH "---- end of $hash_name\n";
}

sub print_array {
    my ($arr_name, $ref, $checked, $FH) = @_;
    $FH = \*STDERR unless(defined($FH) and ref($FH) eq 'GLOB');
    if ($checked and not defined($ref) and (ref($ref) ne 'ARRAY')) {
        print $FH "[" . localtime() . "] [debug] The argument $arr_name is undefined or isn't an array\n";
        return;
    }
    print $FH "[" . localtime() . "] [debug] printing array $arr_name --- \n" if (defined($arr_name));
    my $idx = 0;
    foreach (@$ref) {
        print $FH "item $idx = $_\n";
        $idx++;
    }
    print $FH " --- end of $arr_name\n";
}

sub deep {
    my $ref_type = ref(shift);
    if($ref_type) {
        if(exists($ref_handler{$ref_type})) {
            return $ref_handler{$ref_type};
        } else {
            return \&serialize_all;
        }
    }
}

sub serialize_all {
    my ($classname, $obj_ptr, $FH) = @_;
    eval { keys(%$obj_ptr) };
    if($@) {
        return;
    }
    $FH = \*STDERR unless(defined($FH) and ref($FH) eq 'GLOB');

    my ($k, $v);
    print $FH "[" . localtime() . "] [debug] printing $classname ===> \n" if (defined($classname));
    while (($k, $v) = each %$obj_ptr) {
        print $FH "$k => $v\n"; 
        my $func = deep($v) if($v);
        &$func("Class: $classname->$k", $v, 1, $FH) if ($func);
    }
    print $FH "<===  end of class $classname\n\n";
}

sub deep_print_hash {
    my ($hash_name, $ref, $checked, $FH) = @_;
    $FH = \*STDERR unless(defined($FH) and ref($FH) eq 'GLOB');
    if ($checked and not defined($ref) and (ref($ref) ne 'HASH')) {
        print $FH "[" . localtime() . "] [debug] The argument $hash_name undefined or isn't a hash\n";
        return;
    }
    my ($k, $v);
    print $FH "[" . localtime() . "] [debug] printing hash $hash_name ---- \n" if (defined($hash_name));
    while (($k, $v) = each %$ref) {
        print $FH "$k => $v\n"; 
        my $func = deep($v) if($v);
        &$func("Hash: $hash_name->$k", $v, 1, $FH) if ($func);
    }
    print $FH "---- end of hash $hash_name\n";
}

sub deep_print_array {
    my ($arr_name, $ref, $checked, $FH) = @_;
    $FH = \*STDERR unless(defined($FH) and ref($FH) eq 'GLOB');
    if ($checked and not defined($ref) and (ref($ref) ne 'ARRAY')) {
        print $FH "[" . localtime() . "] [debug] The argument $arr_name is undefined or isn't an array\n";
        return;
    }
    print $FH "[" . localtime() . "] [debug] printing array $arr_name --- \n" if (defined($arr_name));
    my $idx = 0;
    foreach (@$ref) {
        print $FH "item $idx = $_\n";
        my $func = deep($_) if($_);
        &$func("Array: ".$arr_name."[".$idx."]", $_, 1, $FH) if ($func);
        $idx++;
    }
    print $FH " --- end of array $arr_name\n";
}

sub stack_trace {
    my $FH = shift || \*STDERR;
    my ($pack, $file, $line, $subname, $hasargs, $wantarray);
    my $i = 0;
    while(($pack, $file, $line, $subname, $hasargs, $wantarray) = caller(++$i)) {
        print $FH "[" . localtime() . "] [debug] pack = $pack, file = $file, line = $line, subname = $subname, hasargs = $hasargs, wantarray = $wantarray\n";
    }
}

1;
