# Package for different logs
# $Id: Logs.pm,v 2.36 2008/05/08 14:39:22 torubarov Exp $ $Name:  $
#
package Common::Logs;

use strict;
use utf8;
use open qw(:std :utf8);
use POSIX qw(strftime);
use Global;

use History::Logger;
use History::Converter::PassportChanges;
use Time::HiRes;

use vars qw($UUID);
$UUID = '';

sub init {
    $UUID = sprintf '%s-%d-%.6f', $main::Conf->GetVal('this_host_id'), $$, Time::HiRes::time;
}

our $caller_depth = 0;

sub IntErr;
sub DeBug;

my $history_logger;
my $changes_converter;

# в силу частого использования именно этих обращений к логу, добавлен конструктор и эти методы
sub new   {bless{},shift}
sub debug {shift; Common::Logs::DeBug(@_)}
sub int   {shift; Common::Logs::IntErr(@_)}
sub admin {shift; Common::Logs::AdMin(@_)}

sub Init {
    return if $history_logger;

    $history_logger = History::Logger->new(
        host_id     => $main::Conf->GetVal('this_host_id'),
        client_name => $main::Conf->GetVal('client_name'),

        auth_log_path  => File::Spec->catfile($main::Conf->GetVal('child_log_path', 'auth_new_log_file')),
        event_log_path => File::Spec->catfile($main::Conf->GetVal('child_log_path', 'event_log_file')),

        pfile_write_code_ref => \&Common::Pfile::PWrite,
        pfile_close_code_ref => \&Common::Pfile::PClose,
        log_admin_code_ref   => \&Common::Logs::AdMin,
        log_interr_code_ref  => \&Common::Logs::IntErr,
    );
    die "i can't create history logger instance!" unless $history_logger;

    $changes_converter = History::Converter::PassportChanges->new(
        log_admin_code_ref   => \&Common::Logs::AdMin,
        log_debug_code_ref   => \&Common::Logs::DeBug,
    );
    die "i can't create changes converter instance!" unless $changes_converter;
}

sub LogChanges {
    my ($opdata, $changes) = @_;
    unless( defined($opdata) and
            defined($changes) ){
        DeBug("In LogChanges: bad parameters")
            if($main::Conf->GetVal('debug_level'));
        return undef;
    }
    
    $changes->{yandexuid} = $Input::Input->{cookie}->{yandexuid};

    my $bdate = undef;
    if (exists $changes->{birth_date}) {
        $changes->{bdate} = sprintf('%04d-%02d-%02d', split /-/, delete $changes->{birth_date});
    }
    if (defined $changes->{bdate}) {
        $bdate = $changes->{bdate};
        # конвертируем из pdd-формата в обычный
        if ($bdate =~ /^(\d+)\.(\d+)\.(\d\d\d\d)$/) {
            $bdate = sprintf('%04d-%02d-%02d', $3, $2, $1);
        }
    } elsif (exists $changes->{byear} and exists $changes->{bmonth} and exists $changes->{bday}) {
        if ($changes->{byear} and $changes->{bmonth} and $changes->{bday}) {
            $bdate = sprintf('%04d-%02d-%02d', delete @$changes{qw(byear bmonth bday)});
        } else {
            $bdate = '';
        }
    }
    if (defined $bdate) {
        $changes->{bdate} = $bdate;
    }

    LogEventsAsChanges($opdata, $changes);

    return 1;
}

sub LogEventsAsChanges {
    return 1 unless $main::Conf->GetVal('event_log_enabled');

    Init();

    my ($opdata, $changes) = @_;

    my $admin = delete $changes->{admin_login} || delete $changes->{admin_edit}; # в admin_edit при mode=admin лежит uid админа
    my $comment = delete $changes->{admin_comment} || undef;
    my $target = {
        uid => $opdata->{uid},
    };
    my $caller = {
        ip_from => $opdata->{ip_from},
        ip_prox => $opdata->{ip_prox},
        yandexuid => delete $changes->{yandexuid},
    };
    my $events = $changes_converter->convert($opdata, $changes);

    my $mode = $Output::formargs->{passport_call_mode};
    if ($mode and not $Passport::api_modes{$mode}) {
        push @$events, {
            name  => 'user_agent',
            value => substr($Input::io->{req}->header_in('User-Agent'), 0, 200),
        };
    }
    
    return $history_logger->log_events(
        'caller' => $caller,
        target   => $target,
        events   => $events,
        admin    => $admin,
        comment  => $comment,
    );
}

sub LogDebug {
    my ($args) = @_;
    if ($main::Conf->GetVal('debug_log_file_level')) {
        unless(Common::Pfile::PWrite(File::Spec->catfile($main::Conf->GetVal('log_path','debug_log_file')), "[".Common::Misc::Now()."] $$ $args\n")) {
            return undef;
        }
    }
    return 1;
}

sub Caller {
    my $prev = shift;
    $prev = $prev?1:0;
    my @backtrace;
    for (1..3) {
        my @caller_info = caller($_+ $prev);
        last unless(@caller_info);
        my $chunk = join(' : ',$caller_info[0],$caller_info[2]);
        last if ($chunk eq 'main:0');
        push @backtrace, $chunk;
    }
    my $trace_str = join('->', reverse @backtrace);
    print STDERR "[" . localtime() . "] [callstack] " . $trace_str .")\n";
}

sub IntErr {
    my $msg = shift;
    my ($package, undef, $line) = caller $caller_depth;
    $msg ||= 'Unknown error. Bad Common::Logs::IntErr() call.';
    $msg =~ s,[\r|\n]+,^M,g;
    print STDERR "[" . localtime() . "] [interror] " .($UUID ? "[$UUID] " : ''). $msg . " (" . $package . ":" . $line . ")\n";
}

sub DeBug {
    my $msg = shift;
    my ($package, undef, $line) = caller $caller_depth;
    $msg ||= 'Unknown debug message. Bad Common::Logs::DeBug() call.';
    $msg =~ s,[\r|\n]+,^M,g;
    print STDERR "[" . localtime() . "] [debug] " .($UUID ? "[$UUID] " : ''). $msg . " (" . $package . ":" . $line . ")\n";
}

sub AdMin {
    my $msg = shift;
    my ($package, undef, $line) = caller $caller_depth;
    $msg ||= 'Unknown admin message. Bad Common::Logs::AdMin() call.';
    $msg =~ s,[\r|\n]+,^M,g;
    print STDERR "[" . localtime() . "] [admin] " .($UUID ? "[$UUID] " : ''). $msg . " (" . $package . ":" . $line . ")\n";
}

sub DbErr($;$) {
    my $msg = shift;
    my $parent = (shift() ? 1 : 0);
    my ($package, undef, $line) = caller($parent);
    $msg ||= 'Unknown database error message. Bad Common::Logs::DbErr() call.';
    $msg =~ s,[\r|\n]+,^M,g;
    print STDERR "[" . localtime() . "] [dberror] " .($UUID ? "[$UUID] " : '') . $msg . " (" . $package . ":" . $line . ")\n";
}

sub ConfErr($) {
    my $msg = shift;
    my ($package, undef, $line) = caller;
    $msg ||= 'Unknown config error. Bad Common::Logs::ConfErr() call.';
    $msg =~ s,[\r|\n]+,^M,g;
    print STDERR "[" . localtime() . "] [configuration] " . $msg . " (" . $package . ":" . $line . ")\n";
}

sub StatModeError($) {
    my ($msg) = @_;
    my ($package, undef, $line) = caller;
    $msg ||= 'Undefined message. Bad Common::Logs::StatModeError() call.';
    $msg =~ s,[\r|\n]+,^M,g;
    print STDERR "[" . localtime() . "] [modeerror] " . $msg . " (" . $package . ":" . $line . ")\n";
}

sub Event {
    my @args = @_;

    my $time    = localtime;
    my $message = "[$time] [event] " . join('', @args);
    
    print STDERR $message, "\n";
}

sub Statbox {
    my (%data) = @_;

    # http://wiki.yandex-team.ru/AlekseyOzerickijj/push-client
    # http://wiki.yandex-team.ru/Statbox/LogRequirements#tskvformat

    my $time = time;

    my @pairs;
    my @keys = sort keys %data;

    $data{tskv_format} = 'passport-log';
    $data{unixtime}    = $time;
    unshift @keys, qw/tskv_format unixtime/;

    for my $key (@keys) {
        my $value = $data{$key};
        $value = substr $value, 0, 1000;
        $value =~ s/\t/\\t/g;
        $value =~ s/\n/\\n/g;
        $value =~ s/\r/\\r/g;
        $value =~ s/\0/\\0/g;
        $value =~ s/\\/\\\\/g;
        push @pairs, "$key=$value";
    }

    my $string = join("\t", 'tskv', @pairs) . "\n";

    my @today             = localtime($time);
    my @yesterday         = localtime($time - 60 * 60 * 24);

    my $file              = File::Spec->catfile($main::Conf->GetVal('child_log_path', 'statbox_log_file'));
    my $today_logname     = $file . strftime('.%Y%m%d', @today);
    my $yesterday_logname = $file . strftime('.%Y%m%d', @yesterday);

    Common::Pfile::PClose($yesterday_logname);
    return undef unless Common::Pfile::PWrite($today_logname, $string, 'dbutf8');
    return 1;
}

1;
