package Conf;

use strict;

use utf8;
use open qw(:std :utf8);

use Config::ApacheFormat;
use Common::Logs;
use POSIX ();
use Net::Netmask;
use Net::Patricia;
use Readonly;

use IP;

Readonly my $MIN_SECONDS_BETWEEN_STAT_CHECK => 30;

BEGIN {
    %Conf::Files = (
        'passport.conf'    => {
            'reload'    => 1,
            'timestamp' => 0,
            'config'    => undef,
            'tokens'    => [
                # нужно знать сразу о том, является ли данный конфиг с динамическим определением домена для остальной части конфига или нет
                'allowed_domains',

                'yandexteam',
                'yandexteam_passwd_lifetime',

                'pass_domain',

                'pass_http',
                'pass_https',
                'pass_img',
                'passport_backend_host',
                # valid lang
                'valid_lang_for_domain',
                'default_lang_for_domain',
                # mda
                'root_cookie',
                'pass_pass',
                'pass_pass_https',
                'pass_name',
                'md_auth',
                # Host ID for logs and various pathes:
                'this_host_id',
                'log_path',
                'child_log_path',
                'lock_root',
                'debug_log_file',
                'debug_log_file_level',
                'bad_answers_log_file',
                'statbox_log_file',
                'newdbfail_log_file',

                'auth_new_log_enabled',
                'auth_new_log_file',
                'event_log_enabled',
                'event_log_file',
                'client_name',
                # Template pathes
                'tmpl',
                'pdatmpl',
                'tmplcache',
                'template_stat_ttl',
                # DebugLevel
                'debug_level',
                # External HTML calls
                'yandex_global_css_http',
                'yandex_global_css_https',
                'inbox_js_http',
                'inbox_js_https',
                'sql_insert_aggregation',
                'sql_max_insert_aggregation',

                'blackbox_url',

                'upload_tmp_dir',

                # двойные заголовки для user agent's
                'double_headers_ua',

                'social_login_prefix',
                # файл с гастройками для langdetect (см. yandex-lang-detect-data и http://wiki.yandex-team.ru/AlekseyPonomarev/lang-detect)
                'langdetect_source_file',
                'social_api_host',
                'social_host',
                # список всех поддреживаемых яндексом языков
                'all_yandex_lang',
                # максимально широкий набор языков, поддерживаемый Яндексом на этом домене
                'yandex_lang_for_domain',
                # ассоциация домена со страной
                'domains_country',
                # путь к апи yasms
                'phone_validator_api',
                # задержка удаления логинов из таблицы reserved_logins
                'reserved_login_delay',
                'money_registration_url',
                'phones_new_scheme_datetime',
                'mail_normal_registration_host_id',
                'mail_pdd_registration_host_id',

                'new_database_writes_attempts',
                'new_database_writes_delay',
            ],
        },

        'db.conf'          => {
            'reload'    => 0,
            'timestamp' => 0,
            'config'    => undef,
            'tokens'    => [
                'centraldbmaster',
                'sharddbmaster',

                'centraldbslave',
                'sharddbslave',

                'admcentraldbslave',
                'admsharddbslave',

                'heavycentraldbslave',
                'heavysharddbslave',
            ],
        },

        'db_credentials.conf' => {
            'reload'    => 0,
            'timestamp' => 0,
            'config'    => undef,
            'tokens'    => [
                'db_credentials',
            ],
        },

        'access.conf' => {
            'reload' => 1,
            'timestamp' => 0,
            'config' => undef,
            'tokens' => [
                'allow_from',
            ],
        },

        'ranges.conf' => {
            loader => \&_ranges_loader,
            tokens => [
                'uid_collection_ranges',
            ],
        },
    );
    %Conf::Tokens = ();
    %Conf::ConfCache = ();
    %Cond::HardCoded = ();
    while (my ($confname, $settings) = each(%Conf::Files)){
        if (ref $settings->{'tokens'} eq 'ARRAY'){
            foreach my $token (@{$settings->{'tokens'}}){
                $Conf::Tokens{$token} = $confname;
                $Conf::ConfCache{$token} = undef;
            }
        }
    }

    # f -- handler
    # n -- Name in .conf file
    # d -- default value
    # m -- flag, must be defined
    # t -- type
    # attr -- параметр, который уточняет поведение для f

    %Conf::Helpers = (
        'allowed_domains'       => { f => \&_plainvalue, n => 'AllowedDomains',      d => 'yandex.ru,yandex.com,yandex.com.tr,yandex-team.ru,yandex.ua,yandex.by,yandex.kz' },

# if this passport for internal using
        'yandexteam' => {'f' => \&_plainvalue, 'n' => 'YandexTeam', 'd' => '0',},
# на текущий момент использование staff.yandex.ru запрещено
        'yandexteam_passwd_lifetime' => {'f' => \&_plainvalue, 'n' => 'YandexTeamPasswdLifetime', 'd' => '7200',},
# passport.conf
        'pass_domain'       => {'f' => \&_plainvalue,   'n' => 'PassDomain',    'd' => 'passport.yandex.ru', 'attr' => 'domain_dependence'},
# список всех поддреживаемых яндексом языков
        'all_yandex_lang' => {'f' => \&_plainvalue,   'n' => 'AllYandexLang', 'd' => 'ru,be,kk,tt,uk,az,en,tr'},
# максимально широкий набор языков, поддерживаемый Яндексом на этом домене
        'yandex_lang_for_domain' => {'f' => \&_need_eval,   'n' => 'YandexLangForDomain', d => "{'yandex.ru' => ['ru','be','kk','tt','uk','az','en'], 'yandex.com' => ['en'], 'yandex.com.tr' => ['tr'], 'default' => ['ru','be','kk','tt','uk','az','en', 'tr']}", m => 0},

        'social_api_host' => { f => \&_plainvalue, n => 'SocialApiHost', d => 'api.social.yandex.ru' },
        'social_host'     => { f => \&_plainvalue, n => 'SocialHost',    d => 'social.yandex.ru', 'attr' => 'domain_dependence' },

# FIXME: we use PassHTTPFull instead of PassHTTP
        'pass_http' => {'f' => \&_plainvalue, 'n' => 'PassHTTP', 'd' => 'https://passport.yandex.ru/passport', 'attr' => 'domain_dependence'},
        'pass_https' => {'f' => \&_plainvalue, 'n' => 'PassHTTPS', 'd' => 'https://passport.yandex.ru/passport', 'attr' => 'domain_dependence'},
        'pass_img' => {'f' => \&_plainvalue, 'n' => 'PassImg', 'd' => 'https://passport.yandex.ru/images', 'attr' => 'domain_dependence'},
        'passport_backend_host' => { f => \&_plainvalue, n => 'PassportBackendHost', d => '127.0.0.1:6000' },
# valid lang должно быть согласовано с allowed_domains
        'valid_lang_for_domain'   => { f => \&_need_eval, n => 'ValidLangForDomain',   d => "{'yandex.ru' => ['ru'], 'yandex.com' => ['en'], 'yandex-team.ru' => ['ru','en'], 'yandex.com.tr' => ['tr'], 'yandex.ua' => ['ru'], 'yandex.kz' => ['ru'], 'yandex.by' => ['ru']}" },
        'default_lang_for_domain' => { f => \&_need_eval, n => 'DefaultLangForDomain', d => "{'yandex.ru' => 'ru',   'yandex.com' => 'en',   'yandex-team.ru' => 'ru',        'yandex.com.tr' => 'tr',   'yandex.ua' => 'ru', 'yandex.kz' => 'ru', 'yandex.by' => 'ru'}" },
# ассоциация домена со страной
        'domains_country' => { f => \&_need_eval,  n => 'DomainsCountry', d => "{'ru' => 'RU', 'com' => 'US', 'com.tr' => 'TR'}", m => 0 },

# для 67 сида, вытаскиваем в конфиг ограничение по числу последних сопадений паролей

# mda
        'root_cookie' => {'f' => \&_plainvalue, 'n' => 'RootCookie', 'd' => '.yandex.ru', 'attr' => 'domain_dependence'},
        'pass_pass' => {'f' => \&_plainvalue, 'n' => 'PassPass', 'd' => 'https://pass.yandex.ru', 'attr' => 'domain_dependence'},
        'pass_pass_https' => {'f' => \&_plainvalue, 'n' => 'PassPassHttps', 'd' => 'https://pass.yandex.ru', 'attr' => 'domain_dependence'},
        'pass_name' => {'f' => \&_plainvalue, 'n' => 'PassName', 'd' => 'pass'},
        'md_auth' => {'f' => \&_plainvalue, 'n' => 'MdaRequiredForDomains', 'd' => 'yandex.ru', , 'attr' => 'mda_dependence'},

# Host ID for logs and various pathes:
        'debug_log_file_level' => {'f' => \&_plainvalue, 'n' => 'DebugLogFileLevel', 'd' => 0},
        'this_host_id' => {'f' => \&_plainvalue, 'n' => 'ThisHostID', 'm' => 1},
        'log_path' => {'f' => \&_plainvalue, 'n' => 'LogPath', 'm' => 1, 't' => 'DIR',},
        'child_log_path' => {'f' => \&_plainvalue, 'n' => 'ChildLogPath', 'm' => 1, 't' => 'DIR',},
        'lock_root' => {'f' => \&_plainvalue, 'n' => 'LockRoot', 'm' => 1, 't' => 'DIR',},

        'debug_log_file' => {'f' => \&_plainvalue, 'n' => 'DebugLogFile', 'm' => 1,},
        'bad_answers_log_file' => { f => \&_plainvalue, n => 'BadAnswersLogFile', d => 'badanswers.log' },
        'statbox_log_file'     => { f => \&_plainvalue, n => 'StatboxLogFile', m => 1 },
        'newdbfail_log_file'   => { f => \&_plainvalue, n => 'NewDbFailLogFile', m => 0, d => 'newdbfail.log' },

        'auth_new_log_enabled' => { f => \&_plainvalue, n => 'AuthNewLogEnabled', m => 0, d => 1             },
        'auth_new_log_file'    => { f => \&_plainvalue, n => 'AuthNewLogFile',    m => 0, d => 'authnew.log' },
        'event_log_enabled'    => { f => \&_plainvalue, n => 'EventLogEnabled',   m => 0, d => 1             },
        'event_log_file'       => { f => \&_plainvalue, n => 'EventLogFile',      m => 0, d => 'event.log'   },
        'client_name'          => { f => \&_plainvalue, n => 'ClientName',        m => 0, d => 'passport'    },

# Template pathes
        'tmpl' => {'f' => \&_plainvalue, 'n' => 'TMPL', 'm' => 1, 't' => 'DIR',},
        'pdatmpl' => {'f' => \&_plainvalue, 'n' => 'PDATMPL', 't' => 'DIR',},
        'tmplcache' => {'f' => \&_plainvalue, 'n' => 'TMPLCACHE', 't' => 'DIR',},
        'template_stat_ttl' => { f => \&_plainvalue, n => 'TemplateStatTtl' },
# DebugLevel
# 0 - no debug
# 1 - fatal errors reported
# 2 - nonfatal errors reported
# 3 - all possible debug is on
        'debug_level' => {'f' => \&_plainvalue, 'n' => 'DebugLevel', 'd' => 0,},
# External HTML calls
        'yandex_global_css_http' => {'f' => \&_plainvalue, 'n' => 'YandexGlobalCssHTTP', 'd' => 'https://css.yandex.ru/css/_yandex-global.css', 'attr' => 'domain_dependence'},
        'yandex_global_css_https' => {'f' => \&_plainvalue, 'n' => 'YandexGlobalCssHTTPS', 'd' => 'https://css.yandex.ru/css/_yandex-global.css', 'attr' => 'domain_dependence'},
        'inbox_js_http' => {'f' => \&_plainvalue, 'n' => 'InboxJsHTTP', 'd' => '/js/inbox_http.js',},
        'inbox_js_https' => {'f' => \&_plainvalue, 'n' => 'InboxJsHTTPS', 'd' => '/js/inbox_http.js',},

        'sql_insert_aggregation' => {'f' => \&_plainvalue, 'n' => 'SQLInsertAggregation', 'd' => 500,},
        'sql_max_insert_aggregation' => {'f' => \&_plainvalue, 'n' => 'SQLMaxInsertAggregation', 'd' => 1000,},

        'blackbox_url'                          => { f => \&_plainvalue, n => 'BlackboxUrl',   d => 'http://blackbox.yandex.net/blackbox'},

        'phone_validator_api'                     => { f => \&_plainvalue, n => 'PhoneValidatorAPI', d => 'http://phone-passport.yandex.ru/api/'},

        'upload_tmp_dir'                    => { f => \&_plainvalue, n => 'UploadTmpDir', m => 0, d => '/tmp' },

        'double_headers_ua'             => { f => \&_plainvalue, n => 'DoubleHeadersToUA',   d => 'Yandex Photomanager Client'},

        'social_login_prefix'           => { f => \&_plainvalue,        n => 'SocialLoginPrefix',       m => 0, d => 'uid-' },

        'langdetect_source_file'        => { f => \&_file_path,        n => 'LangdetectSourceFile', d => '/usr/share/yandex/lang_detect_data.txt' },
        'reserved_login_delay' =>   { f => \&_plainvalue, n => 'ReservedLoginDelay', d => 1 },
        'money_registration_url'  => { f => \&_plainvalue, n => 'MoneyRegistrationUrl' },
        'phones_new_scheme_datetime'            => { f => \&_plainvalue, n => 'PhonesNewSchemeDatetime',         d => '2013-12-01 00:00:00' },
        'mail_normal_registration_host_id' => { f => \&_plainvalue, n => 'MailNormalRegistrationHostId', d => 1034 },
        'mail_pdd_registration_host_id'    => { f => \&_plainvalue, n => 'MailPddRegistrationHostId',    d => 4 },

        'new_database_writes_attempts'   => { f => \&_plainvalue, n => 'NewDatabaseWritesAttempts',   d => 3 },
        'new_database_writes_delay'      => { f => \&_plainvalue, n => 'NewDatabaseWritesDelay',      d => 0.5 },

        'uid_collection_ranges' => { f => \&_plainvalue, n => 'UidCollectionRanges', m => 1 },

# db.conf
        'centraldbmaster'     => { f => \&_database,         n => 'CentralDbMaster',     m => 0 },
        'sharddbmaster'       => { f => \&_sharded_database, n => 'ShardDbMaster',       m => 0 },

        'centraldbslave'      => { f => \&_database,         n => 'CentralDbSlave',      m => 0, d => {} },
        'sharddbslave'        => { f => \&_sharded_database, n => 'ShardDbSlave',        m => 0, d => {} },

        'admcentraldbslave'   => { f => \&_database,         n => 'AdmCentralDbSlave',   m => 0, d => {} },
        'admsharddbslave'     => { f => \&_sharded_database, n => 'AdmShardDbSlave',     m => 0, d => {} },

        'heavycentraldbslave' => { f => \&_database,         n => 'HeavyCentralDbSlave', m => 0, d => {} },
        'heavysharddbslave'   => { f => \&_sharded_database, n => 'HeavyShardDbSlave',   m => 0, d => {} },

# db_credentials.conf
        'db_credentials'      => { f => \&_db_credentials,       n => 'DBProfile'},
# /db.conf

# access.conf
        'allow_from'          => {'f' => \&_allowfrom,           'n' => 'AllowFrom'},
# /access.conf
    );

    $Conf::ConfDir = undef;

    if ($httpdconf::etcpath && (-d $httpdconf::etcpath)){
        $Conf::ConfDir = $httpdconf::etcpath;
    }
    if ($main::etcpath && (-d $main::etcpath)){
        $Conf::ConfDir = $main::etcpath;
    }

    unless ($Conf::ConfDir && (-d $Conf::ConfDir)){
        die "Cannot discover etc directory";
    }
};

=head1 METHODS

=head2 new()

Считывает конфиги из .conf файлов
и загружает hardcode конфигурационные переменные.
Инициализирует несколько переменных $Conf::*

Конструктор, возвращает объект конфига.

=cut
sub new {
    my $class = shift;
    my $self = bless({}, $class);

    return ($self->_load_conf_files && $self->_load_hard_code) ? $self : undef;
}

=head2 GetValByPos($is_https, $param_for_http, $param_for_https)

Метод объекта.
Получает параметры:
1) флаг https соединения (см $Input::secure_mode)
2) название пераметра конфига для нешифрованного соединения
3) название пераметра конфига для шифрованного соединения
Определяет, какой из параметров сейчас актуален, считывает его и,
сбрасывает в return.

=cut
sub GetValByPos {
    my ($self, $is_https, @conf_params) = @_;

    my $pos = (not $is_https or $is_https !~ /^\d+$/) ? 0 : $is_https;
    my $conf_param = $conf_params[$pos];

    return defined $conf_param ? $self->GetVal($conf_param) : undef;
}

=head2 GetVal($conf_param1,...,$conf_paramN)

Метод объекта.
Получает параметры, которые необходимо считать/вернуть из конфига.
При необходимости обновляет значение переменной (см ключ reload в описании файла-конфига) и,
возвращает в зависимости от контекста либо значения всех переменных. либо только первой.

=cut
sub GetVal {
    my ($self, @conf_params) = @_;
    return unless @conf_params;

    # проверим переденные токены (параметры конфига) на существование в целом
    my %filestoread = ();
    foreach my $conf_param (@conf_params){
        unless(exists $Conf::Tokens{$conf_param}) {
            Common::Logs::IntErr("Unknown token $conf_param requested in GetVal, will return undefined value");
            next;
        }
        $filestoread{$Conf::Tokens{$conf_param}} = 1;
    }

    # обновим файлы с токенами, если такое требуется (см ключ reload в описании файла конфига)
    foreach my $filename (keys %filestoread) {
        unless ($self->_loadfile($filename)){
            Common::Logs::IntErr("Config file $filename not loaded");
        }
    }

    my @rv = map {
        ref $Conf::ConfCache{$_} eq 'CODE' ?
            $Conf::ConfCache{$_}->() :
            $Conf::ConfCache{$_}
    } @conf_params;

    return wantarray ? @rv : $rv[0];
}

=head2 GetHCVal($conf_param1,...,$conf_paramN)

Метод объекта.
Аналогичен GetVal, отличие в том, что работает с hardcode переменными конфига (см. %Conf::HardCoded).
Соответвенно, не перегружает никаких файлов.

Возвращаемое значение см. GetHCVal

=cut
sub GetHCVal {
    my ($self, @conf_params) = @_;

    my @rv = ();
    foreach my $conf_param (@conf_params){
        if (exists $Conf::HardCoded{$conf_param}) {
            push @rv, $Conf::HardCoded{$conf_param};
        } else {
            Common::Logs::ConfErr("Conf hardcoded object not defined for token '$conf_param'");
            push @rv, undef;
        }
    }
    return wantarray ? @rv : $rv[0];
}


=head1 INSIDE METHODS

=head2 _load_conf_files()

Внутренний метод объекта, загружает все конфиги, определённые в %Conf::Files.
Возвращает true.

=cut
sub _load_conf_files {
    my $self = shift;

    foreach my $filename (keys %Conf::Files){
        my $params = $Conf::Files{$filename};

        if ($params->{loader}) {
            unless ($params->{loader}->($filename)) {
                Common::Logs::DeBug("Configuration not loaded from $filename");
            }
        }
        else {
            unless ($self->_loadfile($filename)){
                Common::Logs::DeBug("Configuration not loaded from $filename");
            }
        }
    }

    return 1;
}


=head2 _readval($param_name, $conf)

Внутренний метод объекта, получает название параметра из конфига и
объект Config::ApacheFormat.
Считывает его, отдаёт на валидацию и возвращает результат.

Возвращает массив, первый элемент - значение переменной конфига,
второй - ошибка.

=cut
sub _readval {
    my ($self, $param_name, $conf) = @_;

    # если вдруг объект $conf не соотвествует ожиданиям
    return(undef, 'Undefined conf object') unless ref $conf eq 'Config::ApacheFormat';
    # если для этого параметра не определён обработчик в Helpers
    return(undef, "Helper for token $param_name is not defined") unless exists $Conf::Helpers{$param_name};

    # считаем из конфига переменную и протащим через обработчик значения
    my ($rv, $error) = &{$Conf::Helpers{$param_name}->{'f'}}($param_name, $conf, $Conf::Helpers{$param_name});

    # вернём ошибку или занчение переменной
    return (not defined $rv and $error) ? (undef, $error) : ($rv, undef);
}


=head2 _loadfile($filename)

Внутренний метод объекта, считывает указанный файл и парсит его.
Рассчитывает на файлы в apache like стиле.

Возвращает булево значение

=cut

sub _loadfile {
    my ($self, $filename) = @_;

    unless ($filename) {
        Common::Logs::IntErr('Bad call of Conf::_loadfile');
        return 0;
    }

    my $fileconf = $Conf::Files{$filename};
    my $reload   = $fileconf->{reload};
    my $time     = time;
    my $delta    = $time - ($fileconf->{last_check} || 0);

    return 1
      if $fileconf->{loader}; # Не перегружаем файлы с переопределённым loader'ом

    return 1
      if $reload and $delta <= $MIN_SECONDS_BETWEEN_STAT_CHECK;

    # уходим, если для данного файла не стоит флаг перезагрузки и мы его уже загружали
    return 1
      if not $reload and $fileconf->{timestamp};

    # определим полный путь к файлу
    my $filepath = File::Spec->catfile($Conf::ConfDir, $filename);
    unless (-f $filepath) {
        # если файл не найден, то вернём 0
        Common::Logs::IntErr("Config file $filepath does not exists");
        return 0;
    }

    # если для этого файла конфига стоит флаг перезагрузки, но файл не менялся, то
    # вернём 1
    # * timestamp не 0 только, если файл уже загружался
    my $mtime = [stat $filepath]->[9];
    $Conf::Files{$filename}->{last_check} = $time;

    return 1
      if $reload and $fileconf->{timestamp} == $mtime;

    # дефолтные параметры парсинга файла-конфига
    my %default_config_options = (
        inheritance_support => 0,
        include_support => 0,
        autoload_support => 0,
        case_sensitive => 0,
        fix_booleans => 1,
        expand_vars => 0,
        setenv_vars => 0,
        duplicate_directives => 'combine',
    );

    # создадим объект парсинга
    my $config = Config::ApacheFormat->new(
                    ref $Conf::Files{$filename}->{'options'} eq 'HASH' ?
                        %{$Conf::Files{$filename}->{'options'}} :
                        %default_config_options
    );
    # распарсим
    $config->read($filepath);

    if (ref $Conf::Files{$filename}->{'tokens'} eq 'ARRAY') {
        foreach my $token (grep {$_} @{$Conf::Files{$filename}->{'tokens'}}){
            my ($confvalue, $error) = $self->_readval($token, $config);
            if ($error) {
                Common::Logs::IntErr("Error loading configuration for token $token: $error");
                return;
            }
            $Conf::ConfCache{$token} = $confvalue;
        }
    }

    Common::Logs::DeBug("ConfFile: $filename " . ($Conf::Files{$filename}->{'timestamp'} ? 're' : '') . 'loaded');

    # запоминаем время последней модификации файла-конфига
    $Conf::Files{$filename}->{'config'} = $config;
    $Conf::Files{$filename}->{'timestamp'} = $mtime;

    return 1;
}

sub _ranges_loader {
    my $filename = shift;

    my $filepath = File::Spec->catfile($Conf::ConfDir, $filename);
    unless (-f $filepath) {
        die "Config file $filepath does not exists";
    }

    my @lines = do { local @ARGV = $filepath; <> };

    my (@cutoffs, @shards);

    for my $line (@lines) {
        chomp $line;
        next unless length $line;
        my ($cutoff, $shard) = split / /, $line;
        unless ($cutoff >= 0 and $shard >= 1) {
            die "$filepath has no content";
        }
        push @cutoffs, $cutoff;
        push @shards,  $shard;
    }

    if (@cutoffs and @shards) {
        $Conf::ConfCache{uid_collection_ranges} = {
            cutoffs => \@cutoffs,
            shards  => \@shards,
        };
    }
    else {
        die "Unknown ranges";
    }

    return 1;
}

=head2 _load_hard_code()

Внутренний метод объекта, иницализирует переменные, указанные
хардкодом в этом файле.
Инициализирует хэш %Conf::HardCoded

Возвращает булево значение

=cut
sub _load_hard_code {
    my $this = shift;

    # cookiename => ['hash_key', TTL, flag]
    # values are flags as follows:
    #0x0004 - set on login
    #0x0010 - set persistent
    #0x0020 - set persistent if user choosed so
    #0x0040 - set later if user have change-password-on-next-logon flag (sid=100)
    #0x0080 - url encode it
    #0x0200 - kill on logout
    #0x0400 - kill on soft logout
    #0x0800 - kill always
    #0x2000 - Do Not Empty on login if data not exsists

    $Conf::HardCoded{'Cookies'} = {
        'yandex_login'      => ['login_yandex',     undef,      0x0004 | 0x0020 | 0x0200 ],
        'Session_id'        => ['sessionid',        undef,      0x0004 | 0x0020 | 0x0040 | 0x0400 | 0x0200 ],
        'sessionid2'        => ['sessionid',        undef,      0x0200 ],
        'Secure_session_id' => ['secure_session',   undef,      0x0004 | 0x0400 | 0x0200 ],
        'my'                => ['my',               undef,      0x2000 | 0x0004 | 0x0010 ],
        'L'                 => ['l_info',           94608000,   0x0004 | 0x0010,],
        'fuid01'            => ['fuid01',           undef,      0 ],
        'fuid00'            => ['fuid00',           undef,      0 ],
        'ys'                => ['display_name',     undef,      0x2000 | 0x0004 | 0x0400 ],
        'yp'                => ['display_name',     undef,      0x2000 | 0x0004 | 0x0400 | 0x0010 ],
    };


    $Conf::HardCoded{'CookieCalend'} = {
        WeekFullName => [ qw(Sunday Monday Tuesday Wednesday Thursday Friday Saturday) ],
        WeekShortName => [ qw(Sun Mon Tue Wed Thu Fri Sat) ],
        MonthShortName => [ qw(Jan Feb Mar Apr May Jun Jul Aug Sep Oct Nov Dec) ]
    };


    # $ServiceInfoFieldList is a list of request parameters containing user related information
    # to be sent in the user request for password recover

    $Conf::HardCoded{'ServiceInfoFieldList'} = [qw(useragent ip_from ip_prox)];

    $Conf::HardCoded{LifetimeForever}  = 0x7FFFFFFF;

    $Conf::HardCoded{'YandexDomains'} = {
        'yandex.ru'  => 1,
        'yandex.ua'  => 1,
        'yandex.com' => 1,
        'yandex.com.tr' => 1,
        'yandex.by'  => 1,
        'yandex.kz'  => 1,
        'ya.ru'      => 1,
        'narod.ru'   => 1,
        'яндекс.рф'  => 1,
    };
    $Conf::HardCoded{YandexDomains}->{'yandex-team.ru'} = 1 if $this->GetVal('yandexteam');
    $Conf::HardCoded{ExternalDomains} = {
        'galatasaray.net' => 1,
    };

    $Conf::HardCoded{'from2sid'} = {
        mail => 2,
        narod => 4,
        fotki => 5,
        cards => 6,
        passport => 8,
        podpiska => 9,
        dz => 12,
        moikrug => 13,
        slova => 14,
        sauth => 15,
        so => 17,
        sfera => 18,
        balance => 19,
        money => 20,
        client => 21,
        mpop => 22,
        lenta => 23,
        partner => 24,
        market => 25,
        friends => 26,
        jabber => 27,
        kubok => 28,
        tv => 29,
        afisha => 30,
        calendar => 31,
        crm => 32,
        light => 33,
        nahodki => 34,
        yasms => 36,
        public => 37,
        kuda => 38,
        video => 39,
        kraski => 40,
        slovari => 41,
        wwwdgt => 42,
        disk => 44,
        pvo => 45, # ask
        narod2 => 46,
        webmaster => 47,
        metrika   => 48,
        maps      => 49,
        musicpartner => 50,
        radiojam => 51,
        wikimaps => 52,
        xml      => 53,
        music    => 54,
        mobilmail => 55,
        site     => 56,
        wdgt     => 57,
        social   => 58,
        cloud    => 59,
        newspartner => 60,
        galatasaray => 61,
        cloudpaid   => 64,
        phonenumber => 65,
        ### SOX
        # Номер sid'а, который используется как флаг sox-compliance на аккаунте.
        # Заводится через mode=admsubscribe, с помощью функции InsertStrongPWDInfo
        # в cgi-bin/DBAF/Safe.pm.
        strongpwd => 67,
        phonish => 68,
        simplereg => 69,
        tune      => 70,
        yandsearch => 74,
        gazeta     => 76,
        mobilecloud => 77,
        mobilemusic => 78,
        contest     => 79,
        cloudsoft   => 80,
        cloudweb    => 81,
        mynews          => 83,
        storedevconsole => 84,
        pi              => 85,
        publicdemo      => 86,
        gamification    => 87,
        islandsapi      => 90,
        postoffice      => 91,
        changepasswordonnextlogon => 100,
        backupsauth => 101,
        pddeula     => 102,
        pddadmin    => 104,
        mdalias => 105,
        toloka => 116,
        telephony => 117,
        invest => 119,
        bank => 125,
        mailapp_androidphone  => 201,
        mailapp_androidtablet => 202,
        mailapp_iphone        => 203,
        mailapp_ipad          => 204,
        mailapp_ipadbirdseye  => 205,
        mailexp   => 666,
        mailtest  => 667,
        betatest  => 668,
        yastaff   => 669,
        yacorp    => 670,
        vip       => 671,
        test      => 672,
    };

    $Conf::HardCoded{'sid2from'} = {};
    while (my ($key, $val) = each(%{$Conf::HardCoded{'from2sid'}})) {
        $Conf::HardCoded{'sid2from'}->{$val} = $key;
    }

    # check sid's for domain in mode=auth in this order
    $Conf::HardCoded{'login2sid'} = {
        'yandex.ru' => [2],
        'narod.ru' => [16, 2],
        'ya.ru' => [2],
        'yandex.com' => [2],
        'yandex.ua' => [2],
        'yandex.by' => [2],
        'yandex.kz' => [2],
        'яндекс.рф' => [2],
    };

    # modeset
    # determines what modes are available and on
    #used in Input GetCurrentState
    #see GetFSMTable also
    $Conf::HardCoded{'Modeset'} = {
        'check' => '0',
        'random' => '0',
        'error' => '1',
        'alive' => '1',

        'dump'          =>0,
    };
    #used in Input GetCurrentState
    #see GetFSMTable also
    $Conf::HardCoded{'MdModeset'} = {
        'check'         => 0,
        'random'        => 0,
        'error'         => 1,
        'alive'         => 0,
        'dump'          => 0,
    };

    # Default response stuff
    $Conf::HardCoded{'DefHeaders'} = {
        'ContentCharset' => 'utf-8',
        'StatusLine' => '200 OK',
        'ContentType' => 'text/html',
        'NoCache' => '1',
    };

    # Default request stuff
    $Conf::HardCoded{'DefActions'} = {
        DoProxy     => 1,
        DoQuery     => 0,
        DoBody      => 0,
        DoCookies   => 0,
        DoUAgent    => 1,
        DoReferer   => 1,
    };

    $Conf::HardCoded{'TestPrefixes'} = [
        'yandex-team',
        'xn--',
        'yndx',
    ];

=head2 $Conf::HardCoded{'mimicry'}

Насколько понял, msg это алиас для from в параметрах урла для кастомизации.
Название ключа всегда должно совпадать с аттрибутом from, т.к. они переопределяются и соотносятся.
По регулярке проверяется retpath, если не был указан from.

В AIO::Aliase,
- css_part   будет передан в шаблон в переменной From
*см SetPageStyle и _set_common_vars

=cut
    my $all_domains = '('.join('|', map {s/\./\\\./g; $_} keys %{$Conf::HardCoded{'YandexDomains'}}).')';
    $Conf::HardCoded{'mimicry'} = {
        mail => {
            'from'          => 'mail',
            'css_part'      => '_m',
            'retpath_re' => qr{^http[s]?://(www\.)?mail\.$all_domains(/.*)*$}i,
        },
        money => {
            'from'          => 'money',
            'css_part' => '_d',
            'retpath_re' => qr{^http[s]?://(www\.)?money\.$all_domains(/.*)*$}i,
        },
        mk => {
            'from'          => 'mk',
            'css_part' => '_mk',
            'retpath_re' => qr{^http[s]?://(www\.)?moikrug(\.$all_domains|\.ru)(/.*)*$}i,
        },
        friends => {
            'from'          => 'friends',
            'css_part' => '_fr',
            'retpath_re' => qr{^http[s]?://(www\.)?wow\.$all_domains(/.*)*$}i,
        },
        kuda => {
            'from'          => 'kuda',
            'css_part' => '_kuda',
            'retpath_re' => qr{^http[s]?://(www\.)?kuda\.$all_domains(/.*)*$}i,
        },
        pm => {
            'from'          => 'pm',
            'css_part' => '_pm',
            'retpath_re' => qr{^http[s]?://(www\.)?partner\.market\.$all_domains(/.*)*$}i,
        },
        market => {
            'from'          => 'market',
            'css_part' => '_market',
            'retpath_re' => qr{^http[s]?://(www\.)?market\.$all_domains(/.*)*$}i,
        },
        lenta => {
            'from'          => 'lenta',
            'css_part' => '_lenta',
            'retpath_re' => qr{^http[s]?://(www\.)?lenta\.$all_domains(/.*)*$}i,
        },
        fotki => {
            'from'          => 'fotki',
            'css_part' => '_fotki',
            'retpath_re' => qr{^http[s]?://(www\.)?fotki\.$all_domains(/.*)*$}i,
        },
        direct => {
            'from'          => 'direct',
            'css_part' => '_direct',
            'retpath_re' => qr{^http[s]?://(www\.)?direct\.$all_domains(/.*)*$}i,
        },
        video => {
            'from'          => 'video',
            'css_part' => '_video',
            'retpath_re' => qr{^http[s]?://(www\.)?video\.$all_domains(/.*)*$}i,
        },
        narod => {
            'from'          => 'narod',
            'css_part' => '_n',
            'retpath_re' => qr{^http[s]?://(www\.)?narod(\.$all_domains|\.ru)(?!/disk)(/.*)*$}i,
        },
        kraski => {
            'from'          => 'kraski',
            'css_part' => '_kraski',
            'retpath_re' => qr{^http[s]?://(www\.)?kraski\.$all_domains(/.*)*$}i,
        },
        cards => {
            'from'          => 'cards',
            'css_part' => '_cards',
            'retpath_re' => qr{^http[s]?://(www\.)?cards\.$all_domains(/.*)*$}i,
        },
        slovari => {
            'from'          => 'slovari',
            'css_part' => '_slovari',
            'retpath_re' => qr{^http[s]?://(www\.)?slovari\.$all_domains(/.*)*$}i,
        },
        narod2 => {
            'from'	 => 'narod2',
            'css_part'	 => '_n',
        },
        webmaster => {
            'from'	 => 'webmaster',
            'css_part'	 => '_webmaster',
            'retpath_re' => qr{^http[s]?://(www\.)?webmaster\.$all_domains(/.*)*$}i,
        },
        metrika => {
            'from'	 => 'metrika',
            'retpath_re' => qr{^http[s]?://(www\.)?metri[kc]a\.$all_domains(/.*)*$}i,
        },
        partner => {
            'from'	 => 'partner',
            'retpath_re' => qr{^https?://(www\.)?partners?\.$all_domains(/.*)*$}i,
        },
        # mailexp = mail Experementator
        mailexp => {
            'from'	 => 'mailexp',
        },
        maps => {
            'from'       => 'maps',
            'retpath_re' => qr{^https?://(www\.)?maps?\.$all_domains(/.*)*$}i,
        },
        musicpartner => {
            'from'       => 'musicpartner',
            'retpath_re' => qr{^https?://(www\.)?music-?partners?\.$all_domains(/.*)*$}i,
        },
        radiojam => {
            'from'       => 'radiojam',
            'retpath_re' => qr{^https?://(www\.)?radioprobki\.$all_domains(/.*)*$}i,
        },
        wikimaps => {
            'from'       => 'wikimaps',
            'retpath_re' => qr{^https?://(www\.)?wikimaps\.$all_domains(/.*)*$}i,
        },
        xml => {
            'from'       => 'xml',
            'retpath_re' => qr{^https?://(www\.)?xml\.$all_domains(/.*)*$}i,
        },
        music => {
            'from'       => 'music',
            'retpath_re' => qr{^https?://(www\.)?music\.$all_domains(/.*)*$}i,
        },
        site => {
            'from'       => 'site',
            'retpath_re' => qr{^https?://(www\.)?site\.$all_domains(/.*)*$}i,
        },
        cloud => {
            'from'       => 'cloud',
            'retpath_re' => qr{^http[s]?://(www\.)?disk\.$all_domains(/.*)*$}i,
        },
        contest => {
            'from'       => 'contest',
            'retpath_re' => qr{^http[s]?://(www\.)?contest\.$all_domains(/.*)*$}i,
        },
    };

    # arrays constructed as follows:
    #   ['ask_user_or_not', 'path_redirect_to', reserved]
    $Conf::HardCoded{'Subscribe'} = {
        2 => [0, 'http://mail.yandex.ru/'],
        4 => [0, 'http://narod.yandex.ru/userarea/after_register.xhtml'],
        5 => [0, 'http://fotki.yandex.ru/'],
        6 => [0, 'http://cards.yandex.ru/'],
        9 => [0, 'http://news.yandex.ru/'],
        12 => [0, 'http://dz.yandex.ru/'],
        13 => [0, 'http://moikrug.ru/'],
        14 => [0, 'http://direct.yandex.ru/'],
        17 => [0, 'http://so.yandex.ru/'],
        15 => [0, 'http://sauth.yandex.ru/'],
        18 => [0, 'http://sfera.yandex.ru/'],
        19 => [0, 'http://balance.yandex.ru/'],
        20 => [0, 'https://money.yandex.ru/'],
        21 => [0, 'http://client.yandex.ru/'],
        23 => [0, 'http://lenta.yandex.ru/'],
        24 => [0, 'http://partner.yandex.ru/'],
        25 => [0, 'http://market.yandex.ru/'],
        26 => [0, 'http://friends.yandex.ru/'],
        27 => [0, 'http://online.yandex.ru/'],
        28 => [0, 'http://kubok.yandex.ru/'],
        29 => [0, 'http://tv.yandex.ru/'],
        30 => [0, 'http://afisha.yandex.ru/'],
        31 => [0, 'http://calendar.yandex.ru/'],
        32 => [0, 'http://crm.yandex.ru/'],
        37 => [0, undef],
        38 => [0, undef],
        39 => [0, 'http://video.yandex.ru/'],
        40 => [0, 'http://kraski.yandex.ru/'],
        41 => [0, 'http://slovari.yandex.ru/'],
        42 => [0, undef],
        44 => [0, sub {sprintf 'http://%s.narod2.yandex.ru/edit', $_[0]->[0]} ],
        46 => [0, 'http://narod.yandex.ru/userarea/'],
        47 => [0, 'http://webmaster.yandex.ru/'],
        48 => [0, 'http://metrika.yandex.ru/'],
        49 => [0, 'http://maps.yandex.ru/'],
        50 => [0, 'http://music-partner.yandex.ru/'],
        51 => [0, 'http://radioprobki.yandex.ru/'],
        52 => [0, 'http://n.maps.yandex.ru/'],
        53 => [0, 'http://xml.yandex.ru/ip.xml'],
        54 => [0, 'http://music.yandex.ru/'],
        55 => [0, 'http://mail.yandex.ru/'],
        56 => [0, 'http://site.yandex.ru/'],
        57 => [0, 'http://wdgt.yandex.ru/'],
        58 => [0, 'http://social.yandex.ru/'],
        59 => [0, 'http://disk.yandex.ru/'],
        60 => [0, 'http://partner.news.yandex.ru/'],
        67 => [0, 'http://strongpwd.yandex.ru/'],
        69 => [0, undef],
        76 => [0, 'http://gazeta.yandex.ru/'],
        77 => [0, undef],
        78 => [0, undef],
        90 => [0, 'http://islands.yandex.ru/'],
        91 => [0, 'http://postoffice.yandex.ru/'],
        100 => [0, undef],
        102 => [0, 'http://mail.yandex.ru/'],
        104 => [0, 'http://pdd.yandex.ru/'],
        116 => [0, 'https://toloka.yandex.ru/'],
        117 => [0, 'https://telephony.yandex.ru/'],
        666 => [0, 'http://mail.yandex.ru/'],
        667 => [0, 'http://mail.yandex.ru/'],
        668 => [0, undef],
        669 => [0, undef],
        670 => [0, undef],
    };

    # user with this services should not be dropped, just blocked instead
    # Внимание! Копия списка есть у ПДД (juanych@). При обновлении нужно оповестить их.
    $Conf::HardCoded{'Protected'} = {
        14 => 1, # Директ
        15 => 1, # Платёжный пароль
        19 => 1, # Баланс-Билинг
        20 => 1, # Деньги
        24 => 1, # Рекламная Сеть (бывший Партнерский Интерфейс (ПИ))
        64 => 1, # Оплаченный Диск
        78 => 1, # Мобильное приложение Музыки
        85 => 1, # Новый Партнерский Интерфейс
        104 => 1, # Администратор ПДД домена
        116 => 1, # Толока
        117 => 1, # Телефония
        119 => 1, # Инвестиции
        125 => 1, # Банк
        670 => 1, # Корпоративный аккаунт
    };

    $Conf::HardCoded{'DeleteEvenBlocked'} = {
        44 => 1, # Народный диск
    };

    # list of services are available for subscription only with mode=admsubscribe
    $Conf::HardCoded{'ProtectedSubscribe'} = {
        4  => 1,
        15 => 1,
        20 => 1,
        34 => 1,
        36 => 1,
        39 => 1,
        42 => 1,
        50 => 1,
        55 => 1,
        56 => 1,
        57 => 1,
        58 => 1,
        59 => 1,
        60 => 1,
        61 => 1,
        67 => 1,
        69 => 1,
        70 => 1,
        74 => 1,
        77 => 1,
        78 => 1,
        79 => 1,
        80 => 1,
        81 => 1,
        83 => 1,
        84 => 1,
        85 => 1,
        86 => 1,
        87 => 1,
        100 => 1,
        102 => 1,
        104 => 1,
        105 => 1,
        668 => 1,
        669 => 1,
        670 => 1,
        671 => 1,
        672 => 1,
    };

    # Список подписок, которые нужно скрыть от пользователя при попытке удаления аккаунта
    $Conf::HardCoded{HiddenServices} = [ qw/4 8 27 33 34 36 37 44 46 55 58 67 69 101 102 104 105 666 667 668 669/ ];

    $Conf::HardCoded{'DomainMailRootDir'} = 'for';

    return 1;
}

=head1 INSIDE HELPERS FUNCTIONS

=head2 _plainvalue($token, $conf, $args)

Принимает название параметра, объект Config::ApacheFormat, и соотвествующее значение ключа из %Conf::Helpers
Подставляет при необходимсоти дефолтное значение
и делает небольшую обработку в случае наличия параметра t в описании

Возвращает зачение

=cut
sub _plainvalue {
    my ($token, $conf, $args) = @_;

    #определимся с параметрами
    my $name = $args->{'n'} || $token;

    # считаем из конфига
    my $rv = $conf->get($name);

    # если чтение из конфига ничего не вернуло, то
    unless (defined $rv) {
        if ($args->{'m'}){
            # если параметр обязательный
            return (undef, "Mandatory value for \"$token\" (name \"$name\") not found in config");
        } elsif (defined $args->{'d'}) {
            # если есть дефолтное значение
            # запишем в лог
            Common::Logs::DeBug("Value for \"$token\" (name \"$name\") not found in config, setting default");
            # и продолжим обработку
            $rv = $args->{'d'};
        } else {
            return $rv;
        }
    }

    # значение вернулось, проверим тип
    if (defined $args->{'t'} and $args->{'t'} eq 'HOSTNAME'){
       $rv =~ s,/+$,,;
       $rv .= '/';
    }

    my $attr = $args->{attr};
    # вернём значение
    if ($attr and $attr eq 'domain_dependence') {
        $rv = _sanitize_domain_depended_value($rv);
        return _is_pda_mutation_required($args)
          ? sub { _get_mutated_value_if_pda($args) || _format_domain_depended_value($rv) }
          : sub { _format_domain_depended_value($rv) };
    } elsif ($attr and $attr eq 'mda_dependence') {
        # необходимость mda - динамично
        # 0 - mda не требуется
        # 1 - mda требуется
        # См. passport.conf, переменную WOMdAuth
        return sub{
            if ($Input::Cookie->{mda} eq '0') {
                return 0;
            }
            elsif ($Input::Cookie->{mda} eq '1') {
                return 1;
            }
            return 0 if $Input::display->is_pda and $conf->get('DisableMdaForMobile');
            my @hosts_with_mda = split ',', ($rv || 'yandex.ru');
            for (@hosts_with_mda) {
                return 1 if $_ eq $Input::Input->host;
            };
            return 0;
        };
    } else {
        # общий случай - возврат plainvalue
        return _is_pda_mutation_required($args)
          ? sub { _get_mutated_value_if_pda($args) || $rv }
          : $rv;
    }
}

=head2 _need_eval($token, $conf, $args)

Принимает название параметра, объект Config::ApacheFormat, и соотвествующее значение ключа из %Conf::Helpers
Подставляет при необходимсоти дефолтное значение
Значение отправляется на eval. Это не хороший вариант для "внешних" переменных, но для собственного конфига,
вполне приемлимо.

=cut
sub _need_eval {
    my ($token, $conf, $args) = @_;

    #определимся с параметрами
    my $name = $args->{'n'} || $token;

    # считаем из конфига
    my $rv = $conf->get($name);

    # если чтение из конфига ничего не вернуло, то
    unless (defined $rv) {
        if ($args->{'m'}){
            # если параметр обязательный
            return (undef, "Mandatory value for \"$token\" (name \"$name\") not found in config");
        } elsif (defined $args->{'d'}) {
            # если есть дефолтное значение
            # запишем в лог
            Common::Logs::DeBug("Value for \"$token\" (name \"$name\") not found in config, setting default");
            # и продолжим обработку
            $rv = $args->{'d'};
        } else {
            return $rv;
        }
    }

    # значение вернулось, обратим значение в хэшреф
    my $ret = undef;
    eval{$ret = eval($rv)};
    if ($@ or not defined $ret) {
        return (undef, "Value for \"$token\" (name \"$name\") can't eval from value \"$rv\"");
    }

    return _is_pda_mutation_required($args)
      ? sub { _get_mutated_value_if_pda($args) || $ret }
      : $ret;
}


=head2 _file_path($token, $conf, $args)

Принимает название параметра, объект Config::ApacheFormat, и соотвествующее значение ключа из %Conf::Helpers
Подставляет при необходимсоти дефолтное значение
Получает путь к файлу, проверяет его наличие и доступ на чтение.

=cut
sub _file_path {
    my ($token, $conf, $args) = @_;

    #определимся с параметрами
    my $name = $args->{'n'} || $token;

    # считаем из конфига
    my $rv = $conf->get($name);

    # если чтение из конфига ничего не вернуло, то
    unless (defined $rv) {
        if ($args->{'m'}){
            # если параметр обязательный
            return (undef, "Mandatory value for \"$token\" (name \"$name\") not found in config");
        } elsif (defined $args->{'d'}) {
            # если есть дефолтное значение
            # запишем в лог
            Common::Logs::DeBug("Value for \"$token\" (name \"$name\") not found in config, setting default");
            # и продолжим обработку
            $rv = $args->{'d'};
        } else {
            return $rv;
        }
    }

    # значение вернулось, обратим значение в хэшреф
    unless (-f $rv and -r _) {
        die "Value for \"$token\" (name \"$name\") is not file or can't read";
    }

    return $rv;
}

=head2 _as_array($token, $conf, $args)

Принимает название параметра, объект Config::ApacheFormat, и соотвествующее значение ключа из %Conf::Helpers
Подставляет при необходимсоти дефолтное значение
Значение разбивается по указанному аттрибуту 'splitter' или через ',' по умолчанию

=cut
sub _as_array {
    my ($token, $conf, $args) = @_;

    #определимся с параметрами
    my $name = $args->{'n'} || $token;

    # считаем из конфига
    my $rv = $conf->get($name);

    # если чтение из конфига ничего не вернуло, то
    unless (defined $rv) {
        if ($args->{'m'}){
            # если параметр обязательный
            return (undef, "Mandatory value for \"$token\" (name \"$name\") not found in config");
        } elsif (defined $args->{'d'}) {
            # если есть дефолтное значение
            # запишем в лог
            Common::Logs::DeBug("Value for \"$token\" (name \"$name\") not found in config, setting default");
            # и продолжим обработку
            $rv = $args->{'d'};
        } else {
            return $rv;
        }
    }

    # значение вернулось, разобъём
    my $splitter = $args->{'splitter'} || ',';

    return [split $splitter, $rv];
}


=head2 _db_credentials($name)

Отвечает за профили с БД-креденшлами
Возвращает хэшреф {user => $DBUser, pass => $DBPassword}

=cut
sub _db_credentials {
    my (undef, $conf, undef) = @_;

    my @blocks_names = $conf->get('DBProfile');
    return (undef, "Required <DBProfile *> block not found in config") unless @blocks_names;

    my $result = {};

    for my $block_name (@blocks_names) {
        my $block = $conf->block(@$block_name);
        my $user = $block->get('DBUser');
        my $pass = $block->get('DBPassword');
        unless ($user && $pass) {
            return (undef, "Error parsing config: DBUser and DBPassword are required values for <DBProfile $block_name> block in config");
        }
        $result->{$block_name->[1]} = {user => $user, pass => $pass};
    }

    return $result;
}


=head2 _database($token, $conf, $args)

Отвечает за параметры подключения к базе
Принимает название параметра, объект Config::ApacheFormat, и соотвествующее значение ключа из %Conf::Helpers
Подставляет при необходимсоти дефолтное значение

Возвращает хэшреф параметров подключения

=cut
sub _database {
    my ($token, $conf, $args) = @_;

    # смотрим название блока, описывающего базу
    my $blockname = $args->{'n'};
    unless ($blockname){
        return (undef, "Block name (\$args->{'n'}) is not defined in _database call for token $token");
    }

    # считываем блок
    my $block = eval {$conf->block(Database => $blockname)};
    # блока не было или считывание не удачное
    unless ($block) {
        # если параметр обязательный
        if ($args->{m}) {
            return (undef, "Mandatory value for \"$token\" (name \"$blockname\") not found in config");
        } elsif ($args->{d}) {
            # если есть дефолтное значние
            Common::Logs::DeBug("Value for \"$token\" (name \"$blockname\") not found in config, setting default");
            return $args->{d};
        } else {
            return (undef, "Required <Database $blockname> block not found in config");
        }
    }

    # считываем из блока параметры подключения
    my $dsn = $block->get('DSN');
    my $db_profile = $block->get('DBProfile');
    # если считывание параметров прошло не успешно
    unless ($dsn && $db_profile) {
        return (undef, "Error parsing config: DSN and DBProfile are required values for <Database $blockname> block in config");
    }

    # дополнительные параметры подключения
    my $attr = {};
    my $reise_error = $block->get('RaiseError');
    $attr->{RaiseError} = $reise_error if defined $reise_error;

    # возвращаем параметры подключения
    return {'src' => $dsn, 'profile' => $db_profile, 'attr' => $attr};
}

sub _sharded_database {
    my ($token, $conf, $args) = @_;

    my @blocks_names = $conf->get($args->{n});
    unless (@blocks_names) {
        if ($args->{m}) {
            return (undef, "Required <$args->{n} *> block not found in config");
        }
        else {
            return {};
        }
    }

    my $result = {};

    for my $block_names (@blocks_names) {
        my $block = $conf->block(@$block_names);

        my $dsn  = $block->get('DSN');
        my $db_profile = $block->get('DBProfile');
        my $group_name = $block->get('GroupName');

        unless ($dsn and $db_profile) {
            return (undef, "Error parsing config: DSN and DBProfile are required values for <" . join(' ', @$block_names) . "> block in config");
        }

        my $shard_id = $block_names->[1];
        my $settings = {
            dsn  => $dsn,
            profile => $db_profile,
            shard_id => $shard_id,
            group_name => $group_name,
        };

        $result->{$shard_id} = $settings;
    }

    return $result;
}

sub _by_any_id {
    my ($token, $conf, $args) = @_;

    my $name = $args->{n};
    return (undef, "Variable name is not defined in _by_any_id call")
      unless $name;

    my %result = $conf->get($name);
    return \%result;
}

=head2 _httpreferallow($token, $conf, $args)

Принимает название параметра, объект Config::ApacheFormat, и соотвествующее значение ключа из %Conf::Helpers

Возвращает хэшреф с ключами:
- hosts - список хостов
- addresses - список подсетей

=cut
sub _httpreferallow {
    my ($token, $conf, $args) = @_;

    my $name = $args->{'n'};
    unless ($name){
        return(undef, "Variable name (\$args->{'n'}) is not defined in _httpreferallow call");
    }

    my @all_blocks = $conf->get($name);

    # считем блоки с этим $token и, соберём уникальные значения
    my $ret = {
        hosts => {},
    };
    foreach my $each_block (@all_blocks) {
        my $ret_block = {};

        my $block = $conf->block($each_block);
        my @block_hosts = $block->get('HOST');
        foreach my $each_host (@block_hosts) {
            $ret->{hosts}->{lc $each_host}++;
        }
    }

    return {'hosts' => [keys %{ $ret->{hosts} }]};
}

=head2 _allowfrom($token, $conf, $args)

Принимает название параметра, объект Config::ApacheFormat, и соотвествующее значение ключа из %Conf::Helpers

Возвращает хэшреф

=cut
sub _allowfrom {
    my ($token, $conf, $args) = @_;

    my $name = $args->{'n'};
    unless ($name){
        return(undef, "Variable name (\$args->{'n'}) is not defined in _allowfrom call");
    }

    my $all_modes = {};
    my @entries = $conf->get($name);
    foreach my $entry (@entries){
        my $block = $conf->block($entry);
        my $mode = $entry->[1];

        my @ips = $block->get('IP');
        my $curr_ip;
        foreach my $ip (@ips) {
            my $ipo = IP->new($ip);
            if ($ipo->is_valid) {
                $curr_ip = $ip;
                if ($all_modes->{$mode}->{'allow-ip'}->{$ip}){
                    Common::Logs::ConfErr("Semantic dublicate ip $ip in <$name $mode> block, ignoring");
                    next;
                }
                $all_modes->{$mode}->{'allow-ip'}->{$ip} = {};
            } elsif ($ip =~ /^([a-z][a-z0-9]*)$/ && defined($curr_ip)) {
                my $from = $ip;
                $all_modes->{$mode}->{'allow-ip'}->{$curr_ip}->{'from'}->{$from} = 1;
            } else {
                Common::Logs::ConfErr("Bad IP $ip in <$name $mode> block, ignoring");
                undef $curr_ip;
                next;
            }
        }

        my @subnets = $block->get('SUBNET');
        my $curr_subnet;
        foreach my $subnet (@subnets) {
            if ($subnet =~ /:/) {
                my $patricia6;

                if ($subnet =~ /\@/) {
                    my ($project_id, $mask) = split /\@/, $subnet;
                    $project_id = lc $project_id;
                    $project_id =~ s/^0+//;

                    $patricia6 = $all_modes->{$mode}{'allow-trypo'}{$project_id} ||= Net::Patricia->new(AF_INET6);
                    $subnet = $mask;
                } else {
                    $patricia6 = $all_modes->{$mode}{'allow-patricia6'} ||= Net::Patricia->new(AF_INET6);
                }

                eval {
                    $patricia6->add_string($subnet, $subnet);
                };

                if ($@) {
                    Common::Logs::ConfErr("Bad SUBNET $subnet in <$name $mode> block, ignoring");
                }

                next;
            }

            my $block = Net::Netmask->new2($subnet);
            if (defined($block)) {
                $curr_subnet = $subnet;
                if ($all_modes->{$mode}->{'allow-subnet'}->{$subnet}){
                    Common::Logs::ConfErr("Semantic dublicate SUBNET $subnet in <$name $mode> block, ignoring");
                    next;
                }
                $all_modes->{$mode}->{'allow-subnet'}->{$subnet} = {};
                $all_modes->{$mode}->{'allow-subnet'}->{$subnet}->{'block'} = $block;
            } elsif ($subnet =~ /^([a-z][a-z0-9]*)$/ && defined($curr_subnet)) {
                my $from = $subnet;
                $all_modes->{$mode}->{'allow-subnet'}->{$curr_subnet}->{'from'}->{$from} = 1;
            } else {
                Common::Logs::ConfErr("Bad SUBNET $subnet in <$name $mode> block, ignoring");
                undef $curr_subnet;
                next;
            }
        }
    }

    # FIXME: We are migrate from AllowIP to AllowFrom
    # this code for compatibility
    #
    my @allowip_entries = $conf->get('AllowIP');
    foreach my $entry (@allowip_entries){
        my $mode = $entry->[1];
        Common::Logs::AdMin("[WARNING] Old access.conf style. Replace <AllowIP $mode> with <AllowFrom $mode>");
        my $block = $conf->block($entry);
        my @ips = $block->get('IP');
        foreach my $ip (@ips) {
            next if (defined($all_modes->{$mode}->{'allow-ip'}->{$ip}));
            $all_modes->{$mode}->{'allow-ip'}->{$ip} = {};
        }
    }

    return $all_modes;
}

sub _sanitize_domain_depended_value {
    my $value = shift;

    $value =~ s/%/%%/g;
    $value =~ s/yandex\.ru/%s/;

    return $value;
}

sub _format_domain_depended_value {
    my $format = shift;

    my $replacement = $Input::Input ? $Input::Input->{host} : 'yandex.ru';
    my $result = sprintf $format, $replacement;

    return $result;
}

sub _is_pda_mutation_required {
    shift->{pda_mutation} ? 1 : 0;
}

sub _get_mutated_value_if_pda {
    my $args = shift;

    return unless my $display = $Input::display;
    return unless $display->is_pda;
    return $main::Conf->GetVal($args->{pda_mutation});
}

1;
