package DBAF::Safe;

use strict;
use utf8;

use Carp;
use Data::Dumper;
use Log::Any '$log';
use List::MoreUtils 'uniq';
use LWP::UserAgent;
use MIME::Base64;
use Scalar::Util 'blessed';
use Time::HiRes qw(gettimeofday tv_interval);
use Try::Tiny;

use Common::Geobase;
use Database::Converter;
use Model::Answer;
use Model::Builder;

my %CACHE;
my $PROTECTED_SIDS;
my @UNPROTECTED_SIDS;
my @UNPROTECTED_ATTRIBUTES;
my @UNPROTECTED_ALIASES;

# a kind of constructor
sub new {
    my $proto = shift;
    unless(defined($proto)) {
        return undef;
    }
    my $class = ref($proto) || $proto;
    my $this = {};
    Common::Logs::LogDebug("DBAF::Safe::new(): started");

    my $super = shift;
    if(defined($super)) {
        $this->{'SUPER'} = $super;
    } else {
        Common::Logs::IntErr("In NewSAFE: superclass is not accessible");
        $this->{'SUPER'} = 0;
    }

    bless($this, $class);

    $this->FlushCache();

    return $this;
}

sub GetDbhByConfig($) {
    my $confname = shift;
    unless($confname){
        Common::Logs::IntErr('Bad Call of GetDbhByConfig');
        Common::Logs::Caller();
        return undef;
    }
    my $dbconf = $main::Conf->GetVal($confname);
    unless($dbconf){
        Common::Logs::IntErr("Db config by '$confname' not found");
        return undef;
    }
    my $db_credentials = $main::Conf->GetVal('db_credentials');
    unless(exists $db_credentials->{$dbconf->{profile}}){
        Common::Logs::IntErr("Db profile '$dbconf->{profile}' not found");
        return undef;
    }
    my $dbprofile = $db_credentials->{$dbconf->{profile}};

    unless($dbconf->{'src'} and $dbconf->{'user'} and $dbconf->{'passwd'}){
        Common::Logs::AdMin("Config varible '$confname' seems not to be DB config (no src, user and passwd) ");
    }
    my $dbh = DBI->connect(
        $dbconf->{'src'},
        $dbprofile->{'user'},
        $dbprofile->{'pass'},
        $dbconf->{'attr'},
        );
    unless($dbh) {
        my $errmsg = $DBI::errstr;
        Common::Logs::DbErr("DB connection on '$confname' failed: ".$errmsg);
        return undef;
    }
    #
    # For utf support in latin 1 tables and utf8 ones
    # may be we should use
    my $res = $dbh->do('set names binary');
    unless($res){
        Common::Logs::DbErr($dbh->errstr." in sql initial sql st names");
        # return undef;
    }
    return $dbh;
}

our $db_encoding_zone = 'db';

sub CheckNameOccupied {
    my ($self, $login, $args) = @_;

    return undef
      unless length $login;

    my $normalized_login = lc $login;
    $normalized_login =~ tr/./-/
      unless $login =~ /\@/;

    my %params = (
        logins          => [ $normalized_login ],
        ignore_stoplist => $args->{ignore_stop_words},
    );
    my $requester = AIO::Requester->new(1);
    my $statuses  = eval { $requester->LoginOccupation(%params) };

    return undef
      if $@;

    my $status = $statuses->{$normalized_login};

    return 'STOP_WORD' if $status eq 'stoplist';
    return 0           if $status eq 'free';
    return 2;
}

sub SetKarmaPrefix {
    my $this = shift;
    my $uid  = shift;
    my $prefix = shift;

    unless ($this and $uid and defined $prefix) {
        $this->{'ERROR'} = 'Internal';
        return undef;
    }

    my $karma = $this->GetEnaKarma($uid);

    return undef
      unless $karma;

    my $old_karma = $karma->{karma};
    my $suffix    = $old_karma % 1000;
    my $new_karma = $prefix * 1000 + $suffix;

    my $result = $this->ConverterDo('set_attributes' => $uid, 'karma.value' => $new_karma);

    return undef unless defined $result;
    return 1;
}

sub GetEnaKarmaGLogout {
    my ($self, $uid) = @_;

    my $account = $self->GetAccount($uid);

    return undef unless $account;

    my $res = {
        ena     => $account->is_pdd ? $account->is_enabled_pdd : $account->is_enabled,
        glogout => $account->global_logout_datetime,
        karma   => $account->karma->value,
        acl_id  => $account->karma->activation_datetime,
    };

    return $res;
}

sub GetEnaAndGLogout {
    my ($self, $uid) = @_;

    my $res = $self->GetEnaKarmaGLogout($uid);
    delete $res->{karma} if $res;

    return $res;
}

sub GetEnaKarma {
    my ($this, $uid) = @_;

    my $res = $this->GetEnaKarmaGLogout($uid);
    if ($res){
        delete ($res->{glogout});
    }
    return $res;
}

sub sid_enable {
    my ($self, $uid, $sid) = @_;

    my $account = $self->GetAccount($uid);

    return undef unless $account;

    my $result = $account->subscriptions->get($sid)->is_exists;

    return $result;
}

sub GlobalLogout {
    my ($self, $uid, $glogout) = @_;

    my $account = $self->GetAccount($uid);

    return undef unless $account;

    my %attributes = (
        'account.global_logout_datetime' => $glogout,
    );

    my $result = $self->ConverterDo('set_attributes' => $uid, %attributes);

    return undef unless defined $result;
    return 1;
}

# returns list of services by given uid
# has is $hash->{sid} = ['login', suid, rule]
sub GetActiveServices {
    my ($self, $uid) = @_;

    my $account = $self->GetAccount($uid);

    return undef unless $account;

    my $list = $account->subscriptions->list;
    my $result = {};

    for my $item (@$list) {
        my $login = $item->login || $account->machine_readable_login;
        my $login_rule = $item->login_rule;
        $login_rule = $login_rule & 0b011
          if $item->sid == 8;
        $result->{ $item->sid } = [ $login, $item->suid || 1, $login_rule ];
    }

    return $result;
}

### TODO этот вызов используется в нескольких местах для проверка login_rule = 2|3 на sid = 33, но его мы не переносим в новую базу - нужно проверить
sub GetServiceInfo {
    my ($self, $uid, $sid) = @_;

    my $account = $self->GetAccount($uid);

    return undef unless $account;

    my $subscription = $account->subscriptions->get($sid);

    return {}
      unless $subscription->is_exists;

    my $result = {
        suid       => $subscription->suid || 1,
        login      => $subscription->login || $account->machine_readable_login,
        login_rule => $subscription->login_rule,
        host_id    => $subscription->host_id,
    };
    if ($sid == 8) {
        $result->{login_rule}       = $result->{login_rule} & 0b011;
    }

    return $result;
}

sub FlushCache {
    shift->{cache} = {};
}

sub SaveCachedData {
    my ($self, $method, $uid, $data) = @_;
    $self->{cache}{$method}{$uid} = $data;
    return 1;
}

sub GetCachedData {
    my ($self, $method, $uid) = @_;
    my $data = $self->{cache}{$method}{$uid} || undef;
    return $data;
}

sub GetUserInfo {
    my ($self, $uid) = @_;

    return undef unless $uid;

    my $cache = $self->GetCachedData('UserInfo', $uid);
    return $cache if $cache;

    my $account = $self->GetAccount($uid);
    return undef unless $account;

    my $person = $account->person;
    my $hint   = $account->hint;
    my $country = Common::Geobase::NormalizeCountryCode($person->country);
    my $info = {
        uid          => $uid,
        login        => $account->machine_readable_login,
        login_wanted => $account->human_readable_login,
        display_name => $account->display_name,

        fio       => $person->name,
        iname     => $person->firstname,
        firstname => $person->firstname,
        fname     => $person->lastname,
        lastname  => $person->lastname,
        sex       => $person->sex,
        xcountry  => $country,
        country   => $country,
        city      => $person->city,
        timezone  => $person->timezone,
        lang      => $person->language,
        regdate   => $account->registration_datetime,
        reg_date  => $account->registration_datetime,

        # hint нужен только в mode=changereg
        hintq     => $hint->question->id,
        hintqtext => $hint->question->text,
        hinta     => $hint->answer->text,
    };

    my $birthday = $person->birthday;
    if (defined $birthday && not $birthday eq '0000-00-00') {
        my ($year, $month, $day) = split(/-/, $birthday);

        $info->{byear}  = $year;
        $info->{bmonth} = $month;
        $info->{bday}   = $day;
        $info->{bdate}  = sprintf '%02d.%02d.%04d', $day, $month, $year;
    }

    if ($account->is_pdd) {
        $info->{domain}    = $account->domain->value;
        $info->{domain_id} = $account->domain->id;
    }

    $self->SaveCachedData('UserInfo', $uid, $info);

    return $info;
}

sub is_human_registration { 0 }

sub UpdateAccountInfo ($$$) {
    my ($this, $uid, $toupdate, $AllVals) = @_;
    unless($this and  $uid and $toupdate) {
        $this->{'ERROR'} = 'Internal';
        return undef;
    }

    my $changes = {};
    my $changess= {};

    if(defined($toupdate->{hintq})) {
        $changess->{hintq} = $this->HintQuestion($AllVals);
        $changess->{hintq} = ''
          if $changess->{hintq} =~ /^0/;
        undef($toupdate->{hintq});
        undef($toupdate->{udhintq});
    }
    if(defined($toupdate->{hinta})){
        $changess->{hinta} = $toupdate->{hinta};
        undef($toupdate->{hinta});
    }

    my %fields = (%$changes, %$changess);
    my %attributes = $this->ConvertFieldsToAttributes(update => %fields);

    return 1 unless %attributes;

    my $result = $this->ConverterDo('set_attributes' => $uid, %attributes);

    return undef unless defined $result;

    $this->LogAttributesAsStatboxEvents($uid, update => %attributes);

    return 1;
}

sub ConvertFieldsToAttributes {
    my $self = shift;
    our ($method, %fields) = @_;

    sub is_updated_field {
        my $field = shift;
        return 1 if $method eq 'update' and exists $fields{$field};
        return 1 if $method eq 'insert' and length $fields{$field};
        return 0;
    };

    my %attributes;

    if (is_updated_field('hintq')) {
        my $hintq = $fields{hintq};
        $hintq = substr $hintq, 0, 100;
        $hintq = undef
          if $hintq =~ /^0/;
        $attributes{'hint.question.serialized'} = $fields{hintq}
          if $hintq or $method eq 'update';
    }

    if (is_updated_field('hinta')) {
        my $hinta = $fields{hinta};
        $hinta = substr $hinta, 0, 100;
        $attributes{'hint.answer.encrypted'} = $hinta;
    }

    return %attributes;
}

sub LogAttributesAsStatboxEvents {
    my $self = shift;
    my ($uid, $method, %attributes) = @_;

    my $event_operation
      = $method eq 'update' ? 'updated'
      : $method eq 'insert' ? 'created'
      :                       undef;

    return unless $event_operation;

    my %EVENTS = (
        'hint.question.serialized' => { entity => 'hint.question',      log_new_value => 0 },
        'hint.answer.encrypted'    => { entity => 'hint.answer',        log_new_value => 0 },
    );

    while (my ($attribute_name, $event) = each %EVENTS) {
        my $attribute_value = $attributes{$attribute_name};

        next if not length $attribute_value;

        my @event_args = (
            $uid,
            $event->{entity} || $attribute_name,
            $event_operation,
        );

        if ($event->{log_new_value}) {
            $attribute_value = $event->{default_value}
              unless length $attribute_value;
            push @event_args, new => $attribute_value;
        }

        $self->StatboxAsEvent(@event_args);
    }

    return;
}

sub GetAccountInfo {
    my ($self, $uid) = @_;

    my $account = $self->GetAccount($uid);

    return undef unless $account;

    my $info = {
        login        => $account->machine_readable_login,
        login_wanted => $account->human_readable_login,
    };

    if ($account->is_pdd) {
        $info->{username} = $account->machine_readable_login;

        eval {
            $info->{login_wanted} = Net::IDN::Encode::email_to_unicode( $info->{login_wanted} );
            utf8::decode( $info->{login_wanted} );
        };

        $info->{domain}          = $account->domain->value;
        $info->{domain_id}       = $account->domain->id;
        $info->{karma}           = $account->karma->value;
        $info->{karma_dead_time} = $account->karma->activation_datetime;
    }

    return $info;
}

sub GetLiteAccountInfo {
    my ($self, $uid) = @_;

    my $account = $self->GetAccount($uid);

    return undef unless $account;

    my $UInfo = {
        uid       => $account->uid,
        login     => $account->machine_readable_login,
        ena       => $account->is_enabled,
        litelogin => $account->lite_login,
    };

    return $UInfo;
}

sub GetProjByLogin {
    my ($self, $login, $sid) = @_;

    my $requester = AIO::Requester->new(1);
    my $blackbox_response = eval { $requester->GetUserInfo(
        login => $login,
        sid   => $sid,
        ip    => $Input::Input->{real_ip},
    )};

    return undef
      if $@;

    return $blackbox_response->{uid}{value} || 0;
}

sub InsertMailInfo {
    my ($self, $uid, $UInfo) = @_;

    unless ($uid and ref $UInfo eq 'HASH' and $UInfo->{'login'}) {
        $self->{'ERROR'} = 'Internal';
        return;
    }

    # если речь не о новой регистрации
    unless ($UInfo->{new}) {
        my $account = $self->GetAccount($uid);

        return undef
          unless $account;

        if ($account->subscriptions->get(2)->is_exists) {
            $self->{'ERROR'} = 'EEXIST';
            return;
        }
    }

    my $time = time;
    my $hostid ||= $self->GetMailRegistrationHostId($uid);

    my $suid = $self->ConverterDo('generate_new_suid', Global::IsPddUid($uid));

    return undef unless $suid;

    return undef unless defined $self->ConverterDo('set_attribute_first_host_id' => $uid, $hostid);

    if ($UInfo->{maillist}) {
        return undef unless defined $self->ConverterDo('set_attributes' => $uid, 'account.is_shared_folder' => 1);
    }

    return undef unless defined $self->ConverterDo('insert_suid' => $uid, mail => $suid);

    $self->StatboxAsEvent($uid, 'subscriptions', 'added', sid => 2, suid => $suid);

    my $opdata = {
        uid     => $uid,
        ip_from => $Input::Input->{ip_from},
        ip_prox => $Input::Input->{ip_prox},
        optype  => 'mod',
    };
    my $changes = {
        mail_add => "$suid,$hostid",
    };
    Common::Logs::LogChanges($opdata, $changes);

    return $suid;
}

sub GetMailRegistrationHostId {
    my $self = shift;
    my ($uid) = @_;

    my $key
      = Global::IsPddUid($uid)
      ? 'mail_pdd_registration_host_id'
      : 'mail_normal_registration_host_id';
    my $hostid = $main::Conf->GetVal($key);

    return $hostid;
}

sub RestoreMailInfo {
    my ($this, $uid, $login, $suid, $hostid) = @_;

    unless (defined $this and $uid and $login and $suid and $hostid) {
        $this->{ERROR} = 'Internal';
        return;
    }

    Common::Logs::DeBug("restoremailinfo: suid=$suid uid=$uid login=$login hostid=$hostid");

    my $account = $this->GetAccount($uid);
    return unless $account;

    my $time = time;

    my $opdata = {
        uid     => $uid,
        ip_from => $Input::Input->{ip_from},
        ip_prox => $Input::Input->{ip_prox},
        optype  => 'mod',
    };

    my $changes;

    my $subscription = $account->subscriptions->get(2);
    if ($subscription->is_exists) {
        if ($subscription->suid eq $suid) {
            if (not $subscription->host_id eq $hostid) {
                $changes->{mail_upd} = "$suid,$hostid";
            }
        }
        else {
            $changes->{mail_rm}  = join ',', $subscription->suid, $subscription->host_id;
            $changes->{mail_add} = "$suid,$hostid";
        }
    } else {
        $changes->{mail_add} = "$suid,$hostid";
    }

    return unless defined $this->ConverterDo('delete_suid'    => $uid, 'mail');

    unless (lc $account->login->internal eq lc $login) {
        return unless defined $this->ConverterDo('insert_alias'   => $uid, mail => $login);
    }

    return unless defined $this->ConverterDo('set_attributes' => $uid, 'subscription.mail.host_id' => $hostid);
    return unless defined $this->ConverterDo('insert_suid'    => $uid, mail => $suid);

    $this->StatboxAsEvent($uid, 'subscriptions', 'added', sid => 2, suid => $suid);

    Common::Logs::LogChanges($opdata, $changes)
      if $changes;

    return 1;
}

sub DeleteMailInfo ($$) {
    my ($this, $uid) = @_;

    unless(defined($this) and
           defined($uid) and $uid) {
        $this->{'ERROR'} = 'Internal';
        return undef;
    }

    my $account = $this->GetAccount($uid);
    return undef unless $account;

    my $subscription = $account->subscriptions->get(2);

    my %attributes = (
        'subscription.mail.login_rule' => undef,
        'subscription.mail.host_id'    => undef,
    );

    return undef unless defined $this->ConverterDo('memorize_aliases' => $uid, 'mail', 'narodmail');
    return undef unless defined $this->ConverterDo('delete_aliases'   => $uid, 'mail', 'narodmail');
    return undef unless defined $this->ConverterDo('set_attributes'   => $uid, %attributes);
    return undef unless defined $this->ConverterDo('delete_suid'      => $uid, 'mail');

    $this->StatboxAsEvent($uid, 'subscriptions', 'removed', sid => 2, suid => $subscription->suid);

    my $opdata = {
        uid => $uid,
        ip_from => $Input::Input->{ip_from},
        ip_prox => $Input::Input->{ip_prox},
        optype => 'mod',
    };
    my $changes = {
        rm_subscr => join('|', 2, $uid, $subscription->login, $subscription->suid, $subscription->host_id),
        mail_rm   => join(',', $subscription->suid, $subscription->host_id),
    };
    Common::Logs::LogChanges($opdata, $changes);

    return 1;
}

### TODO после удаления mode=admloginrule и admblock нужен только в админке - может есть смысл упростить?
sub UpdateLoginRule ($$$) {
    my ($this, $uid, $sid, $login_rule) = @_;
    unless(defined($this) and defined($uid) and $uid
            and defined($sid) and $sid and defined($login_rule)) {
        $this->{'ERROR'} = 'Internal';
        return undef;
    }

    my $result = $this->ConverterDo('set_attributes' => $uid, "subscription.$sid.login_rule" => $login_rule);

    return undef unless defined $result;
    return 1;
}

sub UidHavePassword {
    my ($self, $uid) = @_;

    my $account = $self->GetAccount($uid);

    return unless $account;

    return $account->password->is_exists;
}

sub InitializeProtectedAndUnprotectedVariables {
    my ($self) = @_;

    return if $PROTECTED_SIDS;

    my $conf_protected = $main::Conf->GetHCVal('Protected');
    my $conf_sid2from  = $main::Conf->GetHCVal('sid2from');

    my %protected_sids = (%$conf_protected, 8 => 1);
    my @protected_sids = sort { $a <=> $b } keys %protected_sids;
    my @all_sids       = keys %$conf_sid2from;

    $PROTECTED_SIDS   = join ',', @protected_sids;
    @UNPROTECTED_SIDS = sort { $a <=> $b } grep { not $protected_sids{$_} } @all_sids;

    my %attributes_by_sid = Database::Converter->attributes_by_sid;
    my %aliases_by_sid    = Database::Converter->aliases_by_sid;

    my @attributes_sids = sort { $a <=> $b } keys %attributes_by_sid;
    my @aliases_sids    = sort { $a <=> $b } keys %aliases_by_sid;

    my @attributes_unprotected_sids = grep { not $protected_sids{$_} } @attributes_sids;
    my @aliases_unprotected_sids    = grep { not $protected_sids{$_} } @aliases_sids;

    @UNPROTECTED_ATTRIBUTES = map { @{ $attributes_by_sid{$_} } } @attributes_unprotected_sids;
    @UNPROTECTED_ALIASES    = map { @{ $aliases_by_sid{$_}    } } @aliases_unprotected_sids;

    return;
}

=head2 DeleteSubscriptions($uid[, force_drop => 1])

Метод, вызывается для удаления всех подписок юзера.
При наличии среди подписок блокирующих сидов, эти подписки оставит и, в результирующем значении
будет флаг drop_account == 0 (не удалять аккаунт).

=cut
sub DeleteSubscriptions {
    my ($self, $uid, %args) = @_;
    unless ($uid) {
        $self->{'ERROR'} = 'Internal';
        return;
    }

    # забираем все подписки юзера
    my $account = $self->GetAccount($uid);
    return unless $account;

    my $servs = $self->GetActiveServices($uid);
    return unless $servs;

    my $ret = {
        deleted         => {},
        changes         => {},
        drop_account    => 1,
        sids            => []
    };

    $ret->{sids} = [ keys %$servs ];

    my $sid2from        = $main::Conf->GetHCVal('sid2from');
    my $conf_protected  = $main::Conf->GetHCVal('Protected');

    for my $sid (@{$ret->{sids}}) {
        next if $sid == 8;

        # если не был передан флаг игнорировать блокирующие сиды и у юзера подписан на блок-сид, то
        # поднимаем флаг "удалять аккаунт нельзя" и, эту подписку не дропаем
        # NOTE: логику здесь не менял, но фигня может случиться, если у блокирующего сида будет алиас на отписку
        if (not $args{force_drop} and $conf_protected->{$sid}) {
            $ret->{drop_account} = 0;
            next;
        }

        # пытаемся удалить подписку

        # При удалении подписок sid33 и sid58, будут вызваны спец. функции, которые
        # сохранят логины из этих подписок в removed_aliases, для поиска удалённых
        # логинов саппортами и админами.
        unless ($self->UnsubscribeAccountFromSid($uid => $sid)) {
            Common::Logs::AdMin("in del subscriptions: can't unsubscribe user uid $uid sid $sid!");
            $ret->{error}  = 1;
            $ret->{errmsg} = "in del subscriptions: can't unsubscribe user uid $uid sid $sid!";
            last;
        }

        push @{$ret->{changes}->{rm_sid}}, $sid.'|'.$servs->{$sid}->[0];
        # не для всех сидов есть from'ы
        $ret->{changes}->{$sid2from->{$sid}.'_login'} = $servs->{$sid}->[0] if $sid2from->{$sid};
    }

    # перезапишем результат $ret->{changes}->{rm_sid} в вид употребимый хистри
    $ret->{changes}->{rm_sid} = join(',', @{$ret->{changes}->{rm_sid}}) if $ret->{changes}->{rm_sid};

    $self->InitializeProtectedAndUnprotectedVariables;

    # Если аккаунт блокируется, удаляем всё связанное с неблокирующими сидами.
    # То есть, эмулируем удаление всех неблокирующих подписок в новой схеме.
    # Порядок удаления: от наиболее важных к наименее важным.
    # Чистить явно все suid* не надо, т.к. при наличии соответствующих подписок,
    # удаление этих записей будет сделано выше при вызове UnsubscribeAccountFromSid.
    unless ($ret->{drop_account}) {
        if ($account->is_pdd) {
            my $memorize_alias_result = $self->ConverterDo('memorize_pdd_alias' => $uid, 'pddalias', $account->domain->punycoded);

            unless ($memorize_alias_result) {
                $ret->{error}  = 1;
                return $ret;
            }

            my $delete_alias_result = $self->ConverterDo('delete_alias' => $uid, 'pddalias');

            unless ($delete_alias_result) {
                $ret->{error}  = 1;
                return $ret;
            }
        }

        my $delete_attributes_result = $self->ConverterDo('delete_attributes' => $uid, @UNPROTECTED_ATTRIBUTES);

        unless ($delete_attributes_result) {
            $ret->{error}  = 1;
            return $ret;
        }
    }

    return $ret;
}

sub InsertReservedLogin {
    my ($this, $login) = @_;

    unless ($login) {
        $this->{ERROR} = 'Internal';
        return undef;
    }

    my $delay = $main::Conf->GetVal('reserved_login_delay');

    my $time = time;

    my $query
      = 'INSERT INTO reserved_logins (login, free_ts) VALUES '
      . '(?, FROM_UNIXTIME(?) + INTERVAL ? MONTH) '
      . 'ON DUPLICATE KEY UPDATE free_ts = VALUES(free_ts)';
    my @values = ($login, $time, $delay);

    my $result = $this->ConverterDo('do_query_in_central', 'insert_reserved_logins', $query, @values);

    return undef unless defined $result;
    return 1;
}

sub DeleteUser {
    my ($this, $uid, %args) = @_;
    unless(defined($this) and defined($uid) and $uid) {
        $this->{'ERROR'} = 'Internal';
        return undef;
    }

    my $account = $this->GetAccount($uid);

    return undef
      unless $account;

    $this->InsertReservedLogin($account->login->internal)
      unless $account->is_pdd;

    if ($account->is_pdd) {
        # Алиасы у ПДД-шников при удалении аккаунта не удаляются поодиночке специальной функцией DeleteAlias.
        # Поэтому сносим их все сразу: сначала перемещаем в removed_aliases, после чего удаляем.
        return undef unless defined $this->ConverterDo('memorize_pdd_alias' => $uid, 'pddalias', $account->domain->punycoded);
        return undef unless defined $this->ConverterDo('delete_alias'       => $uid, 'pddalias');

        # В последнюю очередь сносим основной ПДД-логин.
        return undef unless defined $this->ConverterDo('memorize_pdd_alias' => $uid, 'pdd',      $account->domain->punycoded);
        return undef unless defined $this->ConverterDo('delete_alias'       => $uid, 'pdd');
    }
    else {
        # При благоприятном стечении обстоятельств, все обычные алиасы должны были быть удалены в DeleteSubscriptions поодиночке.
        # Но на всякий случай, подчищаем их все ещё раз: переносим в removed_aliases, после чего удаляем.
        my @normal_aliases = Database::Converter->normal_aliases;

        return undef unless defined $this->ConverterDo('memorize_aliases' => $uid, @normal_aliases);
        return undef unless defined $this->ConverterDo('delete_aliases' => $uid, @normal_aliases);
    }

    # Удаление из attibutes, password_history
    # Необязательные действия, главное удалить алиасы - тогда аккаунт считается несуществующим,
    # даже если остались хвосты в других таблицах.
    $this->ConverterDo('delete_uid' => $uid);

    return 1;
}

sub SetAccountIsEnabled {
    my ($this, $uid, $value) = @_;

    unless ($uid) {
        $this->{'ERROR'} = 'Internal';
        return undef;
    }

    my $result = $this->ConverterDo('set_attributes' => $uid, 'account.is_disabled' => ! $value);

    return undef unless defined $result;
    return 1;
}

sub SetAccountIsDisabled {
    my ($this, $uid, $value) = @_;

    unless ($uid) {
        $this->{'ERROR'} = 'Internal';
        return undef;
    }

    my $result = $this->ConverterDo('set_attributes' => $uid, 'account.is_disabled' => $value);

    return undef unless defined $result;
    return 1;
}

sub BlockUserSetKarma ($$$) {
    my ($this, $uid, $karma) = @_;
    unless(defined($this) and defined($uid) and $uid and defined($karma)) {
        $this->{'ERROR'} = 'Internal';
        return undef;
    }

    my $result = $this->ConverterDo('set_attributes' => $uid, 'account.is_disabled' => 1, 'karma.value' => $karma);

    return undef unless defined $result;
    return 1;
}

sub SetNeedChangePassword {
    my ($self, $uid) = @_;

    return unless $uid;
    return unless defined $self->ConverterDo('set_attributes' => $uid, 'password.is_changing_required' => 1);
    return 1;
}

sub InsertUserMessage {
    my ($self, $uid, $message_id, $number) = @_;

    return unless $uid and $message_id and $number;

    my $content_id = $message_id;
    $content_id -= 6660
      if $content_id >= 6660;

    my $result = $self->ConverterDo('set_attribute_account_warnings' => $uid, $content_id);

    return unless defined $result;
    return 1;
}

sub AreAnswersMatched {
    my $self = shift;
    my ($uid, $left_hint, $right_answer) = @_;

    my $result = $left_hint->answer->is_equal($right_answer);

    $log->debugf('hint->is_answer_equal: uid=%s result=%s', $uid, $result);

    unless ($result) {
        my $question = $left_hint->question;
        my $badlog = File::Spec->catfile($main::Conf->GetVal('child_log_path', 'bad_answers_log_file'));
        my @fields = (
            $question->id,
            $question->is_user_defined ? $question->text : '',
            $right_answer->normalized,
        );
        my $string = join "\t", @fields;
        unless (Common::Pfile::PWrite($badlog, "$string\n", 'dbutf8')) {
            $log->error("can't log bad answer");
        }
    }

    return $result;
}

sub HintQuestion {
    my ($this, $AllVals) = @_;

    my $hintq = $AllVals->{hintq}.':';

    $hintq = substr $hintq, 0, 100;

    return $hintq;
}

sub DeletePhoneInfo {
    my $self = shift;
    my ($uid) = @_;

    return undef unless defined $self->ConverterDo('set_attributes' => $uid, 'phone.number' => undef, 'phone.confirmation_datetime' => undef);

    $self->StatboxAsEvent($uid, 'subscriptions', 'removed', sid => 36);

    return 1;
}

sub InsertYaStaffInfo {
    my $self = shift;
    my ($uid) = @_;

    my $login = $Input::AllVals->{yastaff_login};
    unless ($login) {
        $self->{ERROR} = "'yastaff_login' not found";
        return;
    }

    my $time = time;

    return undef unless defined  $self->ConverterDo('set_attributes' => $uid, 'account.yandexoid_login' => $login);

    return 1;
}

sub RenameSubscription ($$$$) {
    my ($this, $uid, $newname, $sid) = @_;
    unless ($uid and $newname and $sid) {
        $this->{'ERROR'} = 'Internal';
        return undef;
    }

    my %aliases_by_sid = Database::Converter->aliases_by_sid;
    my $alias_name = $aliases_by_sid{$sid}->[0];

    return undef unless defined $this->ConverterDo('memorize_alias' => $uid, $alias_name);
    return undef unless defined $this->ConverterDo('delete_alias'   => $uid, $alias_name);

    return 1;
}

sub RemoveAlias {
    my ($self, $uid, $alias_name) = @_;

    return undef unless defined $self->ConverterDo('memorize_alias' => $uid, $alias_name);
    return undef unless defined $self->ConverterDo('delete_alias'   => $uid, $alias_name);

    return 1;
}

sub UpdateSubscriptionHostid {
    my $self = shift;
    my (%args) = @_;

    my $result = $self->ConverterDo('set_attributes' => $args{uid}, 'subscription.mail.host_id' => $args{host_id});

    die "can't UpdateSubscriptionHostid"
      unless defined $result;

    my $opdata = {
        uid     => $args{uid},
        ip_from => $Input::Input->{ip_from},
        ip_prox => $Input::Input->{ip_prox},
        optype  => 'mod',
    };
    my $changes = { mail_upd => "$args{suid},$args{host_id}" };

    Common::Logs::LogChanges($opdata, $changes);

    return 1;
}

my $HOSTS_TTL = 600;
sub GetMailHosts {
    my $self = shift;

    my $table_name = 'mail_hosts';
    my $now = time;

    my $cached  = $CACHE{$table_name};
    my $expired = $cached->{expired};

    if ($now < $expired) {
        return $cached->{data};
    }

    $expired = $now + $HOSTS_TTL;

    my $requester = AIO::Requester->new(1);

    my ($hosts1, $hosts2);

    eval {
        $hosts1 = $requester->GetMailHosts(sid => 2, is_pdd => 0);
        $hosts2 = $requester->GetMailHosts(sid => 2, is_pdd => 1);
    };

    if ($@) {
        Common::Logs::IntErr("can't get mailhosts: $@");
        return undef;
    }

    my %by_db_id   = map { $_->{db_id}   => $_ } @$hosts1, @$hosts2;
    my %by_host_id = map { $_->{host_id} => $_ } @$hosts1, @$hosts2;

    my %hosts_by_db_id;
    for my $host (@$hosts1, @$hosts2) {
        my $list = $hosts_by_db_id{ $host->{db_id} } ||= [];
        next if grep { $_->{host_id} == $host->{host_id} } @$list;
        push @$list, $host;
    }

    my @hosts = sort { $a->{host_id} <=> $b->{host_id} } values %by_host_id;

    my $data = {
        by_db_id   => \%by_db_id,
        by_host_id => \%by_host_id,
        hosts_by_db_id => \%hosts_by_db_id,
        list       => \@hosts,
    };

    $cached = {
        created => $now,
        expired => $expired,
        data    => $data,
    };
    $CACHE{$table_name} = $cached;

    return $data;
}

sub ConverterDo {
    my $self = shift;
    my $method_name = shift;
    my @params = @_;

    # Обязательно нужно делать! Выполняем точно такую же перекодировку,
    # что и в SqlExecute. Тогда данные для вставки и в старую базу, и в
    # новую будут совпадать. EncodeAll дополнительно исправляет кривые
    # unicode-строки. Например, плохая последовательность "q\xA0w", пришедшая
    # из запроса, конвертируется в правильную utf8-последовательность "q\xC2\xA0w"
    # (или "q\x{a0}w"). И если в SqlExecute мы делаем такое исправление,
    # а здесь - нет, то в старую базу приедет хорошая строка, а в новую - нет,
    # т.к. в ней данные хранятся как VARBINARY.
    Global::EncodeAll(\@params, 'inner', $db_encoding_zone);

    my $result;

    try {
        no strict 'refs';
        $result = $self->CreateConverter->$method_name(@params);
    } catch {
        Common::Logs::DbErr("method=$method_name error=$_ params=" . Dumper(\@params));
    };

    unless (defined $result) {
        Common::Logs::DbErr("method=$method_name params=" . Dumper(\@params));
    }

    return $result;
}

sub CreateConverter {
    my $self = shift;

    my $result = Database::Converter->new(
        uid_collection  => $self->uid_collection,
        central_group   => $self->central_group,
    );

    return $result;
}

my @SIMPLE_SIDS = qw/
                    5  6        9
             13 14       17    19
             23 24 25 26 27    29
    30 31                37 38 39
    40 41       44       47 48 49
    50 51 52 53 54 55    57    59
    60
                      76 77 78
    80 81    83 84 85
    90 91
    100   102   104
    665 666 667 668 670 671 672 1000
/;

my %SUBSCRIBE_METHOD = (
    2   => sub { shift->ChangeAccountSubscriptionToLegacySid('InsertMailInfo',       @_) },
    669 => sub { shift->ChangeAccountSubscriptionToLegacySid('InsertYaStaffInfo',    @_) },

    map +( $_ => 'SubscribeAccountToSimpleSid' ), @SIMPLE_SIDS,
);
my %UNSUBSCRIBE_METHOD = (
    2   => sub { shift->ChangeAccountSubscriptionToLegacySid('DeleteMailInfo',       @_) },
    36  => sub { shift->ChangeAccountSubscriptionToLegacySid('DeletePhoneInfo',      @_) },

    16  => sub { shift->UnsubscribeAccountFromAliasedSid(@_) },
    33  => sub { shift->UnsubscribeAccountFromAliasedSid(@_) },
    58  => sub { shift->UnsubscribeAccountFromAliasedSid(@_) },
    61  => sub { shift->UnsubscribeAccountFromAliasedSid(@_) },
    65  => sub { shift->UnsubscribeAccountFromAliasedSid(@_) },
    68  => sub { shift->UnsubscribeAccountFromAliasedSid(@_) },
    669 => sub { shift->UnsubscribeAccountFromAliasedSid(@_) },
);

sub SubscribeAccountToSid {
    my $self = shift;
    my ($uid, $sid, %args) = @_;

    my $method = $SUBSCRIBE_METHOD{$sid};

    return undef
      unless $method;

    my $result;
    {
        no strict 'refs';
        $result
          = ref $method eq 'CODE'
          ? $method->($self, @_)
          : $self->$method(@_);
    }

    return $result;
}

sub SubscribeAccountToSimpleSid {
    my $self = shift;
    my ($uid, $sid, %args) = @_;

    $log->debugf('SubscribeAccountToSimpleSid(%s, %s)', $uid, $sid);

    my $time = time;

    my $result = $self->ConverterDo('set_attributes' => $uid, "subscription.$sid" => 1);

    return undef unless defined $result;

    $self->StatboxAsEvent($uid, 'subscriptions', 'added', sid => $sid);

    return 1;
}

sub ChangeAccountSubscriptionToLegacySid {
    my $self = shift;
    my ($method, $uid, $sid, %args) = @_;

    $log->debugf('ChangeAccountSubscriptionToLegacySid(%s)', \@_);

    my $result;
    {
        no strict 'refs';
        $result = $self->$method($uid, { %args });
    }

    return $result;
}

sub UnsubscribeAccountFromSid {
    my $self = shift;
    my ($uid, $sid, %args) = @_;

    my $method = $UNSUBSCRIBE_METHOD{$sid};

    $method ||= 'UnsubscribeAccountFromSimpleSid';

    my $result;
    {
        no strict 'refs';
        $result
          = ref $method eq 'CODE'
          ? $method->($self, @_)
          : $self->$method(@_);
    }

    return $result;
}

sub UnsubscribeAccountFromSimpleSid {
    my $self = shift;
    my ($uid, $sid) = @_;

    $log->debugf('UnsubscribeAccountFromSimpleSid(%s, %s)', $uid, $sid);

    my $result = $self->ConverterDo('set_attributes' => $uid, "subscription.$sid" => undef);

    return undef unless defined $result;

    $self->StatboxAsEvent($uid, 'subscriptions', 'removed', sid => $sid);

    return 1;
}

sub UnsubscribeAccountFromAliasedSid {
    my $self = shift;
    my ($uid, $sid) = @_;

    $log->debugf('UnsubscribeAccountFromAliasedSid(%s, %s)', $uid, $sid);

    my %aliases_by_sid = Database::Converter->aliases_by_sid;
    my $alias_name = $aliases_by_sid{$sid}->[0];

    return undef unless $alias_name;

    return undef unless defined $self->ConverterDo('memorize_alias' => $uid, $alias_name);
    return undef unless defined $self->ConverterDo('delete_alias'   => $uid, $alias_name);

    $self->StatboxAsEvent($uid, 'subscriptions', 'removed', sid => $sid);

    return 1;
}

sub GetAccount {
    my $self = shift;

    my $result = $self->GetAccountFromCache(@_);

    my $method;

    if ($result) {
        $method = 'cache';
    }
    else {
        $result = $self->GetAccountByUid(@_);
        $method = $result ? 'blackbox' : 'blackboxnotfound';
    }

#    $log->debugf('GetAccount: method=%s args=%s caller=%s', $method, \@_, (caller 1)[3]);

    return $result;
}

sub ParseGetAccountArgs {
    my $self = shift;
    my ($key, %args) = @_;

    my %result;

    if (length $key) {
        $result{key_type}   = $key =~ /^\d+$/ ? 'uid'          : 'login';
        $result{cache_type} = $key =~ /^\d+$/ ? 'AccountByUid' : 'AccountByLogin';
        $result{key}        = $key;
    }
    elsif ($args{uid}) {
        $result{key_type}   = 'uid';
        $result{cache_type} = 'AccountByUid';
        $result{key}        = $args{uid};
    }
    elsif ($args{suid}) {
        $result{key_type}   = 'suid';
        $result{cache_type} = 'AccountBySuid';
        $result{key}        = $args{suid};
    }
    elsif ($args{login}) {
        $result{key_type}   = 'login';
        $result{cache_type} = 'AccountByLogin';
        $result{key}        = $args{login};
        $result{sid}        = $args{sid};
    }

    return %result;
}

sub GetAccountFromCache {
    my $self = shift;
    my %args = $self->ParseGetAccountArgs(@_);

    my $account = $self->GetCachedData($args{cache_type}, $args{key});

    return $account;
}

sub GetAccountByUid {
    my $self = shift;
    my %args = $self->ParseGetAccountArgs(@_);

    my %blackbox_params = (
        $args{key_type} => $args{key},
        ip        => $Input::Input->{real_ip}->as_string,
        queries   => Model::Builder->required_blackbox_queries,
    );
    $blackbox_params{sid} = $args{sid}
      if $args{sid};

    my $requester = AIO::Requester->new(1);
    my $blackbox_response = eval { $requester->GetUserInfo(%blackbox_params) };

    return
      if $@;

    my $uid = $blackbox_response->{uid}{value};

    # если обычным запросом ЧЯ не выдал uid, но сообщил, что домен зарегистрирован
    # как ПДД, то пробуем еще раз указав явно, что нужно искать лайта
    if (not $uid and $blackbox_response->{uid}{domid}) {
        $blackbox_params{sid} = 'mk';
        $blackbox_response = eval { $requester->GetUserInfo(%blackbox_params) };

        return
          if $@;

        $uid = $blackbox_response->{uid};
    }

    return
      unless $uid;

    my $account = $self->GetAccountFromBlackboxResponse($uid, $blackbox_response);

    if ($args{key_type} eq 'login') {
        $self->SaveCachedData($args{cache_type}, $args{key}, $account);
    }

    return $account;
}

sub GetAccountByUidExceptionable {
    my $self = shift;
    my %args = $self->ParseGetAccountArgs(@_);

    my %blackbox_params = (
        $args{key_type} => $args{key},
        ip        => $Input::Input->{real_ip}->as_string,
        queries   => Model::Builder->required_blackbox_queries,
    );

    my $requester = AIO::Requester->new(1);
    my $blackbox_response = $requester->GetUserInfo(%blackbox_params);

    my $uid = $blackbox_response->{uid}{value};

    # если обычным запросом ЧЯ не выдал uid, но сообщил, что домен зарегистрирован
    # как ПДД, то пробуем еще раз указав явно, что нужно искать лайта
    if (not $uid and $blackbox_response->{uid}{domid}) {
        $blackbox_params{sid} = 'mk';
        $blackbox_response = $requester->GetUserInfo(%blackbox_params);

        $uid = $blackbox_response->{uid};
    }

    return
      unless $uid;

    my $account = $self->GetAccountFromBlackboxResponse($uid, $blackbox_response);

    if ($args{key_type} eq 'login') {
        $self->SaveCachedData($args{cache_type}, $args{key}, $account);
    }

    return $account;
}

sub GetAccountFromBlackboxResponse {
    my $self = shift;
    my ($uid, $blackbox_response) = @_;

    my $builder = Model::Builder->new;
    $builder->blackbox_response($blackbox_response);

    my $account = $builder->build_account;

    $account->dbaf($self);

    $self->SaveCachedData('AccountByUid', $account->uid, $account)
      if $account;

    return $account;
}

sub GetAccountWithPhones {
    my $self = shift;

    my $account = $self->GetAccount(@_);
    return
      unless $account;

    return $account;
}

sub GetAccountWithPhonesAndSocialProfiles {
    my $self = shift;

    my $account = $self->GetAccount(@_);
    return
      unless $account;

    my $requester = AIO::Requester->new(1);
    my $social_response = eval { $requester->GetSocialUserProfiles(uid => $account->uid, include => 'person') };

    return
      if $@;

    $account->social_profiles($social_response);

    return $account;
}

sub GetAccounts {
    my $self = shift;
    my ($uids) = @_;

    my %blackbox_params = (
        uids      => $uids,
        ip        => $Input::Input->{real_ip}->as_string,
        queries   => Model::Builder->required_blackbox_queries,
    );

    my $requester = AIO::Requester->new(1);
    my $blackbox_response = eval { $requester->GetUsersInfo(%blackbox_params) };

    return
      if $@;

    my %accounts_by_uid;
    for my $user (@$blackbox_response) {
        my $uid = $user->{uid}{value};
        next unless $uid;
        my $account = $self->GetAccountFromBlackboxResponse($uid, $user);
        $accounts_by_uid{$uid} = $account;
    }

    my @accounts = map { $accounts_by_uid{$_} || () } @$uids;

    return \@accounts;
}

sub GetAccountsExceptionable {
    my $self = shift;
    my ($uids) = @_;

    my %blackbox_params = (
        uids      => $uids,
        ip        => $Input::Input->{real_ip}->as_string,
        queries   => Model::Builder->required_blackbox_queries,
    );

    my $requester = AIO::Requester->new(1);
    my $blackbox_response = $requester->GetUsersInfo(%blackbox_params);

    my %accounts_by_uid;
    for my $user (@$blackbox_response) {
        my $uid = $user->{uid}{value};
        next unless $uid;
        my $account = $self->GetAccountFromBlackboxResponse($uid, $user);
        $accounts_by_uid{$uid} = $account;
    }

    my @accounts = map { $accounts_by_uid{$_} || () } @$uids;

    return \@accounts;
}

sub StatboxAsEvent {
    my $self = shift;
    my ($uid, $entity, $operation, %args) = @_;

    my $mode = $Output::formargs->{passport_call_mode};
    $args{user_agent} = $Input::io->{req}->header_in('User-Agent')
      if $mode and not $Passport::api_modes{$mode};

    Common::Logs::Statbox(
        event     => 'account_modification',
        uid       => $uid,
        entity    => $entity,
        operation => $operation,
        %args,
    );
}

1;
