package DBAF::SafeForDomains;

use strict;
use utf8;
use open qw(:std :utf8);
use warnings;

use JSON::XS;

my $DEFAULT_SLICE_LIMIT = 20;
my $MAX_SLICE_LIMIT     = 100;
my $JSON = JSON::XS->new->utf8(0);

sub GetDomainId {
    my ($self, $domain_name) = @_;

    my $domain = $self->GetDomain(domain_name => $domain_name);
    return undef unless $domain;
    return $domain->{domain_id};
}

sub GetDomain {
    my $self = shift;
    my (%args) = @_;

    my $domains = $self->GetHostedDomains(%args);

    return undef unless $domains;

    my $domain = $domains->[0];

    return undef unless $domain;
    return $domain;
}

sub ReformDomain {
    my $self = shift;
    my ($domain) = @_;

    $domain->{domain_id} = delete $domain->{domid};
    $domain->{enabled}   = delete $domain->{ena};

    unless (ref $domain->{options} eq 'HASH') {
        my $options = $JSON->decode($domain->{options} || '{}');

        $options->{can_users_change_password} = 1
          unless defined $options->{can_users_change_password};

        $domain->{options} = $options;
    }

    my @aliases = split /,/, delete $domain->{slaves};
    $domain->{aliases}->{alias} = \@aliases;

    return 1;
}

sub FindAccounts {
    my ($self, %args) = @_;

    my $result;
    my $uids;

    if ($args{uid}) {
        return unless Global::IsPddUid($args{uid});
        $uids = [ $args{uid} ];
    }

    else {
        if ($args{login} and $args{login} =~ /\@/) {
            my ($username, $domain) = split /\@/, $args{login}, 2;
            $args{login} = $username;
            $args{domain_name} = $domain;
        }
    
        $args{limit} ||= $DEFAULT_SLICE_LIMIT;
        $args{limit} = $MAX_SLICE_LIMIT
          if $args{limit} > $MAX_SLICE_LIMIT;

        delete $args{sort}
          if $args{sort} and not $args{sort} =~ /^ (uid|login) \. (de|a)sc $/ix;

        my $requester = AIO::Requester->new(1);
        my $blackbox_response = eval { $requester->FindPddAccounts(%args) };

        return
          if $@;

        $result->{total_count} = $blackbox_response->{total_count};
        $result->{offset} = $args{offset}
          if $args{offset};

        $uids = $blackbox_response->{uids};
    
        return $result unless @$uids;
    }

    my $accounts = $self->GetAccounts($uids);

    return
      unless $accounts;

    $result->{accounts} = $accounts;
    $result->{total_count} = scalar @$accounts
      if $args{uid};

    return $result;
}

sub GetHostedDomains {
    my $self = shift;
    my (%args) = @_;

    if ($args{domain_id}) {
        my $domains = $self->GetCachedData('DomainsById', $args{domain_id});
        return $domains
          if $domains;
    }
    elsif ($args{domain_name}) {
        my $domains = $self->GetCachedData('DomainsByName', $args{domain_name});
        return $domains
          if $domains;
    }
    elsif ($args{admin_uid}) {
        my $domains = $self->GetCachedData('DomainsByAdmin', $args{admin_uid});
        return $domains
          if $domains;
    }

    my $requester = AIO::Requester->new(1);
    my $domains   = eval { $requester->GetHostedDomains(%args) };

    return undef if $@;

    for my $domain (@$domains) {
        $self->ReformDomain($domain);
    }

    if ($args{admin_uid}) {
        $self->SaveCachedData('DomainsByAdmin', $args{admin_uid}, $domains);
    }
    else {
        my $domain = $domains->[0];
        if ($domain) {
            $self->SaveCachedData('DomainsById',   $domain->{domain_id}, $domains);
            $self->SaveCachedData('DomainsByName', $domain->{domain},    $domains);
        }
    }

    return $domains;
}

1;
