package Database::Converter;

use common::sense;

use File::Spec;
use Log::Any '$log';
use List::MoreUtils 'uniq';
use POSIX 'strftime';
use Time::HiRes;
use Try::Tiny;

use Common::Pfile;

use Class::XSAccessor {
    constructor => 'new',
    accessors   => [qw/uid_collection central_group/],
};

my %ATTRIBUTE_NAME_TO_TYPE = (
    'account.registration_datetime'       =>  1,
    'account.user_defined_login'          =>  2,
    'account.is_disabled'                 =>  3,
    'account.global_logout_datetime'      =>  4,
    'account.is_pdd_agreement_accepted'   =>  6,
    'account.is_pdd_admin'                =>  7,
    'account.is_betatester'               =>  8,
    'account.is_corporate'                =>  9,
    'account.is_vip'                      => 10,
    'account.is_test'                     => 11,
    'account.is_employee'                 => 12,
    'account.is_shared_folder'            => 13,
    'account.yandexoid_login'             => 14,
    'person.contact_phone_number'         => 15,
    'account.display_name'                => 16,
    'karma.value'                         => 17,
    'karma.activation_datetime'           => 18,
    'password.encrypted'                  => 19,
    'password.update_datetime'            => 20,
    'password.quality'                    => 21,
    'password.is_changing_required'       => 22,
    'password.is_creating_required'       => 23,
    'password.is_strong_required'         => 24,
    'hint.question.serialized'            => 25,
    'hint.answer.encrypted'               => 26,
    'person.firstname'                    => 27,
    'person.lastname'                     => 28,
    'person.gender'                       => 29,
    'person.birthday'                     => 30,
    'person.country'                      => 31,
    'person.city'                         => 32,
    'person.timezone'                     => 33,
    'person.language'                     => 34,
    'phone.number'                        => 35,
    'phone.confirmation_datetime'         => 36,
    'location.postal'                     => 37,
    'subscription.mail.host_id'           => 38,
    'subscription.mail.login_rule'        => 39,
    'subscription.zakladki.suid'          => 40,
    'subscription.5'                      => 44,
    'subscription.6'                      => 45,
    'subscription.9'                      => 46,
    'subscription.13'                     => 47,
    'subscription.14'                     => 48,
    'subscription.17'                     => 49,
    'subscription.19'                     => 50,
    'subscription.23'                     => 52,
    'subscription.24'                     => 53,
    'subscription.25'                     => 54,
    'subscription.26'                     => 55,
    'subscription.jabber.login_rule'      => 56,
    'subscription.29'                     => 57,
    'subscription.30'                     => 58,
    'subscription.31'                     => 59,
    'subscription.37'                     => 60,
    'subscription.38'                     => 61,
    'subscription.39'                     => 62,
    'subscription.40'                     => 63,
    'subscription.41'                     => 64,
    'subscription.wwwdgt.mode'            => 65,
    'subscription.disk.login_rule'        => 66,
    'subscription.47'                     => 68,
    'subscription.48'                     => 69,
    'subscription.49'                     => 70,
    'subscription.50'                     => 71,
    'subscription.51'                     => 72,
    'subscription.52'                     => 73,
    'subscription.53'                     => 74,
    'subscription.54'                     => 75,
    'subscription.55'                     => 76,
    'subscription.57'                     => 77,
    'subscription.59'                     => 78,
    'subscription.60'                     => 79,
    'subscription.64'                     => 80,
    'subscription.76'                     => 81,
    'subscription.77'                     => 82,
    'subscription.78'                     => 83,
    'subscription.80'                     => 84,
    'subscription.81'                     => 85,
    'subscription.666'                    => 86,
    'subscription.667'                    => 87,
    'account.warnings'                    => 88,
    'subscription.83'                     => 89,
    'subscription.84'                     => 90,
    'subscription.85'                     => 91,
    'subscription.86'                     => 92,
    'subscription.87'                     => 93,
    'account.is_ad_campaign_participant'  => 94,
    'subscription.88'                     => 95,
    'subscription.90'                     => 96,
    'subscription.91'                     => 97,
    'avatar.default'                      => 98,

    'totp.secret'                         => 104,
    'totp.last_successful_time_period'    => 106,

    'subscription.201'                    => 113,
    'subscription.202'                    => 114,
    'subscription.203'                    => 115,
    'subscription.204'                    => 116,
    'subscription.205'                    => 117,

    'totp.failed_pin_checks_count'        => 118,
    'totp.update_datetime'                => 123,

    'subscription.102'                    =>  6, # account.is_pdd_agreement_accepted
    'subscription.104'                    =>  7, # account.is_pdd_admin
    'subscription.668'                    =>  8, # account.is_betatester
    'subscription.670'                    =>  9, # account.is_corporate
    'subscription.671'                    => 10, # account.is_vip
    'subscription.672'                    => 11, # account.is_test
    'subscription.1000'                   => 12, # account.is_employee
    'subscription.89'                     => 15, # person.contact_phone_number
    'subscription.100'                    => 23, # password.is_creating_required
    'subscription.67'                     => 24, # password.is_strong_required
    'subscription.2.host_id'              => 38, # subscription.mail.host_id
    'subscription.2'                      => 39, # subscription.mail
    'subscription.2.login_rule'           => 39, # subscription.mail.login_rule
    'subscription.3'                      => 40, # subscription.zakladki
    'subscription.3.suid'                 => 40, # subscription.zakladki.suid
    'subscription.27'                     => 56, # subscription.jabber
    'subscription.27.login_rule'          => 56, # subscription.jabber.login_rule
    'subscription.42'                     => 65, # subscription.wwwdgt
    'subscription.42.host_id'             => 65, # subscription.wwwdgt.mode
    'subscription.44'                     => 66, # subscription.disk
    'subscription.44.login_rule'          => 66, # subscription.disk.login_rule
);
# Обновление login_rule делается только на следующих сидах: mail money disk narod narod2 (по грантам к admloginrule) и jabber (в админке)

my @PASS_IF_ZERO = qw/
    subscription.mail.login_rule
    subscription.narod.login_rule
    subscription.money.login_rule
    subscription.jabber.login_rule
    subscription.disk.login_rule
    subscription.narod2.login_rule
    person.firstname
    person.lastname
    hint.answer.encrypted
/;
my %PASS_IF_ZERO = map { $ATTRIBUTE_NAME_TO_TYPE{$_} => 1 } @PASS_IF_ZERO;

my %SID_TO_ATTRIBUTES_NAMES;
my @attributes_names_with_subscription_prefix = grep /^subscription\.\d+/, keys %ATTRIBUTE_NAME_TO_TYPE;
for my $name (@attributes_names_with_subscription_prefix) {
    my (undef, $sid) = split /\./, $name;
    push @{ $SID_TO_ATTRIBUTES_NAMES{$sid} }, $name;
}
push @{ $SID_TO_ATTRIBUTES_NAMES{8}    }, 'account.user_defined_login', 'password.is_changing_required';
push @{ $SID_TO_ATTRIBUTES_NAMES{1000} }, 'account.is_shared_folder';

my %SID_TO_ALIASES_NAMES = (
    8   => ['portal', 'pdd'],
    2   => ['mail'],
    16  => ['narodmail'],
    4   => ['narod'],
    33  => ['lite'],
    58  => ['social'],
    105 => ['pddalias'],
    61  => ['altdomain'],
    68  => ['phonish'],
    65  => ['phonenumber'],
    669 => ['yandexoid'],
);

my %ALIAS_NAME_TO_SID;
for my $sid (keys %SID_TO_ALIASES_NAMES) {
    my $names = $SID_TO_ALIASES_NAMES{$sid};
    for my $name (@$names) {
        $ALIAS_NAME_TO_SID{$name} = $sid;
    }
}

my @NORMAL_ALIASES = qw/portal mail narodmail narod lite social altdomain phonish phonenumber yandexoid neophonish kiddish scholar federal bank_phone_number/;
my @PDD_ALIASES    = qw/pdd pddalias/;

my %ALIAS_NAME_TO_TYPE = (
    portal            => 1,
    mail              => 2,
    narodmail         => 3,
    lite              => 5,
    social            => 6,
    pdd               => 7,
    pddalias          => 8,
    altdomain         => 9,
    phonish           => 10,
    phonenumber       => 11,
    yandexoid         => 13,
    neophonish        => 21,
    kiddish           => 22,
    scholar           => 23,
    federal           => 24,
    bank_phone_number => 25,
);

my %UNIQUE_ALIASES = (
    pddalias => 1,
);

my %SERVICE_NAME_TO_SID = (
    mail     => 2,
    zakladki => 3,
);

my %ENTITY_HASH_BY_NAME = (
    attribute => \%ATTRIBUTE_NAME_TO_TYPE,
    alias     => \%ALIAS_NAME_TO_TYPE,
    service   => \%SERVICE_NAME_TO_SID,
);

sub attributes        { %ATTRIBUTE_NAME_TO_TYPE  }
sub attributes_by_sid { %SID_TO_ATTRIBUTES_NAMES }
sub aliases_by_sid    { %SID_TO_ALIASES_NAMES    }
sub sid_by_alias      { %ALIAS_NAME_TO_SID       }
sub normal_aliases    { @NORMAL_ALIASES          }
sub pdd_aliases       { @PDD_ALIASES             }

sub log_fail {
    my $self = shift;
    my ($format, @values) = @_;

    my $time      = time;
    my $localtime = localtime $time;
    my @today     = localtime $time;
    my @yesterday = localtime($time - 60 * 60 * 24);

    my $logname           = File::Spec->catfile($main::Conf->GetVal('child_log_path', 'newdbfail_log_file'));
    my $yesterday_logname = $logname . strftime('.%Y%m%d', @yesterday);
    my $today_logname     = $logname . strftime('.%Y%m%d', @today);

    my $string = join "\t", $localtime, sprintf $format, @values;
    $string =~ s/[\r\n]+/^M/g;
    $string .= "\n";

    utf8::decode $string;

    Common::Pfile::PClose($yesterday_logname);
    Common::Pfile::PWrite($today_logname, $string, 'dbutf8');

    return;
}

sub do_query {
    my $self = shift;
    my ($name, $connection, $query, @values) = @_;

#    $log->debugf(q/do_query '%s'. %s/, $name, { query => $query, values => \@values });

    my $max_attempts = $main::Conf->GetVal('new_database_writes_attempts');
    my $delay        = $main::Conf->GetVal('new_database_writes_delay');

    my $attempts;
    my $is_successful;

    my $result;

    while (not $is_successful) {
        try {
            $result = $connection->do($query, @values);
            $is_successful = 1;
        }
        catch {
            ++$attempts;

            $log->warnf("newdbfail: attempt=$attempts/$max_attempts error=$_");

            if ($attempts < $max_attempts) {
                Time::HiRes::sleep($delay)
                  if $delay;
            }
            else {
                die $_;
            }
        };
    }

    return $result;
}

sub do_query_by_uid {
    my $self = shift;
    my ($name, $uid, $query, @values) = @_;

    my $result;

    try {
        my $shard = $self->uid_collection->get_master_by_key($uid);
        $result   = $self->do_query($name, $shard, $query, @values);
    } catch {
        $self->log_fail("$uid\t$_\t$name\t$query");
        die $_;
    };

    return $result;
}

sub do_query_in_central {
    my $self = shift;
    my ($name, $query, @values) = @_;

    my $result;

    try {
        my $central = $self->central_group->master;
        $result     = $self->do_query($name, $central, $query, @values);
    }
    catch {
        $self->log_fail("central\t$_\t$name\t$query");
        die $_;
    };

    return $result;
}

sub do_query_in_central_by_uid {
    my $self = shift;
    my ($name, $uid, $query, @values) = @_;

    my $result;

    try {
        my $central = $self->central_group->master;
        $result     = $self->do_query($name, $central, $query, @values);
    }
    catch {
        $self->log_fail("$uid\t$_\t$name\t$query");
        die $_;
    };

    return $result;
}

sub do_query_in_central_with_last_insert_id {
    my $self = shift;
    my ($name, $query, @values) = @_;

    my $result;

    try {
        my $central = $self->central_group->master;
        $self->do_query($name, $central, $query, @values);
        $result = $central->dbh->{mysql_insertid};
        die "undefined last_insert_id"
          unless $result;
    }
    catch {
        $self->log_fail("central\t$_\t$name\t$query");
        die $_;
    };

    return $result;
}

sub set_attributes {
    my $self = shift;
    my ($uid, %attributes) = @_;

#    $log->debugf('set_attributes. uid=%s attributes=%s', $uid, \%attributes);

    my $query_head = q/INSERT INTO attributes (uid, type, value) VALUES/;
    my $query_row  = q/(?, ?, ?)/;
    my $query_tail = q/ON DUPLICATE KEY UPDATE value = VALUES(value)/;

    my @values;
    my $rows;
    my @deletes;

    while (my ($name, $value) = each %attributes) {
        my $type = $self->get_attribute_type_by_name($name);
        next unless $type;
        if ($self->is_attribute_passed($type => $value)) {
            push @values, $uid, $type, $value;
            ++$rows;
        } else {
            push @deletes, $name;
        }
    }

    my $result;

    if ($rows) {
        my $query_rows = join ', ', ($query_row) x $rows;
        my $query = join ' ', $query_head, $query_rows, $query_tail;

        $result = $self->do_query_by_uid('set_attributes', $uid, $query, @values);

        return unless defined $result;
    }

    if (@deletes) {
        $result = $self->delete_attributes($uid => @deletes);
    }

    return $result;
}

sub delete_attributes {
    my $self = shift;
    my ($uid, @names) = @_;

#    $log->debugf('delete_attributes. uid=%s attributes=%s', $uid, \@names);

    my $query_head = q/DELETE FROM attributes WHERE uid = ? AND type IN (/;
    my $query_row  = q/?/;
    my $query_tail = q/)/;

    my @values = ($uid);

    my %types;
    for my $name (@names) {
        my $type = $self->get_attribute_type_by_name($name);
        next unless $type;
        $types{$type} = 1;
    }

    my @types = sort keys %types;
    push @values, @types;

    my $rows = @types;

    return unless $rows;

    my $query_rows = join ', ', ($query_row) x $rows;
    my $query = join '', $query_head, $query_rows, $query_tail;

    my $result = $self->do_query_by_uid('delete_attributes', $uid, $query, @values);

    return $result;
}

sub insert_alias {
    my $self = shift;
    my ($uid, $name, $value) = @_;

    $value = lc $value;
#    $log->debugf('set_alias. uid=%s name=%s value=%s', $uid, $name, $value);

    my $type = $self->get_alias_type_by_name($name);
    return unless $type;

    my $surrogate_type
      = $UNIQUE_ALIASES{$name}
      ? "$type-$value"
      : $type;

    my $query
      = 'INSERT INTO aliases (uid, type, value, surrogate_type) VALUES '
      . '(?, ?, ?, ?)';
    my @values = ($uid, $type, $value, $surrogate_type);

    my $result = $self->do_query_in_central_by_uid('create_alias', $uid, $query, @values);

    return $result;
}

sub delete_alias {
    my $self = shift;
    my ($uid, $name, $value) = @_;

#    $log->debugf('delete_alias. uid=%s name=%s', $uid, $name);

    my $type = $self->get_alias_type_by_name($name);
    return unless $type;

    my $query  = q/DELETE FROM aliases WHERE uid = ? AND type = ?/;
    my @values = ($uid, $type);
    if ($value) {
        $value = lc $value;
        $query .= ' AND value = ?';
        push @values, $value;
    }

    my $result = $self->do_query_in_central_by_uid('delete_alias', $uid, $query, @values);

    return $result;
}

sub delete_aliases {
    my $self = shift;
    my ($uid, @names) = @_;

#    $log->debugf('delete_aliases. uid=%s names=%s', $uid, \@names);

    my $query_head = q/DELETE FROM aliases WHERE uid = ? AND type IN (/;
    my $query_row  = q/?/;
    my $query_tail = q/)/;

    my @values = ($uid);
    my $rows;

    for my $name (@names) {
        my $type = $self->get_alias_type_by_name($name);
        next unless $type;
        push @values, $type;
        ++$rows;
    }

    return unless $rows;

    my $query_rows = join ', ', ($query_row) x $rows;
    my $query = join '', $query_head, $query_rows, $query_tail;

    my $result = $self->do_query_in_central_by_uid('delete_aliases', $uid, $query, @values);

    return $result;
}

sub insert_suid {
    my $self = shift;
    my ($uid, $name, $suid) = @_;

#    $log->debugf('insert_suid. uid=%s name=%s suid=%s', $uid, $name, $suid);

    my $sid = $self->get_service_sid_by_name($name);
    return unless $sid;

    my $query
      = "INSERT INTO suid$sid (suid, uid) VALUES "
      . '(?, ?)';
    my @values = ($suid, $uid);

    my $result = $self->do_query_in_central_by_uid('insert_suid', $uid, $query, @values);

    return $result;
}

sub delete_suid {
    my $self = shift;
    my ($uid, $name) = @_;

#    $log->debugf('delete_suid. uid=%s name=%s', $uid, $name);

    my $sid = $self->get_service_sid_by_name($name);
    return unless $sid;

    my $query  = qq/DELETE FROM suid$sid WHERE uid = ?/;
    my @values = ($uid);

    my $result = $self->do_query_in_central_by_uid('delete_suid', $uid, $query, @values);

    return $result;
}

sub memorize_aliases {
    my $self = shift;
    my ($uid, @names) = @_;

#    $log->debugf('memorize_aliases. uid=%s names=%s', $uid, \@names);

    my $query_head
      = q/INSERT IGNORE INTO removed_aliases /
      . q/(uid, type, value) /
      . q/SELECT uid, type, value FROM aliases WHERE uid = ? AND type IN (/;
    my $query_row  = q/?/;
    my $query_tail = q/)/;

    my @values = ($uid);
    my $rows;

    for my $name (@names) {
        my $type = $self->get_alias_type_by_name($name);
        next unless $type;
        push @values, $type;
        ++$rows;
    }

    return unless $rows;

    my $query_rows = join ', ', ($query_row) x $rows;
    my $query = join '', $query_head, $query_rows, $query_tail;

    my $result = $self->do_query_in_central_by_uid('memorize_aliases', $uid, $query, @values);

    return $result;
}

sub memorize_pdd_alias {
    my $self = shift;
    my ($uid, $name, $domain_name, $value) = @_;

#    $log->debugf('memorize_pdd_alias. uid=%s name=%s domain_name=%s value=$value', $uid, $name, $domain_name, $value);

    my $type = $self->get_alias_type_by_name($name);
    return unless $type;

    my $query
      = q{INSERT IGNORE INTO removed_aliases }
      . q{(uid, type, value) }
      . q{SELECT uid, type, CONCAT( ?, SUBSTR(value, LOCATE('/', value)) ) }
      . q{FROM aliases WHERE uid = ? AND type = ?};

    $domain_name = lc $domain_name;

    my @values = ($domain_name, $uid, $type);

    if ($value) {
        $value = lc $value;
        $query .= ' AND value = ?';
        push @values, $value;
    }

    my $result = $self->do_query_in_central_by_uid('memorize_pdd_alias', $uid, $query, @values);

    return $result;
}

sub memorize_alias {
    my $self = shift;
    my ($uid, $name) = @_;

    return $self->memorize_aliases($uid, $name);
}

sub delete_uid {
    my $self = shift;
    my ($uid) = @_;

    $self->do_query_by_uid('delete_attributes',           $uid, 'DELETE FROM attributes WHERE uid = ?',           $uid);
    $self->do_query_by_uid('delete_password_history',     $uid, 'DELETE FROM password_history WHERE uid = ?',     $uid);

    # Не удаляем явно suid*, т.к. они всегда удаляются при удалении подписок в DeleteSubscriptions.
    # Ничего не делаем с алиасам: и memorize, и delete вызываются в Delete*Info функциях, а также в DeleteUser (повторно).

    return 1;
}

sub set_attribute_account_warnings {
    my $self = shift;
    my ($uid, $value) = @_;

#    $log->debugf('set_attribute_account_warnings. uid=%s value=%s', $uid, $value);

    my $type = $self->get_attribute_type_by_name('account.warnings');
    return unless $type;

    my $query
      = q/INSERT INTO attributes (uid, type, value) VALUES /
      . q/(?, ?, ?) /
      . q/ON DUPLICATE KEY UPDATE value = CONCAT(value, ',', VALUES(value))/;
    my @values = ($uid, $type, $value);

    my $result = $self->do_query_by_uid('set_attribute_account_warnings', $uid, $query, @values);

    return $result;
}

sub set_attribute_first_host_id {
    my $self = shift;
    my ($uid, $value) = @_;

#    $log->debugf('set_attribute_first_host_id. uid=%s value=%s', $uid, $value);

    my $type = $self->get_attribute_type_by_name('subscription.mail.host_id');
    return unless $type;

    my $query
      = q/INSERT INTO attributes (uid, type, value) VALUES /
      . q/(?, ?, ?) /;
    my @values = ($uid, $type, $value);

    my $result = $self->do_query_by_uid('set_attribute_first_host_id', $uid, $query, @values);

    return $result;
}

sub set_attribute_next_host_id {
    my $self = shift;
    my ($uid, $value, $old_value) = @_;

#    $log->debugf('set_attribute_next_host_id. uid=%s value=%s old_value=%s', $uid, $value, $old_value);

    my $type = $self->get_attribute_type_by_name('subscription.mail.host_id');
    return unless $type;

    my $query
      = q/INSERT INTO attributes (uid, type, value) VALUES /
      . q/(?, ?, ?) /
      . q/ON DUPLICATE KEY UPDATE value = /;
    my @values = ($uid, $type, $value);

    if ($old_value) {
        $query .= q/IF(value = ?, VALUES(value), value)/;
        push @values, $old_value;
    }
    else {
        $query .= q/VALUES(value)/;
    }

    my $result = $self->do_query_by_uid('set_attribute_next_host_id', $uid, $query, @values);

    return $result;
}

sub generate_new_id {
    my $self = shift;
    my ($name, $is_pdd) = @_;

#    $log->debugf('generate_new_id. name=%s is_pdd=%s', $name, $is_pdd ? 1 : 0);

    my $table
      = $is_pdd
      ? "pdd$name"
      : $name;

    my $query = "INSERT INTO $table VALUES (NULL)";

    my $result = $self->do_query_in_central_with_last_insert_id("generate_new_id_$table", $query);

    return $result;
}

sub generate_new_uid  { shift->generate_new_id('uid',  @_) }
sub generate_new_suid { shift->generate_new_id('suid', @_) }

sub insert_domain {
    my $self = shift;
    my (%values) = @_;

#    $log->debugf('insert_domain. %s', \%values);

    my $query  = 'INSERT INTO domains SET name = ?, master_domain_id = ?, enabled = ?, mx = ?, admin_uid = ?, ts = FROM_UNIXTIME(?)';
    my @values = (
        lc $values{name},
        $values{master_domain_id} || 0,
        $values{enabled}          || 0,
        $values{mx}               || 0,
        $values{admin_uid}        || 0,
        $values{time},
    );

    my $result = $self->do_query_in_central_with_last_insert_id('insert_domain', $query, @values);

    return $result;
}

sub get_entity_value_by_key {
    my $self = shift;
    my ($name, $key) = @_;

    return $key if $key =~ /^\d+$/;
    my $hash  = $ENTITY_HASH_BY_NAME{$name};
    my $value = $hash->{$key} || undef;
    return $value if $value;
#    $log->debugf("unknown $name.name=$key");
    return;
}

sub get_attribute_type_by_name { shift->get_entity_value_by_key('attribute', @_) }
sub get_alias_type_by_name     { shift->get_entity_value_by_key('alias',     @_) }
sub get_service_sid_by_name    { shift->get_entity_value_by_key('service',   @_) }

sub is_attribute_passed {
    my $self = shift;
    my ($type, $value) = @_;

    return $PASS_IF_ZERO{$type} ? 1 : 0
      if $value eq '0';
    return $value               ? 1 : 0;
}

1;
