package History::Converter::PassportChanges;

use strict;
use warnings;

my @mandatory_params = qw(
    log_admin_code_ref
    log_debug_code_ref
);

sub new {
    my $class = shift;
    my $self  = bless {}, $class;
    return $self->init(@_);
}

sub init {
    my ($self, %param) = @_;

    for my $param_name (@mandatory_params) {
        return undef unless exists $param{$param_name};
        $self->{$param_name} = $param{$param_name};
    }

    return $self;
}

sub convert {
    my ($self, $opdata, $changes) = @_;

    $self->_preprocess($opdata, $changes);
    return $self->_convert_changes_to_events($changes);
}

# препроцессинг opdata и changes когда нельзя использовать converters
sub _preprocess {
    my ($self, $opdata, $changes) = @_;

    # удаление аккаунта, только когда optype=del и есть запись с логином (при удалении подписки optype=del, но логина нет)
    if ($opdata->{optype} eq 'del' and exists $changes->{login}) {
        $changes->{action} = 'delete';
        delete $changes->{login};
    }

    my @from;
    for (keys %$changes) {
        if (/^(.+)_login$/) {
            push @from, { from => $1, value => delete $changes->{$_} };
        }
    }

    if (@from) {
        $changes->{from} = join ',', map { join '|', $_->{from}, $_->{value} } @from;
    }
}

my %converter_by_changename = (
    login      => 'info.login',
    login_wanted => 'info.login_wanted',
    login_domain => 'info.login_domain',
    login_renamed => 'info.login_renamed',
    iname      => 'info.firstname',
    fname      => 'info.lastname',
    sex        => 'info.sex',
    birth_date => 'info.birthday',
    bdate      => 'info.birthday',
    country    => 'info.country',
    xcountry   => 'info.country',
    city       => 'info.city',
    email      => 'info.email',
    display_name => 'info.display_name',
    reg_date   => 'info.reg_date',
    tz         => 'info.tz',
    timezone   => 'info.tz',
    hintq      => \&_CreateHintQ,
    udhinta    => \&_CreateHintQ,
    hintqtext  => \&_CreateHintQ,
    hinta      => 'info.hinta',
    karma      => 'info.karma',
    prefix     => 'info.karma_prefix',
    mail_limit => 'info.mail_limit',
    ena        => 'info.ena',
    enabled    => 'info.ena',
    blocked    => \&_CreateDis, # всегда равен yes
    disabling_reason => 'info.dis.reason',
    glogout    => 'info.glogout',
    totp_secret => 'info.totp_secret',

    add_sid         => \&_CreateSidAddList,
    rm_sid          => \&_CreateSidRmList,
    rm_subscr       => 'sid.rm.info', # расширенные данные удаленной подписки
    restore_subscr  => 'sid.restore',
    yasms_number    => 'sid.yasms_number', # при подписке на yasms, в subscription пишется последние цифры телефона
    wwwdgt_wmode    => 'sid.wwwdgt_wmode', # wmode - это битовая маска, также при подписке пихается в host_id
    direct_agency   => 'sid.direct_agency', # при регистрации через admreg с from=direct - содержит имя агентства, для которого создан аккаунт
    login_rule      => \&_CreateLoginRule, # в admloginrule
    sid             => sub {},
    sid_change      => 'sid.change',

    karma_dead_time => 'info.karma_dead_time',
    acl_id          => 'info.karma_dead_time', # используется только в одном месте в контексте karma_dead_time
    
    nocookie        => 'nocookie', # непонятно
    action          => 'action', # например admsubscribe
    sid_8           => 'restore_account',  # значение = suid
    from            => 'from',

    add_name_alias  => 'alias.name.add',
    rm_name_alias   => 'alias.name.rm',

    byear           => sub {},
    bmonth          => sub {},
    bday            => sub {},
    hintaentered    => sub {},
    admin_uid       => sub {},
    '_login'        => sub {}, # где-то не делается проверка на параметр from, который должен быть префиксом       
    'subs'          => sub {}, # содержит uid, когда удаляется подписка на 9 и 11 сид
    lang            => 'info.lang', # язык для юзера

    user_agent      => 'user_agent',
    support_note    => 'support_note',

    mail_add => 'mail.add',
    mail_upd => 'mail.upd',
    mail_rm  => 'mail.rm',
);

sub _convert_changes_to_events {
    my ($self, $changes) = @_;

    my $result = [];

    for my $change_name (keys %$changes) {
        # проверка на существование обязательна, т.к. трансформеры могут удалить часть ключей
        next unless exists $changes->{$change_name};

        # если имени изменения нет в списке, логируем его и переходим на следующее
        unless (exists $converter_by_changename{$change_name}) {
            $self->{log_debug_code_ref}->("found unknown changename=$change_name");
            next;
        }
        
        my $converter = $converter_by_changename{$change_name};

        my @events
            = ref $converter eq 'CODE'
                ? $converter->($changes, $change_name)
                : { name => $converter, value => delete $changes->{$change_name} };
       
        for my $event (@events) {
            next unless $event and $event->{name};
            push @$result, $event;
        }
    }
    
    return $result;
}

# собирает hintq и udhintq в одну строку info.hintq
sub _CreateHintQ {
    my $hintq = delete($_[0]->{hintq}) || 0;
    my $udhintq = delete($_[0]->{udhintq}) || '';
    my $hintqtext = delete($_[0]->{hintqtext}) || '';
    return {
        name => 'info.hintq',
        value => join(':', $hintq, $hintqtext || $udhintq),
    };
}

# создает запись info.ena=0 (вместо blocked=yes)
sub _CreateDis {
    return {
        name => 'info.ena',
        value => 0,
    };
}

# конвертирует строку из сидов в несколько записей sid.add
sub _CreateSidAddList {
    my ($changes, $change_name) = @_;

    my @result;

    my @sids = split /,/, $changes->{$change_name};
    for my $sid (@sids) {
        push @result, {
            name => 'sid.add',
            value => $sid,
        };
    }
    return @result;
}

sub _CreateSidRmList {
    my ($changes, $change_name) = @_;

    my @result;

    my @sids = split /,/, $changes->{$change_name};
    for my $sid (@sids) {
        push @result, {
            name => 'sid.rm',
            value => $sid,
        };
    }
    return @result;
}

# конвертит sid+login_rule
sub _CreateLoginRule {
    return {
        name => 'sid.login_rule',
        value => join('|', delete $_[0]->{sid}, delete $_[0]->{login_rule}),
    };
}

1;

