package History::Logger;

use strict;
use warnings;
use utf8;
use open qw(:std :utf8);
use Data::Dumper;

use POSIX qw(strftime);
use Time::HiRes qw(gettimeofday tv_interval);

my @mandatory_params = qw(
    host_id
    client_name

    auth_log_path
    event_log_path

    pfile_write_code_ref
    pfile_close_code_ref
    log_admin_code_ref
    log_interr_code_ref
);

sub new {
    my $class = shift;
    my $self  = bless {}, $class;
    return $self->init(@_);
}

sub init {
    my ($self, %param) = @_;

    for my $param_name (@mandatory_params) {
        return undef unless exists $param{$param_name};
        $self->{$param_name} = $param{$param_name};
    }

    $self->{null_string} = '-';
    $self->{sep_char}    = ' ';
    $self->{quote_char}  = '`';
    $self->{escape_char} = '`';

    return $self;
}

sub now {
    my ($self) = @_;

    my ($seconds, $microseconds) = gettimeofday;
    my ($sec, $min, $hour, $mday, $mon, $year, $wday, $yday, $isdst) = localtime $seconds;
    my $timezone = strftime('%z', $sec, $min, $hour, $mday, $mon, $year, $wday, $yday, $isdst);
    $timezone =~ s/\d\d$//;

    return sprintf '%04d-%02d-%02dT%02d:%02d:%02d.%06d%s', $year + 1900, $mon + 1, $mday, $hour, $min, $sec, $microseconds, $timezone;
}

sub log_events {
    my ($self, %args) = @_;

    my ($caller, $target, $events, $admin, $comment) = @args{qw(caller target events admin comment)};

    unless ($target->{uid}) {
        local $Data::Dumper::Indent = 0;
        $self->{log_admin_code_ref}->("bad logevents calling\n" . Dumper($caller, $target, $events, $admin, $comment));
        return 0;
    }

    my $ver = '1';
    my $now = $self->now;
    my $client_name = $self->{client_name};
    my $host_id = $self->{host_id};
    
    my $logname = $self->_get_current_logname($self->{event_log_path});

    $self->_escape(\( $admin, $comment ));
    $self->_quote(\( $admin, $comment ));
    $self->_nullify(\( @$caller{qw(ip_from ip_prox yandexuid)}, $admin, $comment ));

    for my $event (@$events) {
        next unless $event->{name};

        $self->_escape(\( $event->{value} ));
        $self->_quote(\( $event->{value} ));
        $self->_nullify(\( $event->{value} ));

        my $string = join ' ', (
            $ver,
            $now,
            $host_id,
            $client_name,
            $target->{uid},
            $event->{name},
            $event->{value},
            $caller->{ip_from},
            $caller->{ip_prox},
            $caller->{yandexuid},
            $admin,
            $comment,
        );
        $string .= "\n";

        unless ($self->{pfile_write_code_ref}->($logname, $string, 'dbutf8')) {
            local $Data::Dumper::Indent = 0;
            $self->{log_interr_code_ref}->("can't log event\n" . Dumper($caller, $target, $events, $admin, $comment));
            return 0;
        }
    }

    return 1;
}

my %auth_comment_alias = (
    captcha_passed => 'cpt',
    session_ttl    => 'ttl',
    session_authid => 'aid',
    session_compressed => 'cmprs',
    is_secure      => 'ssl',
    admin          => 'adm',
);

sub log_auth {
    my ($self, %args) = @_;

    my ($caller, $target, $auth) = @args{qw(caller target auth)};

    unless (
            ($target->{uid} or $target->{login})
        and $auth->{type}
        and $auth->{status}
    ) {
        local $Data::Dumper::Indent = 0;
        $self->{log_admin_code_ref}->("bad logauth calling\n" . Dumper($caller, $target, $auth));
        return 0;
    }

    my $ver = '1';
    my $now = $self->now;
    my $client_name = $self->{client_name};
    my $host_id = $self->{host_id};
    
    my $logname = $self->_get_current_logname($self->{auth_log_path});

    # собираем из комментов строку, заменяя имена входных комментов на соответствующие им алиасы
    my ($comment, $sep) = ('', '');
    for my $comment_name (sort keys %{ $auth->{comments} }) {
        next unless exists $auth_comment_alias{$comment_name};
        my $comment_value = $auth->{comments}->{$comment_name};
        next unless defined $comment_value;
        $comment .= $sep . $auth_comment_alias{$comment_name} . '=' . $comment_value;
        $sep = ';';
    }

    $self->_escape(\( $target->{login}, @$caller{qw(referer retpath useragent)} ));
    $self->_quote( \( $target->{login}, @$caller{qw(referer retpath useragent)} ));
    $self->_nullify(\( @$target{qw(uid login sid)}, $comment, @$caller{qw(ip_from ip_prox yandexuid referer retpath useragent)} ));

    # собираем результирующую строку
    my $string = join ' ', (
        $ver,
        $now,
        $host_id,
        $client_name,
        $target->{uid},
        $target->{login},
        $target->{sid},
        $auth->{type},
        $auth->{status},
        $comment,
        $caller->{ip_from},
        $caller->{ip_prox},
        $caller->{yandexuid},
        $caller->{referer},
        $caller->{retpath},
        $caller->{useragent},
    );
    $string .= "\n";

    # и пишем ее
    unless ($self->{pfile_write_code_ref}->($logname, $string, 'dbutf8')) {
        local $Data::Dumper::Indent = 0;
        $self->{log_interr_code_ref}->("can't log auth\n" . Dumper($caller, $target, $auth));
        return 0;
    }

    return 1;
}

sub _nullify {
    my $self = shift;

    for (@_) {
        next if defined $$_ and $$_ ne '';
        $$_ = $self->{null_string};
    }
}

sub _escape {
    my $self = shift;
    
    for (@_) {
        next if not defined $$_ or $$_ eq '';
        $$_ =~ s/\n/\\n/g;
        $$_ =~ s/\r/\\r/g;
    }
}

sub _quote {
    my $self = shift;

    my ($sep_char, $quote_char, $escape_char, $null_string) = @$self{qw(sep_char quote_char escape_char null_string)};

    for (@_) {
        if (not defined $$_) {
            $$_ = '';
        } elsif ($$_ eq $null_string) {
            $$_ = $quote_char . $$_ . $quote_char;
        } else {
            next unless $$_ =~ m/[$sep_char$quote_char]/o;
            $$_ =~ s/$quote_char/$escape_char$quote_char/go;
            $$_ = $quote_char . $$_ . $quote_char;
        }
    }
}

sub _get_current_logname {
    my ($self, $logname) = @_;
    
    my $time = time;
    my $yesterday_logname = $logname . strftime('.%Y%m%d', localtime($time - 60 * 60 * 24));
    my $today_logname     = $logname . strftime('.%Y%m%d', localtime($time));
    
    $self->{pfile_close_code_ref}->($yesterday_logname);
    return $today_logname;
}

1;

