package Model::Builder;

use common::sense;

use Net::IDN::Encode 'domain_to_unicode';

use Global;
use Model::Account;
use Model::Answer;
use Model::Aliases;
use Model::DisplayName;
use Model::Domain;
use Model::Email;
use Model::Emails;
use Model::FamilyInfo;
use Model::Hint;
use Model::Karma;
use Model::Login;
use Model::Password;
use Model::Person;
use Model::Phone;
use Model::Phones;
use Model::Question;
use Model::Restoration;
use Model::Subscription;
use Model::Subscriptions;

sub required_fields {qw/
    accounts.ena.uid
    accounts.login.uid
    accounts.glogout.uid

    userinfo.firstname.uid
    userinfo.lastname.uid
    userinfo.sex.uid
    userinfo.birth_date.uid
    userinfo.country.uid
    userinfo.city.uid
    userinfo.reg_date.uid
    userinfo.tz.uid
    userinfo.lang.uid
    userinfo.display_name.uid

    userinfo_safe.hintq.uid
    userinfo_safe.hinta.uid

    subscription.suid.2
    subscription.login_rule.2
    subscription.login.2
    subscription.host_id.2
    hosts.db_id.2

    subscription.suid.3

    subscription.suid.5
    subscription.suid.6

    subscription.login_rule.8
    subscription.login.8

    subscription.suid.9
    subscription.suid.13
    subscription.suid.14

    subscription.suid.16
    subscription.host_id.16
    subscription.login.16

    subscription.suid.17
    subscription.suid.19

    subscription.suid.23
    subscription.suid.24
    subscription.suid.25
    subscription.suid.26

    subscription.login_rule.27

    subscription.suid.29
    subscription.suid.30
    subscription.suid.31

    subscription.suid.33
    subscription.login.33

    subscription.suid.36
    subscription.host_id.36

    subscription.suid.37
    subscription.suid.38
    subscription.suid.39
    subscription.suid.40
    subscription.suid.41

    subscription.login_rule.44

    subscription.suid.47
    subscription.suid.48
    subscription.suid.49
    subscription.suid.50
    subscription.suid.51
    subscription.suid.52
    subscription.suid.53
    subscription.suid.54
    subscription.suid.55
    subscription.suid.57

    subscription.login.58

    subscription.suid.59
    subscription.suid.60
    subscription.login.61
    subscription.suid.64

    subscription.login.65

    subscription.suid.67

    subscription.login.68

    subscription.suid.76
    subscription.suid.77
    subscription.suid.78
    subscription.suid.80
    subscription.suid.81
    subscription.suid.83
    subscription.suid.84
    subscription.suid.85
    subscription.suid.86
    subscription.suid.87

    subscription.login.89

    subscription.suid.90
    subscription.suid.91

    subscription.login_rule.100
    subscription.suid.102
    subscription.suid.104

    subscription.suid.666
    subscription.suid.667
    subscription.suid.668
    subscription.login.669
    subscription.suid.670
    subscription.suid.671
    subscription.suid.672
    subscription.suid.1000
/}

sub required_blackbox_queries {[
    regname  => 'yes',
    emails   => 'getall',
    aliases  => 'all',
    attributes => '3,20,107,113,114,115,116,117,123,132,142,148,178,215,225,1003,1015',
    dbfields => join(',', shift->required_fields),
    getphones => 'all',
    phone_attributes => '2,3,4,5,102,106,107,108,109',
    getphoneoperations => 'yes',
    get_family_info => 'yes',
]}

my %ATTRIBUTE_TO_SID = (
    113 => 201,
    114 => 202,
    115 => 203,
    116 => 204,
    117 => 205,
    142 => 116,
    148 => 117,
    178 => 119,
    215 => 125,
);

use Class::XSAccessor {
    constructor => 'new',
    accessors   => [qw/blackbox_response account/],
};

sub build_account {
    my $self = shift;

    # Костыль! Раньше ЧЯ в поле accounts.login.uid для ПДД-шников отдавал только username-часть без домена.
    # Остальной код проекта ожидает именно такой ответ. Но новый ЧЯ теперь отдаёт логин вместе с unicoded доменом.
    # Поэтому возвращаем значение в старом виде, чтобы не переделывать логику в куче мест.
    my $res = $self->blackbox_response;

    if ($res->{uid}{domain}) {
        my $old_login = $res->{dbfields}{'accounts.login.uid'};
        my ($username, $domain) = split /\@/, $old_login, 2;
        my $new_login = $username;

        $res->{dbfields}{'accounts.login.uid'} = $new_login;
    }

    my $account = Model::Account->new;
    $self->account($account);

    $account->subscriptions($self->build_subscriptions);
    $self->fill_account;

    $account->login       ($self->build_login);
    $account->aliases     ($self->build_aliases);
    $account->domain      ($self->build_domain);
    $account->karma       ($self->build_karma);
    $account->display_name($self->build_display_name);
    $account->person      ($self->build_person);
    $account->password    ($self->build_password);
    $account->hint        ($self->build_hint);
    $account->emails      ($self->build_emails);
    $account->phones      ($self->build_phones);
    $account->restoration ($self->build_restoration);
    $account->messages    ($self->build_messages);
    $account->family_info ($self->build_family);

    return $account;
}

sub dbfields { shift->blackbox_response->{dbfields} }

sub build_subscriptions {
    my $self = shift;

    my $res    = $self->blackbox_response;
    my $fields = $self->dbfields;

    my $by_sid;
    for my $field (keys %$fields) {
        my $value = $fields->{$field};
        next unless $field =~ /^subscription\./;
        next unless defined $value and length $value;
        my (undef, $column, $sid) = split /\./, $field;
        $by_sid->{$sid}{$column} = $value;
    }

    # ЧЯ всегда отдаёт заполненный sid=16, если есть sid=2. Поэтому,
    # при отсутствии разницы в логинах, считаем sid=16 несуществующим.
    if ($by_sid->{16} and $by_sid->{16}{login} eq $by_sid->{2}{login}) {
        delete $by_sid->{16};
    }

    my $subscriptions = Model::Subscriptions->new;

    for my $sid (keys %$by_sid) {
        my $params = $by_sid->{$sid};
        $params->{sid} = $sid;
        $params->{login_rule} = 1
          unless defined $params->{login_rule};
        my $subscription = Model::Subscription->new(%$params);
        $subscriptions->add($subscription);
    }

    # Эмулируем sid'ы из некоторых атрибутов
    for my $attribute (keys %ATTRIBUTE_TO_SID) {
        next unless $res->{attributes}{$attribute};
        my $params = {
            suid       => 1,
            sid        => $ATTRIBUTE_TO_SID{$attribute},
            login_rule => 1,
        };
        my $subscription = Model::Subscription->new(%$params);
        $subscriptions->add($subscription);
    }

    $subscriptions->get(4)->host_number($fields->{'hosts.host_number.4'})
      if $subscriptions->get(4)->is_exists;

    $subscriptions->get(2)->db_id($fields->{'hosts.db_id.2'})
      if $subscriptions->get(2)->is_exists;

    return $subscriptions;
}

sub fill_account {
    my $self = shift;

    my $res    = $self->blackbox_response;
    my $fields = $self->dbfields;

    my $account = $self->account;
    my $subscriptions = $account->subscriptions;

    $account->uid                  ($res->{uid}{value} || $res->{liteuid}{value} || 0);
    $account->is_enabled           ($fields->{'accounts.ena.uid'} ? 1 : 0);
    $account->disabling_reason     ($res->{attributes}{3});
    $account->global_logout_datetime($fields->{'accounts.glogout.uid'} || 0);
    $account->global_logout_datetime_local(Global::TimestampToLocalDatetime($fields->{'accounts.glogout.uid'} || 0));
    $account->registration_datetime($fields->{'userinfo.reg_date.uid'} || '');
    $account->registration_ts      (Global::DatetimeToTimestamp($account->registration_datetime));
    $account->short_phone_number   ($subscriptions->get(36)->host_id || '');
    $account->lite_login           ($subscriptions->get(33)->login || '');
    $account->is_pdd_agreement_accepted($subscriptions->get(102)->is_exists);
    $account->is_betatester        ($subscriptions->get(668)->is_exists);
    $account->yandexoid_login      ($subscriptions->get(669)->login || '');
    $account->is_vip               ($subscriptions->get(671)->is_exists);
    $account->is_shared            ($res->{attributes}{132} ? 1 : 0);
    $account->have_plus            ($res->{attributes}{1015} ? 1 : 0);
    $account->can_manage_children  ($res->{attributes}{225} ? 1 : 0);

    return $account;
}

sub build_login {
    my $self = shift;

    my $fields = $self->dbfields;

    my $accounts_login
      = length($fields->{'accounts.login.uid'})
      ? $fields->{'accounts.login.uid'}
      : '';

    my $login = Model::Login->new(
        internal     => $accounts_login,
        user_defined => $fields->{'subscription.login.8'} || '',
    );

    return $login;
}

sub build_aliases {
    my $self = shift;

    my $res  = $self->blackbox_response;
    my $hash = $res->{aliases};

    $hash->{8} ||= [];

    $hash->{8} = [ $hash->{8} ]
      unless ref $hash->{8} eq 'ARRAY';

    my $aliases = Model::Aliases->new(
        portal             => $hash->{1} || '',
        mail               => $hash->{2} || '',
        narodmail          => $hash->{3} || '',
        narod              => $hash->{4} || '',
        lite               => $hash->{5} || '',
        social             => $hash->{6} || '',
        pdd                => $hash->{7} || '',
        pddalias           => $hash->{8} || [],
        altdomain          => $hash->{9}  || '',
        phonish            => $hash->{10} || '',
        phonenumber        => $hash->{11} || '',
        yandexoid          => $hash->{13} || '',
        neophonish         => $hash->{21} || '',
        kiddish            => $hash->{22} || '',
        scholar            => $hash->{23} || '',
        federal            => $hash->{24} || '',
        bank_phone_number  => $hash->{25} || '',
    );

    return $aliases;
}

sub build_domain {
    my $self = shift;

    my $res = $self->blackbox_response;
    my $domain = Model::Domain->new(
        id    => $res->{uid}{domid} || 0,
        value => $res->{uid}{domain} || '',
        is_enabled => $res->{uid}{domain_ena} || 0,
    );

    return $domain;
}

sub build_karma {
    my $self = shift;

    my $res = $self->blackbox_response;
    my $karma = Model::Karma->new(
        activation_datetime => $res->{karma}{'allow-until'} || 0,
    );
    $karma->value($res->{karma_status}{value} || 0);

    return $karma;
}

sub build_display_name {
    my $self = shift;

    my $res    = $self->blackbox_response;
    my $fields = $self->dbfields;

    my $display_name = Model::DisplayName->new;

    $display_name->deserialize($fields->{'userinfo.display_name.uid'});

    unless (length $display_name->value) {
        $display_name->prefix('p');
        $display_name->value($res->{display_name}{name});
    }

    return $display_name;
}

sub build_family {
    my $self = shift;

    my $res = $self->blackbox_response;

    my $family_info = Model::FamilyInfo->new;
    $family_info->family_id($res->{'family_info'}{'family_id'} || '');
    $family_info->admin_uid($res->{'family_info'}{'admin_uid'} || '');

    return $family_info;
}

sub build_person {
    my $self = shift;

    my $fields = $self->dbfields;
    my $person = Model::Person->new(
        firstname => length($fields->{'userinfo.firstname.uid'}) ? $fields->{'userinfo.firstname.uid'} : '',
        lastname  => length($fields->{'userinfo.lastname.uid'})  ? $fields->{'userinfo.lastname.uid'}  : '',
        sex       => $fields->{'userinfo.sex.uid'} || '',
        birthday  => $fields->{'userinfo.birth_date.uid'} || '',
        country   => uc $fields->{'userinfo.country.uid'} || '',
        city      => $fields->{'userinfo.city.uid'} || '',
        timezone  => $fields->{'userinfo.tz.uid'} || '',
        language  => $fields->{'userinfo.lang.uid'} || '',
    );

    return $person;
}

sub build_messages {
    my $self = shift;

    my $res = $self->blackbox_response;

    my @messages = split /\D/, $res->{user_messages};

    return \@messages;
}

sub build_password {
    my $self = shift;

    my $res           = $self->blackbox_response;
    my $subscriptions = $self->account->subscriptions;
    my $fields        = $self->dbfields;

    my $update_ts = $res->{attributes}{20} || 0;
    my $update_dt = $update_ts ? Global::TimestampToLocalDatetime($update_ts) : '';
    my $totp_enable_ts = $res->{attributes}{123} || 0;
    my $totp_enable_dt = $totp_enable_ts ? Global::TimestampToLocalDatetime($totp_enable_ts) : '';

    my $password = Model::Password->new(
        is_exists            => $res->{have_password} ? 1 : 0,
        is_creating_required => $subscriptions->get(100)->login_rule == 1  ? 1 : 0,
        is_changing_required => $subscriptions->get(8)->login_rule & 0b100 ? 1 : 0,
        is_strong_required   => $subscriptions->get(67)->is_exists,
        is_2fa_enabled       => $res->{attributes}{1003} ? 1 : 0,
        is_app_specific_passwords_enabled => $res->{attributes}{107} ? 1 : 0,
        update_ts            => $update_ts,
        update_dt            => $update_dt,
        totp_enable_ts       => $totp_enable_ts,
        totp_enable_dt       => $totp_enable_dt,
    );

    return $password;
}

sub build_hint {
    my $self = shift;

    my $fields  = $self->dbfields;
    my $account = $self->account;

    my $question = Model::Question->new;
    $question->serialized($fields->{'userinfo_safe.hintq.uid'});

    my $answer = Model::Answer->new;
    $answer->text($fields->{'userinfo_safe.hinta.uid'});

    my $hint = Model::Hint->new(
        question => $question,
        answer   => $answer,
    );

    return $hint;
}

sub build_emails {
    my $self = shift;

    my $res     = $self->blackbox_response;
    my $account = $self->account;
    my $list    = $res->{'address-list'};

    my $emails = Model::Emails->new(list => []);

    for my $item (@$list) {
        my $email = Model::Email->new(
            address       => $item->{address},
            is_native     => $item->{native}    ? 1 : 0,
            is_confirmed  => $item->{validated} ? 1 : 0,
            is_rpop       => $item->{rpop}      ? 1 : 0,
            is_unsafe     => $item->{unsafe}    ? 1 : 0,
            is_default    => $item->{default}   ? 1 : 0,
            confirmation_ts => Global::DatetimeToTimestamp($item->{'born-date'}),
            confirmation_dt => $item->{'born-date'},
        );
        $emails->add($email);
    }

    # Лайтам явно добавляем их мыло
    if ($account->is_lite) {
        my $email = Model::Email->new(
            address       => $account->machine_readable_login,
            is_native     => 0,
            is_confirmed  => 1,
            is_rpop       => 0,
            is_unsafe     => 0,
            is_default    => 1,
            confirmation_ts => Global::DatetimeToTimestamp($account->registration_datetime),
            confirmation_dt => $account->registration_datetime,
        );
        $emails->add($email);
    }

    return $emails;
}

sub build_phones {
    my $self = shift;

    my $res     = $self->blackbox_response;
    my $account = $self->account;
    my $list    = $res->{phones};

    my $bindings = {};

    my $operations = $res->{phone_operations} || {};
    for my $operation (values %$operations) {
        my @values = split /,/, $operation;

        # Формат строки:
        # https://github.yandex-team.ru/passport/passport-core/blob/master/passport/builders/blackbox/parsers.py#L110
        my ($phone_id, $type) = @values[2, 4];

        next unless $type eq '1'; # type=bind

        $bindings->{$phone_id} = 1;
    }

    my $phones = Model::Phones->new(list => []);

    for my $item (@$list) {
        my $is_bound    = $item->{attributes}{106}   ? 1 : 0;
        my $has_binding = $bindings->{ $item->{id} } ? 1 : 0;

        # Эмулируем фильтр /userphones
        # https://github.yandex-team.ru/passport/passport-api/blob/master/passport_api/yasms/api.py#L439
        next if not $is_bound and not $has_binding;

        my $time = $item->{attributes}{5} || $item->{attributes}{4} || $item->{attributes}{3} || $item->{attributes}{2}; # admitted || confirmed || bound || created

        my $phone = Model::Phone->new(
            id           => $item->{id},
            number       => $item->{attributes}{102},
            status       => $is_bound ? 'valid' : 'msgsent',
            is_active    => $item->{attributes}{107} ? 1 : 0,
            is_secure    => $item->{attributes}{108} ? 1 : 0,
            is_bank      => $item->{attributes}{109} ? 1 : 0,
            confirmation_ts => $time,
            confirmation_dt => Global::TimestampToLocalDatetime($time),
        );
        $phones->add($phone);
    }

    return $phones;
}

sub build_restoration {
    my $self = shift;

    my $account = $self->account;
    my $restoration = Model::Restoration->new(
        hint        => $account->hint,
        emails      => $account->emails,
        phones      => $account->phones,
    );

    return $restoration;
}

1;
